#!/usr/bin/env python

# Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from datetime import datetime
import dbus
import getopt
import hashlib
import os
import sys

try:
    import json
except:
    import simplejson as json

THUMBNAIL_PATH = os.path.expanduser('~/.thumbnails/cropped/')

try:
    from configparser import ConfigParser
except ImportError:
    from ConfigParser import ConfigParser
    
class ResourceError(Exception):
    pass
    
def format_datetime(timestamp):
    try:
        return datetime.fromtimestamp(timestamp).strftime('%d %b %Y')
    except:
        return datetime.now().strftime('%d %b %Y')
        
def format_duration(seconds):
    try:
        return datetime.fromtimestamp(seconds).strftime('%M:%S')
    except:
        return ''

def list_songs(filter = '', offset = 0, limit = 20):
    bus = dbus.SessionBus()
    obj = bus.get_object('org.freedesktop.Tracker', '/org/freedesktop/Tracker/Search')
    iface = dbus.Interface(obj, 'org.freedesktop.Tracker.Search')
    
    items = iface.Query(-1, 'Music', ['File:Added', 'File:Name', 'Audio:Artist', 'Audio:Duration', 'Audio:Genre',
    ['Audio:Title'], '', [], filter, False, ['Audio:Title'], False, offset, limit)
    
    result = {}
    songs = []
    
    id = json.dumps({'filter': filter, 'offset': 0})
    
    for item in items:
        try:
            url = item[0]
            date = item[2]
            name = item[3]
            artist = item[4]
            duration = item[5]
            genre = item[6]
            title = item[7]
            thumbnail = format_thumbnail(url)
            
            song = {}
            song['artist'] = artist
            song['date'] = format_datetime(int(date))
            song['downloadable'] = False
            song['duration'] = format_duration(int(duration))
            song['genre'] = genre
            song['id'] = id
            song['largeThumbnailUrl'] = thumbnail
            song['streamUrl'] = 'file://' + url
            song['thumbnailUrl'] = thumbnail
            
            if title != '':
                song['title'] = title
            else:
                song['title'] = name[0 : name.rfind('.')]
            
            song['url'] = url
            songs.append(song)
        except:
            pass
    
    if len(items) > 0:
        result['next'] = json.dumps({'filter': filter, 'offset': offset + limit})
    
    result['items'] = songs
    return result

def search_songs(query):
    filter = '<rdfq:Condition> <rdfq:or><rdfq:contains><rdfq:Property name="File:Name" /><rdf:String>%s</rdf:String>\
</rdfq:contains><rdfq:contains><rdfq:Property name="Audio:Title" /><rdf:String>%s</rdf:String></rdfq:contains>\
</rdfq:or> </rdfq:Condition>' % (query, query)

    return list_songs(filter)

def list_albums(filter = '', offset = 0, limit = 20):
    bus = dbus.SessionBus()
    obj = bus.get_object('org.freedesktop.Tracker', '/org/freedesktop/Tracker/Search')
    iface = dbus.Interface(obj, 'org.freedesktop.Tracker.Search')
    
    items = iface.Query(-1, 'Music', ['File:Added', 'File:Name', 'Audio:Artist', 'Audio:Duration', 'Audio:Genre',
    ['Audio:Title'], '', [], filter, False, ['Audio:Title'], False, offset, limit)
    
    result = {}
    songs = []
    
    for item in items:
        try:
            url = item[0]
            date = item[2]
            name = item[3]
            duration = item[4]
            title = item[5]
            thumbnail = format_thumbnail(url)
            
            song = {}
            song['date'] = format_datetime(int(date))
            song['duration'] = format_duration(int(duration))
            song['id'] = id
            song['largeThumbnailUrl'] = thumbnail
            song['streamUrl'] = 'file://' + url
            song['thumbnailUrl'] = thumbnail
            
            if title != '':
                song['title'] = title
            else:
                song['title'] = name[0 : name.rfind('.')]
            
            song['url'] = url
            songs.append(song)
        except:
            pass
    
    if len(items) > 0:
        result['next'] = json.dumps({'filter': filter, 'order': order, 'offset': offset + limit})
    
    result['items'] = songs
    return result

def search_albums(query):
    filter = '<rdfq:Condition> <rdfq:or><rdfq:contains><rdfq:Property name="File:Name" /><rdf:String>%s</rdf:String>\
</rdfq:contains><rdfq:contains><rdfq:Property name="Audio:Title" /><rdf:String>%s</rdf:String></rdfq:contains>\
</rdfq:or> </rdfq:Condition>' % (query, query)

    return list_songs(filter)

def list_artists(filter = '', offset = 0, limit = 20):
    bus = dbus.SessionBus()
    obj = bus.get_object('org.freedesktop.Tracker', '/org/freedesktop/Tracker/Search')
    iface = dbus.Interface(obj, 'org.freedesktop.Tracker.Search')
    
    items = iface.Query(-1, 'Music', ['File:Added', 'File:Name', 'Audio:Artist', 'Audio:Duration', 'Audio:Genre',
    ['Audio:Title'], '', [], filter, False, ['Audio:Title'], False, offset, limit)
    
    result = {}
    songs = []
    
    for item in items:
        try:
            url = item[0]
            date = item[2]
            name = item[3]
            duration = item[4]
            title = item[5]
            thumbnail = format_thumbnail(url)
            
            song = {}
            song['date'] = format_datetime(int(date))
            song['duration'] = format_duration(int(duration))
            song['id'] = id
            song['largeThumbnailUrl'] = thumbnail
            song['streamUrl'] = 'file://' + url
            song['thumbnailUrl'] = thumbnail
            
            if title != '':
                song['title'] = title
            else:
                song['title'] = name[0 : name.rfind('.')]
            
            song['url'] = url
            songs.append(song)
        except:
            pass
    
    if len(items) > 0:
        result['next'] = json.dumps({'filter': filter, 'order': order, 'offset': offset + limit})
    
    result['items'] = songs
    return result

def search_artists(query):
    filter = '<rdfq:Condition> <rdfq:or><rdfq:contains><rdfq:Property name="File:Name" /><rdf:String>%s</rdf:String>\
</rdfq:contains><rdfq:contains><rdfq:Property name="Audio:Title" /><rdf:String>%s</rdf:String></rdfq:contains>\
</rdfq:or> </rdfq:Condition>' % (query, query)

    return list_songs(filter)

def list_playlists(filter = '', offset = 0, limit = 20):
    bus = dbus.SessionBus()
    obj = bus.get_object('org.freedesktop.Tracker', '/org/freedesktop/Tracker/Search')
    iface = dbus.Interface(obj, 'org.freedesktop.Tracker.Search')
    
    items = iface.Query(-1, 'Playlists', ['File:Added', 'File:Name', 'Playlist:Duration', 'Playlist:Songs'], '', [],
    filter, False, ['File:Name'], False, offset, limit)
    
    result = {}
    songs = []
    
    for item in items:
        try:
            url = item[0]
            date = item[2]
            name = item[3]
            duration = item[4]
            title = item[5]
            thumbnail = format_thumbnail(url)
            
            song = {}
            song['date'] = format_datetime(int(date))
            song['duration'] = format_duration(int(duration))
            song['id'] = id
            song['largeThumbnailUrl'] = thumbnail
            song['streamUrl'] = 'file://' + url
            song['thumbnailUrl'] = thumbnail
            
            if title != '':
                song['title'] = title
            else:
                song['title'] = name[0 : name.rfind('.')]
            
            song['url'] = url
            songs.append(song)
        except:
            pass
    
    if len(items) > 0:
        result['next'] = json.dumps({'filter': filter, 'order': order, 'offset': offset + limit})
    
    result['items'] = songs
    return result

def search_playlists(query):
    filter = '<rdfq:Condition> <rdfq:contains><rdfq:Property name="File:Name" /><rdf:String>%s</rdf:String>\
</rdfq:contains> </rdfq:Condition>' % query

    return list_playlists(filter)

def list_items(resource, id):
    if not resource or resource == 'track':
        try:
            id = json.loads(id)
            return list_songs(id['filter'], id['order'], id['offset'])
        except:
            return list_songs()
    return []
        
def search_items(resource, query, order):
    if not resource or resource == 'track':
        return search_songs(query)
    return []
        
def main(method, resource, id, query, order):
    if method == 'list':
        print json.dumps(list_items(resource, id))
    elif method == 'search':
        print json.dumps(search_items(resource, query, order))
    else:
        raise ResourceError('{"error": "Invalid method specified: %s"}' % method)

if __name__ == '__main__':
    (opts, args) = getopt.getopt(sys.argv[1:], 'm:r:i:q:o:')
    
    method = 'list'
    resource = 'track'
    id = ''
    query = ''
    order = ''
    
    for o, a in opts:
        if o == '-m':
            method = a
        elif o == '-r':
            resource = a
        elif o == '-i':
            id = a
        elif o == '-q':
            query = a
        elif o == '-o':
            order = a
    
    try:
        main(method, resource, id, query, order)
        exit(0)
    except ResourceError, e:
        print e.args[0]
        exit(1)
