#include "settings.h"
#include <QDir>
#include <QAuthenticator>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif

Settings* Settings::self = 0;

Settings::Settings(QObject *parent) :
    QSettings(QString("MusiKloud"), QString("MusiKloud"), parent),
    m_scrobble(false),
    m_archiveOnline(false),
    m_autoTransfers(true),
    m_maxConcurrentTransfers(1),
    m_searchOrder(Queries::Hotness),
    m_searchType(Queries::Tracks),
    #if (defined (SYMBIAN_OS) || (defined (MEEGO_EDITION_HARMATTAN)))
    m_activeColor("#0881cb"),
    m_activeColorString("color7"),
    #endif
    m_language("en"),
    #ifdef MEDIA_PLAYERS_ENABLED
    m_mediaplayer("musikloud"),
    #endif
    #ifndef SYMBIAN_OS
    m_monitorClipboard(false),
    #endif
    m_orientation(ScreenOrientation::Automatic),
    m_proxyType(NetworkProxyType::HttpProxy),
    m_proxyPort(80)
  #ifdef Q_WS_MAEMO_5
  ,m_eventFeedEnabled(false),
    m_eventFeedInstalled(false)
  #endif
  #ifdef IN_APP_UPDATES
  ,m_checkUpdate(false)
  #endif
{
    if (!self) {
        self = this;
    }
}

Settings::~Settings() {
    this->saveSettings();
}

Settings* Settings::instance() {
    return !self ? new Settings : self;
}

void Settings::restoreSettings() {
    this->beginGroup("Media");
    this->setScrobbleTracks(this->value("scrobbleTracks", false).toBool());
    this->setArchiveOnline(this->value("archiveOnline", false).toBool());
#if (defined (Q_WS_MAEMO_5)) || (defined (Q_WS_MAEMO_4)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->setDownloadPath(this->value("downloadPath", QString("/home/user/MyDocs/MusiKloud/")).toString());
#elif defined (SYMBIAN_OS)
    this->setDownloadPath(this->value("downloadPath", QString("E:/MusiKloud/")).toString());
#elif QT_VERSION >= 0x050000
    this->setDownloadPath(this->value("downloadPath", QStandardPaths::writableLocation(QStandardPaths::MusicLocation).append("/MusiKloud/")).toString());
#else
    this->setDownloadPath(this->value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::MusicLocation).append("/MusiKloud/")).toString());
#endif
    this->setCreateArtistSubfolders(this->value("createArtistSubfolders", true).toBool());
    this->setDownloadFormat(static_cast<AudioFormats::Format>(this->value("downloadFormat", AudioFormats::OriginalFormat).toInt()));
#ifdef MEDIA_PLAYERS_ENABLED
    this->setMediaPlayer(this->value("mediaPlayer", QString("musikloud")).toString());
#ifndef SYMBIAN_OS
    this->setMediaPlayerCommand(this->value("mediaPlayerCommand", QString()).toString());
#endif
#endif
    this->endGroup();

    this->beginGroup("Transfers");
    this->setStartTransfersAutomatically(this->value("startTransfersAutomatically", true).toBool());
    this->setMaximumConcurrentTransfers(this->value("maximumConcurrentTransfers", 1).toInt());
    this->endGroup();

    this->beginGroup("Search");
    this->setSearchOrder(static_cast<Queries::QueryOrder>(this->value("searchOrder", Queries::Hotness).toInt()));
    this->setSearchType(static_cast<Queries::QueryType>(this->value("searchType", Queries::Tracks).toInt()));
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setEventFeedEnabled((this->value("eventFeedEnabled", false).toBool()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#ifdef MEEGO_EDITION_HARMATTAN
    this->beginGroup("Appearance");
    this->setActiveColor(this->value("activeColor", QString("#0881cb")).toString());
    this->setActiveColorString(this->value("activeColorString", QString("color7")).toString());
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setLanguage(this->value("language", QString("en")).toString());
    this->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(this->value("screenOrientation", ScreenOrientation::Automatic).toInt()));
    this->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(this->value("networkProxyType", NetworkProxyType::ProxyType(NetworkProxyType::HttpProxy)).toInt()));
    this->setNetworkProxyHostName(this->value("networkProxyHostName", QString()).toString());
    this->setNetworkProxyPort(this->value("networkProxyPort", 80).toUInt());
    this->setNetworkProxyUser(this->value("networkProxyUser", QString()).toString());
    this->setNetworkProxyPassword(QString(QByteArray::fromBase64(this->value("networkProxyPassword", QByteArray()).toByteArray())));
    this->setNetworkProxy();
#ifndef SYMBIAN_OS
    this->setMonitorClipboard(this->value("monitorClipboard", false).toBool());
#endif
#ifdef IN_APP_UPDATES
    this->setCheckForUpdateOnStartup(this->value("checkForUpdateOnStartup", false).toBool());
#endif
    this->endGroup();
}

void Settings::saveSettings() {
    this->beginGroup("Media");
    this->setValue("scrobbleTracks", this->scrobbleTracks());
    this->setValue("archiveOnline", this->archiveOnline());
    this->setValue("downloadPath", this->downloadPath());
    this->setValue("createArtistSubfolders", this->createArtistSubfolders());
    this->setValue("downloadFormat", AudioFormats::Format(this->downloadFormat()));
#ifdef MEDIA_PLAYERS_ENABLED
    this->setValue("mediaPlayer", this->mediaPlayer());
#ifndef SYMBIAN_OS
    this->setValue("mediaPlayerCommand", this->mediaPlayerCommand());
#endif
#endif
    this->endGroup();

    this->beginGroup("Transfers");
    this->setValue("startTransfersAutomatically", this->startTransfersAutomatically());
    this->setValue("maximumConcurrentTransfers", this->maximumConcurrentTransfers());
    this->endGroup();

    this->beginGroup("Search");
    this->setValue("searchOrder", Queries::QueryOrder(this->searchOrder()));
    this->setValue("searchType", Queries::QueryType(this->searchType()));
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setValue("eventFeedEnabled", (this->eventFeedEnabled()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#ifdef MEEGO_EDITION_HARMATTAN
    this->beginGroup("Appearance");
    this->setValue("activeColor", this->activeColor());
    this->setValue("activeColorString", this->activeColorString());
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setValue("language", this->language());
    this->setValue("screenOrientation", ScreenOrientation::Orientation(this->screenOrientation()));
    this->setValue("networkProxyType", NetworkProxyType::ProxyType(this->networkProxyType()));
    this->setValue("networkProxyHostName", this->networkProxyHostName());
    this->setValue("networkProxyPort", this->networkProxyPort());
    this->setValue("networkProxyUser", this->networkProxyUser());
    this->setValue("networkProxyPassword", this->networkProxyPassword().toUtf8().toBase64());
#ifndef SYMBIAN_OS
    this->setValue("monitorClipboard", this->monitorClipboard());
#endif
#ifdef IN_APP_UPDATES
    this->setValue("checkForUpdateOnStartup", this->checkForUpdateOnStartup());
#endif
    this->endGroup();
}

void Settings::setScrobbleTracks(bool scrobble) {
    if (scrobble != this->scrobbleTracks()) {
        m_scrobble = scrobble;
        emit scrobbleTracksChanged(scrobble);
    }
}

void Settings::setArchiveOnline(bool online) {
    if (online != this->archiveOnline()) {
        m_archiveOnline = online;
        emit archiveOnlineChanged(online);
    }
}

void Settings::setDownloadFormat(AudioFormats::Format audioFormat) {
    if (audioFormat != this->downloadFormat()) {
        m_downloadFormat = audioFormat;
        emit downloadFormatChanged(audioFormat);
    }
}

void Settings::setStartTransfersAutomatically(bool automatic) {
    if (automatic != this->startTransfersAutomatically()) {
        m_autoTransfers = automatic;
        emit startTransfersAutomaticallyChanged(automatic);
    }
}

void Settings::setMaximumConcurrentTransfers(int maximum) {
    int oldMaximum = this->maximumConcurrentTransfers();

    if ((maximum != oldMaximum) && (maximum > 0) && (maximum <= 3)) {
        m_maxConcurrentTransfers = maximum;
        emit maximumConcurrentTransfersChanged(oldMaximum, maximum);
    }
}

void Settings::setDownloadPath(const QString &path) {
    m_path = path;

    if (!m_path.endsWith("/")) {
        m_path.append("/");
    }

    QDir pathCheck(m_path + ".artwork/");

    if (pathCheck.exists(m_path + "artwork/")) {
        pathCheck.rename(m_path + "artwork/", pathCheck.path());
    }

    pathCheck.mkpath(pathCheck.path());

    emit downloadPathChanged(m_path);
}

void Settings::setCreateArtistSubfolders(bool subfolders) {
    if (subfolders != this->createArtistSubfolders()) {
        m_subfolders = subfolders;
        emit createArtistSubfoldersChanged(subfolders);
    }
}

void Settings::setSearchOrder(Queries::QueryOrder order) {
    if (order != this->searchOrder()) {
        m_searchOrder = order;
        emit searchOrderChanged(order);
    }
}

void Settings::setSearchType(Queries::QueryType type) {
    if (type != this->searchType()) {
        m_searchType = type;
        emit searchTypeChanged(type);
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void Settings::setActiveColor(const QString &color) {
    if (color != this->activeColor()) {
        m_activeColor = color;
        emit activeColorChanged(color);
    }
}

void Settings::setActiveColorString(const QString &colorString) {
    if (colorString != this->activeColorString()) {
        m_activeColorString = colorString;
        emit activeColorStringChanged(colorString);
    }
}
#endif

void Settings::setLanguage(const QString &lang) {
    if (lang != this->language()) {
        m_language = lang;
        emit languageChanged(lang);
    }
}

#ifdef MEDIA_PLAYERS_ENABLED
void Settings::setMediaPlayer(const QString &player) {
    if (player != this->mediaPlayer()) {
        m_mediaplayer = player;
        emit mediaPlayerChanged(player);
    }
}

#ifndef SYMBIAN_OS
void Settings::setMediaPlayerCommand(const QString &command) {
    if (command != this->mediaPlayerCommand()) {
        m_command = command;
        emit mediaPlayerCommandChanged(command);
    }
}
#endif
#endif

#ifndef SYMBIAN_OS
void Settings::setMonitorClipboard(bool monitor) {
    if (monitor != this->monitorClipboard()) {
        m_monitorClipboard = monitor;
        emit monitorClipboardChanged(monitor);
    }
}
#endif

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        m_orientation = orientation;
        emit screenOrientationChanged(orientation);
    }
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType proxyType) {
    if (proxyType != this->networkProxyType()) {
        m_proxyType = proxyType;
        emit networkProxyTypeChanged(proxyType);
    }
}

void Settings::setNetworkProxyHostName(const QString &hostName) {
    if (hostName != this->networkProxyHostName()) {
        m_proxyHost = hostName;
        emit networkProxyHostNameChanged(hostName);
    }
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        m_proxyPort = port;
        emit networkProxyPortChanged(port);
    }
}

void Settings::setNetworkProxyUser(const QString &user) {
    if (user != this->networkProxyUser()) {
        m_proxyUser = user;
        emit networkProxyUserChanged(user);
    }
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        m_proxyPass = password;
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    QNetworkProxy proxy;

    if (!this->networkProxyHostName().isEmpty()) {
        proxy.setHostName(this->networkProxyHostName());
        proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));

        if (this->networkProxyPort() > 0) {
            proxy.setPort(this->networkProxyPort());
        }
        else {
            proxy.setPort(80);
        }

        if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
            proxy.setUser(this->networkProxyUser());
            proxy.setPassword(this->networkProxyPassword());
        }
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

void Settings::onNetworkProxyAuthenticationRequested(const QNetworkProxy &proxy, QAuthenticator *authenticator) {
    Q_UNUSED(proxy)

    authenticator->setUser(this->networkProxyUser());
    authenticator->setPassword(this->networkProxyPassword());
}

#ifdef Q_WS_MAEMO_5
void Settings::setEventFeedEnabled(bool enabled) {
    if (enabled != this->eventFeedEnabled()) {
        m_eventFeedEnabled = enabled;
        emit eventFeedEnabledChanged(enabled);
    }
}

bool Settings::eventFeedInstalled() const {
    return QDir().exists("/opt/hildonevents/bin/hildonevents");

}
#endif

#ifdef IN_APP_UPDATES
void Settings::setCheckForUpdateOnStartup(bool check) {
    if (check != this->checkForUpdateOnStartup()) {
        m_checkUpdate = check;
        emit checkForUpdateOnStartupChanged(check);
    }
}
#endif
