#ifndef PLAYBACKLISTMODEL_H
#define PLAYBACKLISTMODEL_H

#include "trackitem.h"
#include <QSharedPointer>
#include <QAbstractListModel>

#ifndef QML_USER_INTERFACE
class ThumbnailCache;
#endif

class PlaybackListModel : public QAbstractListModel
{
    Q_OBJECT

#if QT_VERSION >= 0x040700
    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool itemsSelected
               READ itemsSelected
               NOTIFY itemsSelectedChanged)
#endif

public:
    enum Roles {
        IdRole = Qt::UserRole + 1,
        PlaylistIdRole,
        TitleRole,
        ArtistRole,
        DateRole,
        DurationRole,
        UrlRole,
        StreamUrlRole,
#ifndef QML_USER_INTERFACE
        ThumbnailRole,
        ThumbnailPressedRole,
#endif
        ThumbnailUrlRole,
        ServiceRole,
        FavouriteRole,
        SelectedRole
    };

public:
    explicit PlaybackListModel(QObject *parent = 0);
    ~PlaybackListModel();
#if QT_VERSION >= 0x050000
    QHash<int, QByteArray> roleNames() const;
#endif
    void addTrack(QSharedPointer<TrackItem> track, bool notify = true);
    void addTracks(QList< QSharedPointer<TrackItem> > tracks, bool notify = true);
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE void addTrackFromQML(TrackItem *track, bool notify = true);
    Q_INVOKABLE void addTracksFromQML(QList<TrackItem*> tracks, bool notify = true);
#endif
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
#if QT_VERSION >= 0x040600
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
#endif
#ifndef QML_USER_INTERFACE
    bool setData(const QModelIndex &index, const QVariant &value, int role);
#endif
    QSharedPointer<TrackItem> get(int row) const;
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE TrackItem* getFromQML(int row) const;
#endif
    Q_INVOKABLE QList<int> selectedRows() const { return m_selectedRows; }
    QList< QSharedPointer<TrackItem> > selectedItems() const;
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE QList<TrackItem*> selectedItemsFromQML() const;
#endif
    inline bool itemsSelected() const { return !this->selectedRows().isEmpty(); }

    static PlaybackListModel* playbackQueue();

public slots:
    void clear();
    void removeTrack(int row);
    void removeSelectedTracks();
    void downloadSelectedTracks();
    void selectAll();
    void selectNone();
    void toggleSelected(int row);

private:
    bool removeRow(int row, const QModelIndex &parent = QModelIndex());
    void insertTrack(int row, QSharedPointer<TrackItem> track);
    void appendTrack(QSharedPointer<TrackItem> track);

#ifndef QML_USER_INTERFACE
private slots:
    void onThumbnailReady();
#endif

signals:
    void alert(const QString &message);
    void countChanged(int count);
    void itemsSelectedChanged(bool selected);

private:
    static PlaybackListModel* m_queue;
#ifndef QML_USER_INTERFACE
    ThumbnailCache *m_cache;
    int m_thumbnailPressedRow;
#endif
    QList< QSharedPointer<TrackItem> > m_list;
    QList<int> m_selectedRows;
#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> m_roleNames;
#endif
};

#endif // PLAYBACKLISTMODEL_H
