#include "connectionlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "utils.h"
#include "notifications.h"
#include "soundcloud.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

ConnectionListModel::ConnectionListModel(QObject *parent) :
    QAbstractListModel(parent),
    m_loading(true),
    m_error(false)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[DisplayNameRole] = "displayName";
    roles[ServiceRole] = "service";
    roles[IconSourceRole] = "iconSource";
    roles[SelectedRole] = "selected";
    this->setRoleNames(roles);
}

ConnectionListModel::~ConnectionListModel() {
    m_list.clear();
}

void ConnectionListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
    this->setLoading(false);
    this->setError(false);
}

void ConnectionListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getConnections();
    }
}

int ConnectionListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant ConnectionListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).id;
    case DisplayNameRole:
        return m_list.at(index.row()).displayName;
    case ServiceRole:
        return m_list.at(index.row()).service;
    case IconSourceRole:
        return m_list.at(index.row()).iconSource;
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

QVariant ConnectionListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

void ConnectionListModel::getConnections() {
    QNetworkReply *reply = SoundCloud::instance()->createReply(CONNECTIONS_FEED);
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addConnections()));
}

void ConnectionListModel::addConnections() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries = Json::parse(response, ok).toList();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            for (int i = 0; i < entries.size(); i++) {
                QVariantMap entry = entries.at(i).toMap();
                Connection connection;
                connection.id = entry.value("id").toString();
                connection.displayName = entry.value("display_name").toString();
                connection.service = entry.value("service").toString();
                connection.iconSource = QString("images/%1.png").arg(connection.service);
                this->appendConnection(connection);
            }

            this->setLoading(false);
            emit countChanged(this->rowCount());
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void ConnectionListModel::appendConnection(Connection connection) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(connection);
    this->endInsertRows();
}

QStringList ConnectionListModel::connections() const {
    QStringList connections;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        connections.append(this->data(this->index(m_selectedRows.at(i)), IdRole).toString());
    }

    return connections;
}

QStringList ConnectionListModel::connectionNames() const {
    QStringList names;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        names.append(this->data(this->index(m_selectedRows.at(i)), DisplayNameRole).toString());
    }

    return names;
}

void ConnectionListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
    emit itemsSelectedChanged(this->itemsSelected());
}
