#include "profiletab.h"
#include "../base/soundcloud.h"
#include "../base/notifications.h"
#include "../base/thumbnailcache.h"
#include "separatorlabel.h"
#include "avatarlabel.h"
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QGridLayout>
#include <QMaemo5Style>

ProfileTab::ProfileTab(const QString &userId, QWidget *parent) :
    AbstractTab(parent),
    m_avatar(new AvatarLabel(this)),
    m_channelLabel(new QLabel(this)),
    m_tracksLabel(new QLabel(this)),
    m_playlistsLabel(new QLabel(this)),
    m_followersLabel(new QLabel(this)),
    m_fullnameLabel(new QLabel(this)),
    m_locationLabel(new QLabel(this)),
    m_bandnameLabel(new QLabel(this)),
    m_websiteLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_noProfileLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No profile found")), this)),
    m_followingButton(new QPushButton(tr("Follow"))),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_channelLabel, 0, 0, 1, 3);
    grid->addWidget(m_avatar, 1, 0, 3, 1, Qt::AlignTop);
    grid->addWidget(m_tracksLabel, 1, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_playlistsLabel, 2, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_followersLabel, 3, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_followingButton, 4, 2);
    grid->addWidget(new SeparatorLabel(tr("Profile"), this), 5, 0, 1, 3);
    grid->addWidget(m_fullnameLabel, 6, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_locationLabel, 7, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_bandnameLabel, 8, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_websiteLabel, 9, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(new SeparatorLabel(tr("Description"), this), 10, 0, 1, 3);
    grid->addWidget(m_descriptionLabel, 11, 0, 1, 3);

    m_grid->setContentsMargins(0, 0 ,0, 0);
    m_grid->addWidget(m_scrollArea);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_descriptionLabel->setWordWrap(true);
    m_scrollArea->hide();
    m_noProfileLabel->hide();
    m_tracksLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_followingButton->setFixedWidth(150);
    m_followingButton->setVisible(SoundCloud::instance()->userSignedIn());

    this->connect(SoundCloud::instance(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->connect(SoundCloud::instance(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
    this->connect(m_followingButton, SIGNAL(clicked()), this, SLOT(toggleFollowing()));

    if ((userId == "me") || (userId == SoundCloud::instance()->username())) {
        SoundCloud::instance()->getCurrentUserProfile();
        m_followingButton->hide();
    }
    else {
        SoundCloud::instance()->getUserProfile(userId);
    }

    Notifications::toggleBusy(true);
    this->onOrientationChanged();
}

void ProfileTab::setLandscapeLayout() {
    m_noProfileLabel->setFixedSize(800, 400);
}

void ProfileTab::setPortraitLayout() {
    m_noProfileLabel->setFixedSize(480, 700);
}

void ProfileTab::setUser(QSharedPointer<UserItem> user) {
    m_user = user;
    m_avatar->setName(m_user.data()->username());
    m_channelLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(m_user.data()->username()));
    m_tracksLabel->setText(!m_user.data()->trackCount() ? tr("No tracks") : QString("%1 %2").arg(m_user.data()->trackCount()).arg(tr("tracks")));
    m_playlistsLabel->setText(!m_user.data()->playlistCount() ? tr("No sets") : QString("%1 %2").arg(m_user.data()->playlistCount()).arg(tr("sets")));
    m_followersLabel->setText(!m_user.data()->followersCount() ? tr("No followers") : QString("%1 %2").arg(m_user.data()->followersCount()).arg(tr("followers")));
    m_fullnameLabel->setText(QString("%1: %2").arg(tr("Name")).arg(m_user.data()->fullname().isEmpty() ? tr("Unknown") : m_user.data()->fullname()));
    m_locationLabel->setText(QString("%1: %2").arg(tr("Location")).arg(m_user.data()->country().isEmpty() ? tr("Unknown") : QString("%1, %2").arg(m_user.data()->city()).arg(m_user.data()->country())));
    m_bandnameLabel->setText(QString("%1: %2").arg(tr("Band")).arg(m_user.data()->bandname().isEmpty() ? tr("Unknown") : m_user.data()->bandname()));
    m_websiteLabel->setText(QString("%1: %2").arg(tr("Website")).arg(m_user.data()->websiteUrl().isEmpty() ? tr("Unknown") : QString("<a href='%1'>%2</a>").arg(m_user.data()->websiteUrl().toString()).arg(m_user.data()->websiteTitle())));
    m_descriptionLabel->setText(m_user.data()->description().isEmpty() ? tr("No description") : m_user.data()->description());
    m_followingButton->setText(m_user.data()->followed() ? tr("Unfollow") : tr("Follow"));

    m_scrollArea->show();

    this->setAvatar();
    Notifications::toggleBusy(false);

    this->connect(m_user.data(), SIGNAL(followedChanged(bool)), this, SLOT(onFollowedChanged(bool)));
    this->disconnect(SoundCloud::instance(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->disconnect(SoundCloud::instance(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
}

void ProfileTab::setAvatar() {
    QImage avatar = ThumbnailCache::instance()->thumbnail(m_user.data()->avatarUrl(), QSize(128, 128));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void ProfileTab::onProfileError() {
    m_noProfileLabel->show();
}

void ProfileTab::toggleFollowing() {
    if (m_user.data()->followed()) {
        SoundCloud::instance()->unfollowUser(m_user.data()->id());
    }
    else {
        SoundCloud::instance()->followUser(m_user.data()->id());
    }
}

void ProfileTab::onFollowedChanged(bool followed) {
    m_followingButton->setText(followed ? tr("Unfollow") : tr("Follow"));
}
