#include "playlisttrackswindow.h"
#include "tracklistdelegate.h"
#include "trackinfowindow.h"
#include "channelwindow.h"
#include "trackselectionwindow.h"
#include "nowplayingwindow.h"
#include "../base/tracklistmodel.h"
#include "../base/thumbnailcache.h"
#include "../base/playlistitem.h"
#include "../base/playbacklistmodel.h"
#include "../base/transfermanager.h"
#include "../base/settings.h"
#include "../base/soundcloud.h"
#include "../base/tracklauncher.h"
#include "../base/mediaplayer.h"
#include "listview.h"
#include "thumbnaillabel.h"
#include "separatorlabel.h"
#include <QAction>
#include <QLabel>
#include <QScrollArea>
#include <QPushButton>
#include <QMenuBar>
#include <QGridLayout>
#include <QScrollBar>
#include <QTimer>

PlaylistTracksWindow::PlaylistTracksWindow(QSharedPointer<PlaylistItem> playlist, QWidget *parent) :
    StackedWindow(parent),
    m_playlist(playlist),
    m_listView(new ListView(this)),
    m_model(new TrackListModel(this)),
    m_addToFavouritesAction(this->menuBar()->addAction(tr("Add to favourites"), this, SLOT(addTracksToFavourites()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueTracks()))),
    m_downloadTracksAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addTrackDownloads()))),
    m_downloadPlaylistAction(this->menuBar()->addAction(tr("Download set"), this, SLOT(addPlaylistDownload()))),
    m_contextMenu(new QMenu(this)),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addTrackToFavourites()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueTrack()))),
    m_downloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addTrackDownload()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_playlist.data()->title()), this)),
    m_descriptionLabel(new QLabel(m_playlist.data()->description().isEmpty() ? tr("No description") : m_playlist.data()->description(), this)),
    m_artistLabel(new QLabel(QString("%1: %2").arg(tr("Artist")).arg(m_playlist.data()->artist()), this)),
    m_dateLabel(new QLabel(QString("%1: %2").arg(tr("Date")).arg(m_playlist.data()->date()), this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator1(new SeparatorLabel(tr("Description"), this)),
    m_separator2(new SeparatorLabel(QString("-- %1").arg(tr("tracks")), this)),
    m_scrollArea(new QScrollArea(this))
{
    this->setWindowTitle(tr("Set"));
    this->setCentralWidget(m_scrollArea);

    TrackListDelegate *delegate = new TrackListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_titleLabel->setWordWrap(true);
    m_descriptionLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_thumbnail->setInteractive(false);
    m_thumbnail->setFixedSize(120, 120);
    m_playButton->setFixedWidth(150);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_thumbnail, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 2, Qt::AlignTop);
    grid->addWidget(m_playButton, 3, 3, Qt::AlignRight);
    grid->addWidget(m_separator1, 4, 0, 1, 4);
    grid->addWidget(m_descriptionLabel, 5, 1, 1, 3);
    grid->addWidget(m_separator2, 6, 0, 1, 4);
    grid->addWidget(m_listView, 7, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playTrack(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(playPlaylist()));

    m_model->getPlaylistTracks(m_playlist.data()->id());

    this->setThumbnail();
    this->setMenuActions();
    this->onOrientationChanged();
}

void PlaylistTracksWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void PlaylistTracksWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void PlaylistTracksWindow::setThumbnail() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_playlist.data()->thumbnailUrl(), QSize(120, 120));

    if (thumbnail.isNull()) {
        m_thumbnail->setImage(QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png"));
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
    else {
        m_thumbnail->setImage(thumbnail);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnail()));
    }
}

void PlaylistTracksWindow::setMenuActions() {
    m_addToFavouritesAction->setEnabled(SoundCloud::instance()->userSignedIn());
}

void PlaylistTracksWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_playButton->setEnabled((!loading) && (m_model->rowCount() > 0));

    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(m_model->rowCount()).arg(tr("tracks")));

        if (m_model->rowCount() > 0) {
            m_listView->setFixedHeight(m_model->rowCount() * m_listView->sizeHintForRow(0));
            this->setThumbnail();
        }
    }
}

void PlaylistTracksWindow::onCountChanged(int count) {
    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(count).arg(tr("tracks")));
    }
}

void PlaylistTracksWindow::addTracksToFavourites() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to favourites"), SLOT(addTracksToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void PlaylistTracksWindow::addTrackDownloads() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Download"), SLOT(addTrackDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void PlaylistTracksWindow::addPlaylistDownload() {
    QList< QSharedPointer<TrackItem> > tracks;

    for (int i = 0; i < m_model->rowCount(); i++) {
        tracks.append(m_model->get(i));
    }

    if (!tracks.isEmpty()) {
        TransferManager::instance()->addDownloadTransfer(m_playlist, tracks);
    }
}

void PlaylistTracksWindow::queueTracks() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to queue"), SLOT(queueTracks()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void PlaylistTracksWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void PlaylistTracksWindow::addTrackToFavourites() {
    SoundCloud::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), TrackListModel::IdRole).toString()));
}

void PlaylistTracksWindow::addTrackDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void PlaylistTracksWindow::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_listView->currentIndex().row()));
}

void PlaylistTracksWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showTrackInfo(m_model->get(index.row()));
    }
}

void PlaylistTracksWindow::showTrackInfo(QSharedPointer<TrackItem> track) {
    TrackInfoWindow *window = new TrackInfoWindow(track, this);
    window->show();
}

void PlaylistTracksWindow::playTrack(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(m_model->get(index.row()));
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        TrackLauncher::launchTrack(SoundCloud::instance()->getStreamUrl(index.data(TrackListModel::StreamUrlRole).toUrl()));
    }
}

void PlaylistTracksWindow::playPlaylist() {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        QList< QSharedPointer<TrackItem> > tracks;

        for (int i = 0; i < m_model->rowCount(); i++) {
            tracks.append(m_model->get(i));
        }

        MediaPlayer::instance()->playTracks(tracks);
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        QList<QUrl> urls;

        for (int i = 0; i < m_model->rowCount(); i++) {
            urls.append(SoundCloud::instance()->getStreamUrl(m_model->data(m_model->index(i), TrackListModel::StreamUrlRole).toUrl()));
        }

        TrackLauncher::launchTracks(urls);
    }
}

void PlaylistTracksWindow::showUserChannel() {
    ChannelWindow *window = new ChannelWindow(m_playlist.data()->artistId(), this);
    window->show();
}
