#ifndef TRANSFERITEM_H
#define TRANSFERITEM_H

#include "enums.h"
#include "trackitem.h"
#include "playlistitem.h"
#include <QObject>
#include <QUrl>
#include <QTime>
#include <QTimer>
#include <QProcess>
#include <QFile>

class TrackItem;
class PlaylistItem;
class QNetworkAccessManager;
class QNetworkReply;

class TransferItem : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QString id
               READ id
               CONSTANT)
    Q_PROPERTY(QString title
               READ title
               CONSTANT)
    Q_PROPERTY(Services::AudioService service
               READ service
               CONSTANT)
    Q_PROPERTY(Transfers::TransferStatus status
               READ status
               WRITE setStatus
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusText
               READ statusText
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusInfo
               READ statusInfo
               NOTIFY statusInfoChanged)
    Q_PROPERTY(Transfers::TransferPriority priority
               READ priority
               WRITE setPriority
               NOTIFY priorityChanged)
    Q_PROPERTY(QString priorityText
               READ priorityText
               NOTIFY priorityChanged)
    Q_PROPERTY(int progress
               READ progress
               NOTIFY progressChanged)
    Q_PROPERTY(int speed
               READ speed
               NOTIFY speedChanged)
    Q_PROPERTY(qint64 size
               READ size
               NOTIFY sizeChanged)
    Q_PROPERTY(int currentIndex
               READ currentIndex
               NOTIFY currentIndexChanged)
    Q_PROPERTY(int totalFiles
               READ totalFiles
               CONSTANT)
    Q_PROPERTY(Transfers::TransferType transferType
               READ transferType
               CONSTANT)

public:
    explicit TransferItem(QSharedPointer<TrackItem> track, Transfers::TransferStatus status = Transfers::Paused, QObject *parent = 0);
    explicit TransferItem(QSharedPointer<PlaylistItem> playlist, QList< QSharedPointer<TrackItem> > tracks, Transfers::TransferStatus status = Transfers::Paused, QObject *parent = 0);
    ~TransferItem();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString id() const { return m_id; }
    inline void setId(const QString &id) { m_id = id; }
    inline QString title() const { return m_title; }
    inline void setTitle(const QString &title) { m_title = title; }
    inline Services::AudioService service() const { return m_service; }
    inline void setService(Services::AudioService service) { m_service = service; }
    inline QString downloadPath() const { return m_downloadPath; }
    inline void setDownloadPath(const QString &path) { m_downloadPath = path; }
    inline Transfers::TransferStatus status() const { return m_status; }
    QString statusText() const;
    inline QString statusInfo() const { return m_statusInfo; }
    inline Transfers::TransferPriority priority() const { return m_priority; }
    inline int progress() const { return m_progress; }
    inline void setProgress(int progress) { m_progress = progress; }
    inline int speed() const { return m_speed; }
    inline void setSpeed(int speed) { m_speed = speed; }
    void setStatus(Transfers::TransferStatus status);
    void setStatusInfo(const QString &info);
    QString priorityText() const;
    inline qint64 size() const { return m_size; }
    void setSize(qint64 size);
    inline int currentIndex() const { return m_currentIndex; }
    void setCurrentIndex(int index);
    inline int totalFiles() const { return m_totalFiles; }
    inline void setTotalFiles(int total) { m_totalFiles = total; }
    inline Transfers::TransferType transferType() const { return m_transferType; }
    inline void setTransferType(Transfers::TransferType type) { m_transferType = type; }
    inline QSharedPointer<TrackItem> track() const { return m_track; }
    inline QSharedPointer<PlaylistItem> playlist() const { return m_playlist; }
    inline QList< QSharedPointer<TrackItem> > playlistTracks() const { return m_playlistTracks; }

public slots:
    void startTransfer(bool resetRetries = true);
    void setPriority(Transfers::TransferPriority priority);

private:
    void pauseTransfer();
    void cancelTransfer();
    void downloadThumbnail();
    void downloadWaveform();
    void downloadPlaylistThumbnail();
    void moveDownloadedFiles();

private slots:
    void performDownload(const QUrl &url, AudioFormats::Format audioFormat = AudioFormats::Unknown);
    void onDownloadMetadataChanged();
    void onDownloadReadyRead();
    void onDownloadProgressChanged(qint64 received, qint64 total);
    void updateProgress();
    void onDownloadFinished();
    void onThumbnailDownloadFinished();
    void onWaveformDownloadFinished();
    void onPlaylistThumbnailDownloadFinished();

signals:
    void progressChanged(int progress);
    void speedChanged(int speed);
    void statusChanged(Transfers::TransferStatus status);
    void statusInfoChanged(const QString &info);
    void priorityChanged(Transfers::TransferPriority priority);
    void sizeChanged(qint64 size);
    void currentIndexChanged(int index);
    void trackDownloadCompleted(QSharedPointer<TrackItem>);
    void playlistDownloadCompleted(QSharedPointer<PlaylistItem>);

private:
    QNetworkAccessManager *m_nam;
    QNetworkReply *m_reply;
    QSharedPointer<PlaylistItem> m_playlist;
    QList< QSharedPointer<TrackItem> > m_playlistTracks;
    QSharedPointer<TrackItem> m_track;
    QTimer m_progressTimer;
    QFile m_file;
    QString m_id;
    QString m_title;
    Services::AudioService m_service;
    QString m_downloadPath;
    Transfers::TransferStatus m_status;
    QString m_statusText;
    QString m_statusInfo;
    Transfers::TransferPriority m_priority;
    int m_progress;
    int m_speed;
    qint64 m_resumeSize;
    qint64 m_size;
    int m_currentIndex;
    int m_totalFiles;
    QString m_priorityText;
    QTime m_transferTime;
    AudioFormats::Format m_format;
    Transfers::TransferType m_transferType;
    int m_retries;
};

#endif // TRANSFERITEM_H
