#include "tracklistmodel.h"
#include "soundcloud.h"
#include "feedurls.h"
#include "json.h"
#include "utils.h"
#include "notifications.h"
#include "playbacklistmodel.h"
#include "transfermanager.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

TrackListModel::TrackListModel(QObject * parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    m_thumbnailPressedRow(-1),
    #endif
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(0),
    m_refresh(false)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[PlaylistIdRole] = "playlistId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[DurationRole] = "duration";
    roles[UrlRole] = "url";
    roles[StreamUrlRole] = "streamUrl";
#ifndef QML_USER_INTERFACE
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
#endif
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[ServiceRole] = "service";
    roles[FavouriteRole] = "favourite";
    roles[SelectedRole] = "selected";
    this->setRoleNames(roles);
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

TrackListModel::~TrackListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void TrackListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void TrackListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreTracks();
    }
}

int TrackListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant TrackListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case PlaylistIdRole:
        return m_list.at(index.row()).data()->playlistId();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case DurationRole:
        return m_list.at(index.row()).data()->duration();
    case UrlRole:
        return m_list.at(index.row()).data()->url();
    case StreamUrlRole:
        return m_list.at(index.row()).data()->streamUrl();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(64, 64));
    case ThumbnailPressedRole:
        return QVariant(m_thumbnailPressedRow == index.row());
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case ServiceRole:
        return m_list.at(index.row()).data()->service();
    case FavouriteRole:
        return m_list.at(index.row()).data()->favourite();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

QVariant TrackListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

#ifndef QML_USER_INTERFACE
bool TrackListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case ThumbnailPressedRole:
        m_thumbnailPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}
#endif

QSharedPointer<TrackItem> TrackListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<TrackItem>(m_list.at(row));
    }

    return QSharedPointer<TrackItem>();
}

#ifdef QML_USER_INTERFACE
TrackItem* TrackListModel::getFromQML(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void TrackListModel::insertTrack(int row, QSharedPointer<TrackItem> track) {
    Q_ASSERT((row >= 0) && (row <= this->rowCount()));

    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, track);
    this->endInsertRows();
}

void TrackListModel::appendTrack(QSharedPointer<TrackItem> track) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(track);
    this->endInsertRows();
}

bool TrackListModel::removeTrack(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool TrackListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

void TrackListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
}

void TrackListModel::selectAll() {
    m_selectedRows.clear();

    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}

void TrackListModel::selectNone() {
    m_selectedRows.clear();

    if (this->rowCount() > 0) {
        emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    }
}

QList< QSharedPointer<TrackItem> > TrackListModel::selectedItems() const {
    QList< QSharedPointer<TrackItem> > items;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->get(m_selectedRows.at(i)));
    }

    return items;
}

#ifdef QML_USER_INTERFACE
QList<TrackItem*> TrackListModel::selectedItemsFromQML() const {
    QList<TrackItem*> items;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->getFromQML(m_selectedRows.at(i)));
    }

    return items;
}
#endif

#ifndef QML_USER_INTERFACE
void TrackListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif

void TrackListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        this->disconnect(SoundCloud::instance());

        if (feed == FAVOURITES_FEED) {
            this->connect(SoundCloud::instance(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        }
        else if (feed.toString().startsWith("http://api.soundcloud.com/playlists")) {
            this->setPlaylistId(feed.toString().section('/', -1).section('.', 0, 0));
        }
        else if (feed.toString().startsWith("http://api.soundcloud.com/groups")) {
            this->setGroupId(feed.toString().section('/', -2, -2));
        }

        emit feedChanged(feed);
    }
}

void TrackListModel::getMoreTracks() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getTracks();
        }
    }
}

void TrackListModel::getTracks(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addTracks()));
}

void TrackListModel::getPlaylistTracks(const QString &id) {
    this->setFeed(QUrl(QString("http://api.soundcloud.com/playlists/%1.json?limit=200").arg(id)));
    this->getTracks();
}

void TrackListModel::getGroupTracks(const QString &id) {
    this->setFeed(QUrl(QString("http://api.soundcloud.com/groups/%1/tracks.json?limit=30").arg(id)));
    this->getTracks();
}

void TrackListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = SoundCloud::instance()->createSearchReply(Queries::Tracks, this->query(), this->order(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addTracks()));
}

void TrackListModel::addTracks() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 200:
    case 201: {
        QString response(reply->readAll());
        bool ok;
        QVariantList entries;

        if (!this->playlistId().isEmpty()) {
            entries = Json::parse(response, ok).toMap().value("tracks").toList();
        }
        else {
            entries = Json::parse(response, ok).toList();
        }

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
            Notifications::instance()->onError(tr("Cannot parse server response"));
        }
        else {
            if (!this->playlistId().isEmpty()) {
                for (int i = 0; i < entries.size(); i++) {
                    TrackItem *track = new TrackItem(entries.at(i).toMap());
                    track->setPlaylistId(this->playlistId());
                    track->setTrackNumber(i + 1);
                    QMetaObject::invokeMethod(this, "appendTrack", Qt::QueuedConnection, Q_ARG(QSharedPointer<TrackItem>, QSharedPointer<TrackItem>(track)));
                }
            }
            else {
                for (int i = 0; i < entries.size(); i++) {
                    QMetaObject::invokeMethod(this, "appendTrack", Qt::QueuedConnection, Q_ARG(QSharedPointer<TrackItem>, QSharedPointer<TrackItem>(new TrackItem(entries.at(i).toMap()))));
                }
            }

            this->setMoreResults(!entries.isEmpty());
            this->setOffset(this->offset() + entries.size());

            if ((!this->playlistId().isEmpty()) && (this->moreResults())) {
                this->getTracks();
            }
            else {
                QTimer::singleShot(1000, this, SLOT(stopLoading()));
            }
        }
    }
        break;
    default:
        this->setLoading(false);
        this->setError(true);
        Notifications::instance()->onError(Utils::httpErrorString(statusCode));
        break;
    }

    reply->deleteLater();
}

void TrackListModel::onFavouriteChanged(const QString &id, bool favourite) {
    if (favourite) {
        this->setRefreshRequired(true);
    }
    else if (this->removeTrack(id)) {
        emit countChanged(this->rowCount());
    }
}

void TrackListModel::addSelectedTracksToFavourites() {
    QStringList ids;

    foreach (int row, this->selectedRows()) {
        ids << this->data(this->index(row), IdRole).toString();
    }

    SoundCloud::instance()->addToFavourites(ids);
    this->selectNone();
}

void TrackListModel::deleteSelectedTracksFromFavourites() {
    if (this->feed() == FAVOURITES_FEED) {
        QStringList ids;

        foreach (int row, this->selectedRows()) {
            ids << this->data(this->index(row), IdRole).toString();
        }

        SoundCloud::instance()->deleteFromFavourites(ids);
    }

    this->selectNone();
}

void TrackListModel::downloadSelectedTracks() {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems());
    this->selectNone();
}

void TrackListModel::queueSelectedTracks() {
    PlaybackListModel::playbackQueue()->addTracks(this->selectedItems());
    this->selectNone();
}
