#include "settings.h"
#include <QDir>
#include <QDesktopServices>
#include <QNetworkProxy>
#include <QAuthenticator>
#if (defined (SYMBIAN_OS)) && (defined (CUTETUBE_PLAYER_ENABLED))
#include <QSystemDeviceInfo>
#endif

Settings* settingsInstance = 0;

Settings::Settings(QObject *parent) :
    QSettings(QString("MusiKloud"), QString("MusiKloud"), parent),
    m_scrobble(false),
    m_archiveOnline(false),
    m_autoTransfers(true),
    m_maxConcurrentTransfers(1),
    m_searchOrder(Queries::Hotness),
    m_searchType(Queries::Tracks),
  #if (defined (SYMBIAN_OS) || (defined (MEEGO_EDITION_HARMATTAN)))
    m_theme("dark"),
    m_activeColor("#0881cb"),
    m_activeColorString("color7"),
  #endif
    m_language("en"),
    m_mediaplayer("musikloud"),
    m_orientation(ScreenOrientation::Automatic),
    m_monitorClipboard(false),
    m_proxyType(NetworkProxyType::HttpProxy),
    m_proxyPort(80)
  #ifdef Q_WS_MAEMO_5
    ,m_eventFeedEnabled(false),
    m_eventFeedInstalled(false)
  #endif
{
    if (!settingsInstance) {
        settingsInstance = this;
    }
}

Settings::~Settings() {
    this->saveSettings();
}

Settings* Settings::instance() {
    return settingsInstance;
}

void Settings::restoreSettings() {
    this->beginGroup("Media");
    this->setScrobbleTracks(this->value("scrobbleTracks", false).toBool());
    this->setArchiveOnline(this->value("archiveOnline", false).toBool());
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->setDownloadPath(this->value("downloadPath", QString("/home/user/MyDocs/MusiKloud/")).toString());
#elif defined (SYMBIAN_OS)
    this->setDownloadPath(this->value("downloadPath", QString("E:/MusiKloud/")).toString());
#else
    this->setDownloadPath(this->value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::HomeLocation).append("/MusiKloud/")).toString());
#endif
    this->setCreateArtistSubfolders(this->value("createArtistSubfolders", true).toBool());
    this->setDownloadFormat(static_cast<AudioFormats::Format>(this->value("downloadFormat", AudioFormats::OriginalFormat).toInt()));
    this->setMediaPlayer(this->value("mediaPlayer", QString("musikloud")).toString());
#ifndef SYMBIAN_OS
    this->setMediaPlayerCommand(this->value("mediaPlayerCommand", QString()).toString());
#endif
    this->endGroup();

    this->beginGroup("Transfers");
    this->setStartTransfersAutomatically(this->value("startTransfersAutomatically", true).toBool());
    this->setMaximumConcurrentTransfers(this->value("maximumConcurrentTransfers", 1).toInt());
    this->endGroup();

    this->beginGroup("Search");
    this->setSearchOrder(static_cast<Queries::QueryOrder>(this->value("searchOrder", Queries::Hotness).toInt()));
    this->setSearchType(static_cast<Queries::QueryType>(this->value("searchType", Queries::Tracks).toInt()));
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setEventFeedEnabled((this->value("eventFeedEnabled", false).toBool()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Appearance");
#ifdef SYMBIAN_OS
    this->setTheme(this->value("theme", QString("dark")).toString());
#else
    this->setTheme(this->value("theme", QString("dark_plain")).toString());
    this->setActiveColor(this->value("activeColor", QString("#0881cb")).toString());
    this->setActiveColorString(this->value("activeColorString", QString("color7")).toString());
#endif
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setLanguage(this->value("language", QString("en")).toString());
    this->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(this->value("screenOrientation", ScreenOrientation::Automatic).toInt()));
    this->setMonitorClipboard(this->value("monitorClipboard", false).toBool());
    this->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(this->value("networkProxyType", NetworkProxyType::HttpProxy).toInt()));
    this->setNetworkProxyHostName(this->value("networkProxyHostName", QString()).toString());
    this->setNetworkProxyPort(this->value("networkProxyPort", 80).toUInt());
    this->setNetworkProxyUser(this->value("networkProxyUser", QString()).toString());
    this->setNetworkProxyPassword(QString(QByteArray::fromBase64(this->value("networkProxyPassword", QByteArray()).toByteArray())));
    this->setNetworkProxy();
    this->endGroup();
}

void Settings::saveSettings() {
    this->beginGroup("Media");
    this->setValue("scrobbleTracks", this->scrobbleTracks());
    this->setValue("archiveOnline", this->archiveOnline());
    this->setValue("downloadPath", this->downloadPath());
    this->setValue("createArtistSubfolders", this->createArtistSubfolders());
    this->setValue("downloadFormat", AudioFormats::Format(this->downloadFormat()));
    this->setValue("mediaPlayer", this->mediaPlayer());
#ifndef SYMBIAN_OS
    this->setValue("mediaPlayerCommand", this->mediaPlayerCommand());
#endif
    this->endGroup();

    this->beginGroup("Transfers");
    this->setValue("startTransfersAutomatically", this->startTransfersAutomatically());
    this->setValue("maximumConcurrentTransfers", this->maximumConcurrentTransfers());
    this->endGroup();

    this->beginGroup("Search");
    this->setValue("searchOrder", Queries::QueryOrder(this->searchOrder()));
    this->setValue("searchType", Queries::QueryType(this->searchType()));
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setValue("eventFeedEnabled", (this->eventFeedEnabled()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Appearance");
    this->setValue("theme", this->theme());
#ifndef SYMBIAN_OS
    this->setValue("activeColor", this->activeColor());
    this->setValue("activeColorString", this->activeColorString());
#endif
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setValue("language", this->language());
    this->setValue("screenOrientation", ScreenOrientation::Orientation(this->screenOrientation()));
    this->setValue("monitorClipboard", this->monitorClipboard());
    this->setValue("networkProxyType", NetworkProxyType::ProxyType(this->networkProxyType()));
    this->setValue("networkProxyHostName", this->networkProxyHostName());
    this->setValue("networkProxyPort", this->networkProxyPort());
    this->setValue("networkProxyUser", this->networkProxyUser());
    this->setValue("networkProxyPassword", this->networkProxyPassword().toUtf8().toBase64());
    this->endGroup();
}

void Settings::setScrobbleTracks(bool scrobble) {
    if (scrobble != this->scrobbleTracks()) {
        m_scrobble = scrobble;
        emit scrobbleTracksChanged(scrobble);
    }
}

void Settings::setArchiveOnline(bool online) {
    if (online != this->archiveOnline()) {
        m_archiveOnline = online;
        emit archiveOnlineChanged(online);
    }
}

void Settings::setDownloadFormat(AudioFormats::Format audioFormat) {
    if (audioFormat != this->downloadFormat()) {
        m_downloadFormat = audioFormat;
        emit downloadFormatChanged(audioFormat);
    }
}

void Settings::setStartTransfersAutomatically(bool automatic) {
    if (automatic != this->startTransfersAutomatically()) {
        m_autoTransfers = automatic;
        emit startTransfersAutomaticallyChanged(automatic);
    }
}

void Settings::setMaximumConcurrentTransfers(int maximum) {
    int oldMaximum = this->maximumConcurrentTransfers();

    if ((maximum != oldMaximum) && (maximum > 0) && (maximum <= 3)) {
        m_maxConcurrentTransfers = maximum;
        emit maximumConcurrentTransfersChanged(oldMaximum, maximum);
    }
}

void Settings::setDownloadPath(const QString &path) {
    m_path = path;

    if (!m_path.endsWith("/")) {
        m_path.append("/");
    }

    QDir pathCheck(m_path + ".artwork/");

    if (pathCheck.exists(m_path + "artwork/")) {
        pathCheck.rename(m_path + "artwork/", pathCheck.path());
    }

    pathCheck.mkpath(pathCheck.path());

    emit downloadPathChanged(m_path);
}

void Settings::setCreateArtistSubfolders(bool subfolders) {
    if (subfolders != this->createArtistSubfolders()) {
        m_subfolders = subfolders;
        emit createArtistSubfoldersChanged(subfolders);
    }
}

void Settings::setSearchOrder(Queries::QueryOrder order) {
    if (order != this->searchOrder()) {
        m_searchOrder = order;
        emit searchOrderChanged(order);
    }
}

void Settings::setSearchType(Queries::QueryType type) {
    if (type != this->searchType()) {
        m_searchType = type;
        emit searchTypeChanged(type);
    }
}

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
void Settings::setTheme(const QString &theme) {
    if (theme != this->theme()) {
        m_theme = theme;
        emit themeChanged(theme);
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void Settings::setActiveColor(const QString &color) {
    if (color != this->activeColor()) {
        m_activeColor = color;
        emit activeColorChanged(color);
    }
}

void Settings::setActiveColorString(const QString &colorString) {
    if (colorString != this->activeColorString()) {
        m_activeColorString = colorString;
        emit activeColorStringChanged(colorString);
    }
}
#endif
#endif

void Settings::setLanguage(const QString &lang) {
    if (lang != this->language()) {
        m_language = lang;
        emit languageChanged(lang);
    }
}

void Settings::setMediaPlayer(const QString &player) {
    if (player != this->mediaPlayer()) {
        m_mediaplayer = player;
        emit mediaPlayerChanged(player);
    }
}

#ifndef SYMBIAN_OS
void Settings::setMediaPlayerCommand(const QString &command) {
    if (command != this->mediaPlayerCommand()) {
        m_command = command;
        emit mediaPlayerCommandChanged(command);
    }
}
#endif

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        m_orientation = orientation;
        emit screenOrientationChanged(orientation);
    }
}

void Settings::setMonitorClipboard(bool monitor) {
    if (monitor != this->monitorClipboard()) {
        m_monitorClipboard = monitor;
        emit monitorClipboardChanged(monitor);
    }
}

QString Settings::networkProxyTypeString() const {
    switch (this->networkProxyType()) {
    case NetworkProxyType::Socks5Proxy:
        return QString("SOCKS5");
    case NetworkProxyType::HttpProxy:
        return QString("HTTP");
    case NetworkProxyType::HttpCachingProxy:
        return QString("HTTP %1").arg(tr("caching"));
    default:
        return QString("HTTP");
    }
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType proxyType) {
    if (proxyType != this->networkProxyType()) {
        m_proxyType = proxyType;
        emit networkProxyTypeChanged(proxyType);
        emit networkProxyTypeStringChanged(this->networkProxyTypeString());
    }
}

void Settings::setNetworkProxyHostName(const QString &hostName) {
    if (hostName != this->networkProxyHostName()) {
        m_proxyHost = hostName;
        emit networkProxyHostNameChanged(hostName);
    }
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        m_proxyPort = port;
        emit networkProxyPortChanged(port);
    }
}

void Settings::setNetworkProxyUser(const QString &user) {
    if (user != this->networkProxyUser()) {
        m_proxyUser = user;
        emit networkProxyUserChanged(user);
    }
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        m_proxyPass = password;
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    QNetworkProxy proxy;

    if (!this->networkProxyHostName().isEmpty()) {
        proxy.setHostName(this->networkProxyHostName());
        proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));

        if (this->networkProxyPort() > 0) {
            proxy.setPort(this->networkProxyPort());
        }
        else {
            proxy.setPort(80);
        }

        if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
            proxy.setUser(this->networkProxyUser());
            proxy.setPassword(this->networkProxyPassword());
        }
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

void Settings::onNetworkProxyAuthenticationRequested(const QNetworkProxy &proxy, QAuthenticator *authenticator) {
    Q_UNUSED(proxy)

    authenticator->setUser(this->networkProxyUser());
    authenticator->setPassword(this->networkProxyPassword());
}

#ifdef Q_WS_MAEMO_5
void Settings::setEventFeedEnabled(bool enabled) {
    if (enabled != this->eventFeedEnabled()) {
        m_eventFeedEnabled = enabled;
        emit eventFeedEnabledChanged(enabled);
    }
}

bool Settings::eventFeedInstalled() const {
    return QDir().exists("/opt/hildonevents/bin/hildonevents");

}
#endif
