/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "bluetooth.h"
#include "bluetooth_p.h"
#include <QBluetoothDeviceDiscoveryAgent>
#include <QBluetoothServiceDiscoveryAgent>
#include <QBluetoothAddress>
#include <QBluetoothTransferManager>
#include <QBluetoothTransferRequest>
#include <QBluetoothTransferReply>
#include <QFile>
#include <QDebug>

Bluetooth::Bluetooth(QObject *parent) :
    QObject(parent),
    d_ptr(new BluetoothPrivate(this))
{
    qRegisterMetaType<QBluetoothTransferReply*>("QBluetoothTransferReply*");
}

Bluetooth::Bluetooth(BluetoothPrivate &dd, QObject *parent) :
    QObject(parent),
    d_ptr(&dd)
{
}

Bluetooth::~Bluetooth() {}

void Bluetooth::discoverDevices() {
    Q_D(Bluetooth);

    if (!d->deviceAgent) {
        d->deviceAgent = new QBluetoothDeviceDiscoveryAgent(this);
        this->connect(d->deviceAgent, SIGNAL(finished()), this, SLOT(_q_onDeviceDiscoveryFinished()));
    }

    d->deviceAgent->start();
}

void Bluetooth::discoverServices() {
    Q_D(Bluetooth);

    if (!d->serviceAgent) {
        d->serviceAgent = new QBluetoothServiceDiscoveryAgent(this);
        this->connect(d->serviceAgent, SIGNAL(finished()), this, SLOT(_q_onServiceDiscoveryFinished()));
    }

    d->serviceAgent->start();
}

void Bluetooth::send(const QString &address, const QString &data) {
    qDebug() << address << data;
    Q_D(Bluetooth);

    if (!d->socket) {
        d->socket = new QBluetoothSocket(QBluetoothSocket::RfcommSocket, this);
        this->connect(d->socket, SIGNAL(connected()), this, SLOT(_q_onSocketConnected()));
        this->connect(d->socket, SIGNAL(disconnected()), this, SLOT(_q_onSocketDisconnected()));
        this->connect(d->socket, SIGNAL(readyRead()), this, SLOT(_q_onSocketReadyRead()));
        this->connect(d->socket, SIGNAL(stateChanged(QBluetoothSocket::SocketState)), this, SLOT(_q_onSocketStateChanged(QBluetoothSocket::SocketState)));
        this->connect(d->socket, SIGNAL(error(QBluetoothSocket::SocketError)), this, SLOT(_q_onSocketError(QBluetoothSocket::SocketError)));
    }
}

void Bluetooth::sendFile(const QString &address, const QString &path) {
    qDebug() << address << path;
    Q_D(Bluetooth);

    if (!d->manager) {
        d->manager = new QBluetoothTransferManager(this);
    }

    if (!d->file) {
        d->file = new QFile(this);
    }
    else if (d->file->isOpen()) {
        d->file->close();
    }

    d->file->setFileName(path);
    d->file->open(QIODevice::ReadOnly);

    QBluetoothAddress btAddress(address);
    QBluetoothTransferRequest request(btAddress);
    QBluetoothTransferReply *reply = d->manager->put(request, d->file);
    this->connect(reply, SIGNAL(finished(QBluetoothTransferReply*)), this, SLOT(_q_onReplyFinished(QBluetoothTransferReply*)));
}

BluetoothPrivate::BluetoothPrivate(Bluetooth *parent) :
    q_ptr(parent),
    deviceAgent(0),
    serviceAgent(0),
    socket(0),
    manager(0),
    file(0)
{
}

BluetoothPrivate::~BluetoothPrivate() {}

void BluetoothPrivate::_q_onDeviceDiscoveryFinished() {
    qDebug() << deviceAgent->errorString();

    foreach (QBluetoothDeviceInfo info, deviceAgent->discoveredDevices()) {
        qDebug() << info.address().toString();
    }
}

void BluetoothPrivate::_q_onServiceDiscoveryFinished() {
    qDebug() << serviceAgent->errorString();

    foreach (QBluetoothServiceInfo info, serviceAgent->discoveredServices()) {
        qDebug() << "Address:" << info.device().address().toString()
                 << "Service Uuid:" << info.serviceUuid().toString()
                 << "Service name:" << info.serviceName()
                 << "Socket protocol:" << info.socketProtocol();
    }
}

void BluetoothPrivate::_q_onSocketConnected() {
    qDebug() << "Socket connected";
}

void BluetoothPrivate::_q_onSocketDisconnected() {
    qDebug() << "Socket disconnected";
}

void BluetoothPrivate::_q_onSocketReadyRead() {
    qDebug() << "Socket readyRead:" << socket->readAll();
}

void BluetoothPrivate::_q_onSocketStateChanged(QBluetoothSocket::SocketState state) {
    qDebug() << "Socket state changed:" << state;
}

void BluetoothPrivate::_q_onSocketError(QBluetoothSocket::SocketError error) {
    qDebug() << "Socket error:" << error;
}

void BluetoothPrivate::_q_onReplyFinished(QBluetoothTransferReply *reply) {
    qDebug() << reply->errorString();

    reply->deleteLater();
}

#include "moc_bluetooth.cpp"
