/*
 * Copyright (C) 2007 Milosz Wasilewski.
 *
 * This file is part of Stopwatch.
 *
 * Stopwatch is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * Stopwatch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Stopwatch.  If not, see <http://www.gnu.org/licenses/>.
 */


/* Includes */
#include <unistd.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon.h>
#include <gtk/gtkmain.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <glib-object.h>

#include "stopwatch.h"

void update_clock();

/* clock formatting function */
void pretty_print(gdouble seconds, char * buffer) 
{
	gint hours = seconds/3600;
	gint minutes = seconds/60 - (hours * 3600);
	gdouble secs = seconds - (minutes * 60) - (hours * 3600);
	if (secs < 10) 
	{
		sprintf(buffer, 
			"%02d:%02d:0%2.2f", 
			hours, 
			minutes, 
			secs);
	}
	else
	{
		sprintf(buffer, 
			"%02d:%02d:%2.2f", 
			hours, 
			minutes, 
			secs);
	}
}

void adjust_start_button(struct ApplicationState * state) 
{
	if (state->state == STOPWATCH_STOPPED | 
			state->state == STOPWATCH_ZERO) 
	{
		gtk_button_set_label(GTK_BUTTON(start_button), "Start");
	}
	else {
		g_debug("Stopwatch type: %d", state->type);
		switch (state->type) 
		{
			case STOPWATCH_SPLIT:
				/* change start button label to "split" */
				gtk_button_set_label(
					GTK_BUTTON(start_button), "Split");
				break;
			case STOPWATCH_LAP:
				/* change start button label to "lap" */
				gtk_button_set_label(
					GTK_BUTTON(start_button), "Lap");
				break;
			default:
				g_debug("What to do?");
				break;
		}
	}
}

/* callbacks */
void combo_changed_cb (GtkComboBox *widget, gpointer user_data)
{
	/*switch (gtk_combo_box_get_active (GTK_COMBO_BOX(widget)))*/
	switch (hildon_picker_button_get_active (HILDON_PICKER_BUTTON(widget)))
	{
		case STOPWATCH_SPLIT:
			current_state.type = STOPWATCH_SPLIT;
			break;
		case STOPWATCH_LAP:
			current_state.type = STOPWATCH_LAP;
			break;
		default:
			break;
	}
}

void table_row_inserted_cb(GtkTreeModel *tree_model,
	GtkTreePath  *path,
	GtkTreeIter  *iter,
	gpointer      user_data) 
{
	GdkRectangle rect;
	gint y;
	 
	gtk_tree_view_get_background_area (GTK_TREE_VIEW (stopwatch_table),
		path, NULL, &rect);
	gtk_tree_view_convert_bin_window_to_tree_coords (GTK_TREE_VIEW (stopwatch_table),
		0, rect.y, NULL, &y);
	hildon_pannable_area_jump_to (scrolled_win, -1, y);
}

void start_stopwatch_cb (GtkButton *button, gpointer user_data) 
{
	gulong *micro = NULL;
	char new_label[25] = {0};
	GtkTreeIter iter;
	GtkAdjustment *adjustment;
	gdouble value = 0.0;
	gdouble lower = 0.0;
	gdouble upper = 0.0;
	gdouble page_size = 0.0;
	/* make combo inactive
	 * user should not be able to change the stopwatch type
	 * when the stopwatch is running
	 */
	gtk_widget_set_sensitive(GTK_WIDGET(stopwatch_type_combo), FALSE);
	if (current_state.state == STOPWATCH_ZERO) 
	{
		/*  make start inactive */ 
		/* gtk_widget_set_sensitive(GTK_WIDGET(start_button), FALSE); */
		current_state.state = STOPWATCH_RUNNING;
						     
		adjust_start_button(&current_state);
		/* make stop active */
		gtk_widget_set_sensitive(GTK_WIDGET(stop_button), TRUE);

		/* make reset inactive while clock is running */
		gtk_widget_set_sensitive(GTK_WIDGET(reset_button), FALSE);

		current_state.main_timer = g_timer_new();
	
		/* start updating thread */
		current_state.func_id = gtk_timeout_add(150,
			(GtkFunction)update_clock,
			NULL);
	}
	else if (current_state.state == STOPWATCH_RUNNING) 
	{
		/* g_debug ("Running in lap or split state"); */
		/* add another record to results table */
		switch (current_state.type)
		{
			case STOPWATCH_SPLIT:
				/* add current timer to stopwatch_table_model */
				pretty_print(current_state.elapsed 
					+ current_state.stop_value, 
					new_label);
				gtk_tree_store_prepend(
					GTK_TREE_STORE(stopwatch_table_model),
					&iter,
					NULL);
				gtk_tree_store_set(
					GTK_TREE_STORE(stopwatch_table_model),
					&iter,
					0, current_state.num_rows,
					1, new_label,
					-1);
				current_state.num_rows++;
				break;
			case STOPWATCH_LAP:
				/* add current timer to stopwatch_table_model */
				pretty_print(current_state.elapsed 
					+ current_state.stop_value, 
					new_label);
				gtk_tree_store_prepend(
					GTK_TREE_STORE(stopwatch_table_model),
					&iter,
					NULL);
				gtk_tree_store_set(
					GTK_TREE_STORE(stopwatch_table_model),
					&iter,
					0, current_state.num_rows,
					1, new_label,
					-1);
				current_state.num_rows++;
				g_timer_reset(current_state.main_timer);
				current_state.total_elapsed += current_state.elapsed;
				current_state.stop_value = 0;	
				break;
			default:
				break;
		}
	}
}

void stop_stopwatch_cb (GtkButton *button, gpointer user_data) 
{
	gulong *micro = NULL;
	char new_label[25] = {0};
	current_state.state = STOPWATCH_STOPPED;
	/*  make start inactive */ 
	gtk_widget_set_sensitive(GTK_WIDGET(start_button), FALSE); 
	adjust_start_button(&current_state);
	
	/* make stop inactive */
	gtk_widget_set_sensitive(GTK_WIDGET(stop_button), FALSE);

	/* make reset active */
	gtk_widget_set_sensitive(GTK_WIDGET(reset_button), TRUE);
	
	g_timer_stop(current_state.main_timer);
	current_state.elapsed = g_timer_elapsed(current_state.main_timer, micro);
	current_state.total_elapsed += current_state.elapsed;
	if (current_state.type == STOPWATCH_SPLIT) {
		pretty_print(current_state.elapsed + current_state.stop_value, new_label);
	} else {
		pretty_print(current_state.total_elapsed, new_label);
	}
	/* sprintf(new_label, "%10.2f", elapsed); */
	/* gtk_label_set_text(GTK_LABEL(stopwatch_clock), new_label);*/
	hildon_entry_set_text(GTK_ENTRY(stopwatch_clock), new_label);
	
	current_state.stop_value += current_state.elapsed;
	g_source_remove(current_state.func_id);
}

void reset_stopwatch_cb (GtkButton *button, gpointer user_data) 
{
	current_state.state = STOPWATCH_ZERO;
	/*  make start active */ 
	gtk_widget_set_sensitive(GTK_WIDGET(start_button), TRUE);
	adjust_start_button(&current_state);
	
	/* make combo active */
	gtk_widget_set_sensitive(GTK_WIDGET(stopwatch_type_combo), TRUE);
	
	/* make stop inactive */
	gtk_widget_set_sensitive(GTK_WIDGET(stop_button), FALSE);

	/* make reset active */
	gtk_widget_set_sensitive(GTK_WIDGET(reset_button), FALSE);

	g_timer_reset(current_state.main_timer);
	current_state.stop_value = 0;
	current_state.num_rows = 0;
	current_state.total_elapsed = 0;
	/*gtk_label_set_text(GTK_LABEL(stopwatch_clock), "00:00:00.00");*/
	hildon_entry_set_text(GTK_ENTRY(stopwatch_clock), "00:00:00.00");

	/* delete rows from the table */
	gtk_tree_store_clear(GTK_TREE_STORE(stopwatch_table_model));
}

/* thread function updates clock label */
void update_clock(gpointer data) 
{
	gulong *micro = NULL;
	char new_label[25] = {0};

	current_state.elapsed = g_timer_elapsed(current_state.main_timer, micro);
	/*current_state.total_elapsed = current_state.total_elapsed + current_state.elapsed;*/
	pretty_print(current_state.elapsed + current_state.stop_value, new_label);
	/*gtk_label_set_text(GTK_LABEL(stopwatch_clock), new_label);*/
	hildon_entry_set_text(GTK_ENTRY(stopwatch_clock), new_label);
}

void add_table_row() 
{
	/* adds new row to the table with results */
}

int main(int argc, char *argv[])
{
	GtkTreeIter stopwatch_iter;
	current_state.type = STOPWATCH_SPLIT;
	current_state.state = STOPWATCH_ZERO;
	current_state.elapsed = 0;
	current_state.total_elapsed = current_state.elapsed;
	current_state.stop_value = 0;
	current_state.num_rows = 0; /* set initial number of table rows */
	/* Initialize the GTK. */
	gtk_init(&argc, &argv);

	stopwatch_program = HILDON_PROGRAM(hildon_program_get_instance());
	g_set_application_name("Stopwatch");
	stopwatch_window = HILDON_WINDOW(hildon_window_new());
	hildon_program_add_window(stopwatch_program, stopwatch_window);

	/*Create containers and add to witdow */
	container = gtk_hbox_new(FALSE, 10);
	buttonbox = gtk_vbutton_box_new();
	box = gtk_vbox_new(FALSE, 0);
	right_box = gtk_vbox_new(FALSE, 0);

	/* scrolled window for table */
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	scrolled_win = hildon_pannable_area_new();
	
	/* results table */
	stopwatch_table = gtk_tree_view_new();
	stopwatch_table_model = gtk_tree_store_new (2, 
	                                            G_TYPE_INT, 
						    G_TYPE_STRING);
	g_signal_connect(GTK_TREE_MODEL(stopwatch_table_model), 
		"row-inserted", 
		G_CALLBACK(table_row_inserted_cb), 
		NULL);
	gtk_tree_view_set_model(GTK_TREE_VIEW(stopwatch_table),
	                        GTK_TREE_MODEL(stopwatch_table_model));
	gtk_container_add(GTK_CONTAINER(scrolled_win),
	                  GTK_WIDGET(stopwatch_table));
	gtk_box_pack_end(GTK_BOX(box), 
		GTK_WIDGET(scrolled_win),
		TRUE,
		TRUE,
		0);

	/* columns and cell renderers */
	/* left column - lap number */
	stopwatch_table_column0 = gtk_tree_view_column_new();
	stopwatch_table_column0_cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(stopwatch_table_column0, 
	                                stopwatch_table_column0_cell_renderer, 
					TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(stopwatch_table), 
	                            stopwatch_table_column0);
	gtk_tree_view_column_add_attribute(stopwatch_table_column0,
		stopwatch_table_column0_cell_renderer,
		"text",
		0);
	/* right column - time */
	stopwatch_table_column1 = gtk_tree_view_column_new();
	stopwatch_table_column1_cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(stopwatch_table_column1, 
	                                stopwatch_table_column1_cell_renderer, 
					TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(stopwatch_table), 
	                            stopwatch_table_column1);
	gtk_tree_view_column_add_attribute(stopwatch_table_column1,
		stopwatch_table_column1_cell_renderer,
		"text",
		1);
	
	gtk_container_add(GTK_CONTAINER(stopwatch_window),
		GTK_WIDGET(container));
	
	gtk_container_add(GTK_CONTAINER(container),
		GTK_WIDGET(box));
	gtk_container_add(GTK_CONTAINER(container),
		GTK_WIDGET(right_box));
	gtk_box_pack_end(GTK_BOX(right_box), 
		GTK_WIDGET(buttonbox),
		TRUE,
		TRUE,
		0);
	

	/* create buttons and add to container */
	/* start_button = gtk_button_new_with_label("Start"); */
	start_button = hildon_button_new (HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_THUMB_HEIGHT,
			                                HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (start_button), "Start");
	
	stop_button = hildon_button_new (HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_THUMB_HEIGHT,
			                                HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (stop_button), "Stop");
	/* stop_button = gtk_button_new_with_label("Stop"); */
	/* reset_button = gtk_button_new_with_label("Reset"); */
	reset_button = hildon_button_new (HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_THUMB_HEIGHT,
			                                HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (reset_button), "Reset");

	gtk_container_add(GTK_CONTAINER(buttonbox),
		GTK_WIDGET(start_button));
	g_signal_connect(G_OBJECT(start_button), 
		"clicked", 
		G_CALLBACK(start_stopwatch_cb), 
		NULL);
	gtk_container_add(GTK_CONTAINER(buttonbox),
		GTK_WIDGET(stop_button));
	g_signal_connect(G_OBJECT(stop_button), 
		"clicked", 
		G_CALLBACK(stop_stopwatch_cb), 
		NULL);
	gtk_container_add(GTK_CONTAINER(buttonbox),
		GTK_WIDGET(reset_button));
	g_signal_connect(G_OBJECT(reset_button), 
		"clicked", 
		G_CALLBACK(reset_stopwatch_cb), 
		NULL);

	/* create stopwatch type combo box */
	selector = hildon_touch_selector_new_text ();
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), _STOPWATCH_SPLIT);
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), _STOPWATCH_LAP);
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 0);
		    
	stopwatch_type_combo = hildon_picker_button_new(HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_THUMB_HEIGHT,
			HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (stopwatch_type_combo), "Stopwatch type");
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (stopwatch_type_combo),
			HILDON_TOUCH_SELECTOR (selector));
	g_signal_connect(G_OBJECT(stopwatch_type_combo),
		"value-changed",
		G_CALLBACK(combo_changed_cb),
		NULL);
	
	/* add combo to container */
	gtk_box_pack_start(GTK_BOX(right_box), 
		GTK_WIDGET(stopwatch_type_combo),
		FALSE,
		FALSE,
		10);
	
	/* create clock label */
	stopwatch_clock = hildon_entry_new (HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_THUMB_HEIGHT);
	/* set font description */
	font_desc = pango_font_description_from_string("Monospace 32");
	gtk_widget_modify_font(GTK_WIDGET(stopwatch_clock), font_desc);
	/*gtk_label_set_text(GTK_LABEL(stopwatch_clock), "00:00:00.00");*/
	hildon_entry_set_text(GTK_ENTRY(stopwatch_clock), "00:00:00.00");
	/* gtk_entry_set_has_frame(GTK_ENTRY(stopwatch_clock), FALSE);*/
	gtk_entry_set_editable(GTK_ENTRY(stopwatch_clock), FALSE);
	gdk_color_parse("#000", &color);
	gtk_widget_modify_base(stopwatch_clock, GTK_STATE_NORMAL, &color); 
	gtk_widget_modify_base(stopwatch_clock, GTK_STATE_ACTIVE, &color); 
	gtk_widget_modify_base(stopwatch_clock, GTK_STATE_PRELIGHT, &color); 
	gtk_widget_modify_base(stopwatch_clock, GTK_STATE_SELECTED, &color); 
	gtk_widget_modify_base(stopwatch_clock, GTK_STATE_INSENSITIVE, &color); 
	gtk_box_pack_start(GTK_BOX(box), 
		GTK_WIDGET(stopwatch_clock),
		FALSE,
		FALSE,
		0);
	/*gtk_widget_set_size_request(stopwatch_clock, 300, 60);*/

	/* make stop and reset inactive at first */
	gtk_widget_set_sensitive(GTK_WIDGET(stop_button), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(reset_button), FALSE);
	
	/* Begin the main application */
	gtk_widget_show_all(GTK_WIDGET(stopwatch_window));

	/* Connect signal to X in the upper corner */
	g_signal_connect(G_OBJECT(stopwatch_window), "delete_event",
		G_CALLBACK(gtk_main_quit), NULL);

	gtk_main();

	/* Exit */
	return 0;
}
