#include "channelsform.h"
#include "ui_channelsform.h"
#include "channelsreader.h"
#include "constants.h"
#include "playlistreader.h"
#include "songsreader.h"

#include <QUrl>
#include <QWebFrame>
#include <QHostInfo>
#include <QDateTime>
#include <QSignalMapper>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QTimer>

// qt-mobility
#include <QMediaPlaylist>

ChannelsForm::ChannelsForm(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ChannelsForm),
    httpResponseCode(0)
{
    ui->setupUi(this);

    ui->channelInfo->setHtml(tr("Welcome to mSoma!"));

    ui->playWidget->setVisible(false);

    ui->playerWidget->setVisible(false);
    ui->stopButton->setIcon(style()->standardIcon(QStyle::SP_MediaStop));
    ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPause));
    ui->stopButton->setEnabled(false);
    ui->playButton->setEnabled(true);

    ui->songsList->setVisible(false);

    connect(ui->updateButton, SIGNAL(clicked()), this, SLOT(updateChannels()));

    connect(&http, SIGNAL(readyRead(QHttpResponseHeader)),
             this, SLOT(readData(QHttpResponseHeader)));

    connect(&http, SIGNAL(requestFinished(int,bool)),
             this, SLOT(finished(int,bool)));

    connect(ui->channelsWidget, SIGNAL(currentRowChanged(int)), this, SLOT(updateChannelInfo(int)));

    connect(ui->abortButton, SIGNAL(clicked()), this, SLOT(abortFetching()));

    // maps for playlist buttons
    signalMapper = new QSignalMapper(this);
    connect(ui->mp3FastButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->mp3SlowButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->aacpFastButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(ui->aacpSlowButton, SIGNAL(clicked()), signalMapper, SLOT(map()));
    connect(signalMapper, SIGNAL(mapped(const QString &)), this, SLOT(playPlaylist(const QString &)));

    playlistNetworkReader = new QNetworkAccessManager(this);
    connect(playlistNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingPlaylist(QNetworkReply*)));

    songsNetworkReader = new QNetworkAccessManager(this);
    connect(songsNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingSongs(QNetworkReply*)));

    songsByteNetworkReader = new QNetworkAccessManager(this);
    connect(songsByteNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingByteSongs(QNetworkReply*)));


    connect(ui->stopButton, SIGNAL(clicked()), this, SLOT(stopPlaying()));
    connect(ui->playButton, SIGNAL(clicked()), this, SLOT(pausePlaying()));

    // Set autorotation on
    #if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    #endif

    player = new QMediaPlayer(this);
    mediaplaylist = new QMediaPlaylist;

// hack to make things work with maemo qt-mobility version
#if defined(Q_WS_MAEMO_5)
    mediaplaylist->setMediaObject(player);
#else
    player->setPlaylist(mediaplaylist);
#endif

    connect(player, SIGNAL(stateChanged(QMediaPlayer::State)),
            this, SLOT(setState(QMediaPlayer::State)));
    connect(player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)),
            this, SLOT(setMediaStatus(QMediaPlayer::MediaStatus)));

    nowPlayingUrlTimer = new QTimer(this);
    connect(nowPlayingUrlTimer, SIGNAL(timeout()), this, SLOT(checkByteSongs()));
}

ChannelsForm::~ChannelsForm()
{
    delete ui;
}

void ChannelsForm::updateChannels()
{
    // disable button while updating
    startUpdating();

    data.clear();
    ui->channelsWidget->clear();

    QUrl url(channelsFeedUrl);

    // clear the channel info that we'll use for displaying status
    ui->channelInfo->setHtml("");

    if (url.isValid()) {
        QHostInfo info = QHostInfo::fromName(url.host());
        if (info.error() == QHostInfo::NoError) {
            http.setHost(url.host());
            connectionId = http.get(url.path());

            setStatus(tr("fetching from %1 ...").arg(channelsFeedUrl));
        } else {
            setErrorStatus(tr("Host resolution error: %1").arg(info.errorString()));
            endUpdating();
        }
    } else {
        setErrorStatus(tr("Invalid URL: %1").arg(url.toString()));
        endUpdating();
    }
}

/*
    Reads data received from the RDF source.

    We read all the available data, and pass it to the XML
    stream reader. Then we call the XML parsing function.

    If parsing fails for any reason, we abort the fetch.
*/

void ChannelsForm::readData(const QHttpResponseHeader &resp)
{
    httpResponseCode = resp.statusCode();
    httpResponsePhrase = resp.reasonPhrase();
    if (httpResponseCode != 200) {
        qDebug() << "error: " << httpResponsePhrase;
        http.abort();
    } else {
        data.append(http.readAll());

        setStatus(tr("reading from %1 ...").arg(channelsFeedUrl));
    }
}

/*
    Finishes processing an HTTP request.

    The default behavior is to keep the text edit read only.

    If an error has occurred, the user interface is made available
    to the user for further input, allowing a new fetch to be
    started.

    If the HTTP get request has finished, we make the
    user interface available to the user for further input.
*/

void ChannelsForm::finished(int id, bool error)
{
    if (error) {
        setErrorStatus("HTTP error: " + httpResponsePhrase, httpResponseCode);
        qWarning("Received error during HTTP fetch.");
        endUpdating();
    } else if (id == connectionId) {
        fillChannelList();
        setStatus(tr("finished fetching from %1").arg(channelsFeedUrl));
        endUpdating();
    }
}

void ChannelsForm::fillChannelList()
{
    if (data.size() == 0)
        return;

    ui->channelsWidget->clear();

    ChannelsReader channelsReader;
    channelsReader.addData(data);
    // replace channels
    channels = channelsReader.readChannels();
    for (MSomaChannels::ConstIterator it = channels.begin(); it != channels.end(); ++it)
    {
        ui->channelsWidget->addItem(it->values["title"]);
    }
}

void ChannelsForm::abortFetching()
{
    http.abort();
    setStatus(tr("Aborted"));
}

void ChannelsForm::updateStatus(const QString &s)
{
    ui->channelInfo->setHtml
            (ui->channelInfo->page()->mainFrame()->toHtml() +
             "<br>" + s);
}

void ChannelsForm::setStatus(const QString &s)
{
    updateStatus(s);
}

void ChannelsForm::setErrorStatus(const QString &s)
{
    updateStatus("<b>" + s + "</b>");
}

void ChannelsForm::setErrorStatus(const QString &s, int error)
{
    setErrorStatus(s + " (" + QString::number(error) + ")");
}

void ChannelsForm::startUpdating()
{
    ui->updateButton->setEnabled(false);
    ui->abortButton->setEnabled(true);
}

void ChannelsForm::endUpdating()
{
    ui->updateButton->setEnabled(true);
    ui->abortButton->setEnabled(false);
}

bool channelInfoAlreadyShown(const QString &value)
{
    return (value == "image" || value == "title");
}

void ChannelsForm::updateChannelInfo(int position)
{
    if (position < 0)
        return;

    if (position < channels.size()) {
        //ui->channelInfo->clear();
        QString info;

        channel = channels[position];

        info += "<img src=\""
                +
                channel.values["image"]
                + "\"><br>" +
                "<b>title</b>: " +
                channel.values["title"] + "<br>\n";
                ;

        for (MSomaChannel::ValueIterator value = channel.values.begin(); value != channel.values.end(); ++value)
        {
            if (value.key() == "updated") {
                info += "<b>" + value.key() + "</b>: " + QDateTime::fromTime_t((value.value()).toUInt()).toString() + "<br>\n";
            } else if (!channelInfoAlreadyShown(value.key())) {
                info += "<b>" + value.key() + "</b>: " + value.value() + "<br>\n";
            }
        }

        ui->channelInfo->setHtml(info);

        enablePlayButtons(channel);
    }
}

void ChannelsForm::playPlaylist(const QString &url)
{
    ui->playWidget->setVisible(false);
    enableStopButton();
    qDebug() << "playing " << url;

    startSongsList();

    // starts the async read of the contents of the playlist
    playlistNetworkReader->get(QNetworkRequest(url));
}

void ChannelsForm::finishReadingPlaylist(QNetworkReply *reply)
{
    QByteArray playlistData = reply->readAll();
    //qDebug() << "playlist info: " << playlistData;
    PlaylistReader playlistReader(playlistData);
    // now get the stream urls of the playlist
    PlaylistReader::StreamUrls streamUrls = playlistReader.getStreamUrls();
    qDebug() << "stream urls: " << streamUrls;

    QString streamUrl = streamUrls[0];
    // we only have one element in the playlist
    mediaplaylist->removeMedia(0);
    mediaplaylist->addMedia(QUrl(streamUrl));
    player->play();

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelsForm::readSongs()
{
    ui->playerStatusLabel->setText(tr("Updating song list"));
    QUrl songsUrl(SONGS_FEED_URL + channel.id + ".xml");
    // starts the async read of the songs of the playlist
    songsNetworkReader->get(QNetworkRequest(songsUrl));
}

void ChannelsForm::finishReadingSongs(QNetworkReply *reply)
{
    QByteArray songsData = reply->readAll();
    SongsReader songsReader(songsData);
    // now get the songs information
    MSomaSongs songs = songsReader.readSongs();

    ui->songsList->clear();
    ui->songsList->addItems(songs.toStringList());

    ui->playerStatusLabel->setText(tr("Song list updated"));

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelsForm::checkByteSongs()
{
    QUrl songsUrl(SONGS_POLL_URL + channel.id + ".test.html");
    // starts the async read of the byte song of the playlist
    // to see whether we need to update the songs list
    songsByteNetworkReader->get(QNetworkRequest(songsUrl));
}

void ChannelsForm::finishReadingByteSongs(QNetworkReply *reply)
{
    QString byteSongs = reply->readAll();

    if (byteSongs != nowPlayingUrlByte)
    {
        nowPlayingUrlByte = byteSongs;
        readSongs();
    }

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelsForm::startSongsList()
{
    ui->songsList->setVisible(true);
    readSongs();
}

void ChannelsForm::stopSongsList()
{
    ui->songsList->setVisible(false);
}

void ChannelsForm::stopPlaying()
{
    player->stop();
    disableStopButton();
    ui->playWidget->setVisible(true);
    stopSongsList();
}

void ChannelsForm::pausePlaying()
{
    if (playerState == QMediaPlayer::PlayingState) {
        player->pause();
    } else {
        player->play();
    }
}

void ChannelsForm::enableStopButton()
{
    ui->playerWidget->setVisible(true);
    ui->stopButton->setEnabled(true);
}

void ChannelsForm::disableStopButton()
{
    ui->playerWidget->setVisible(false);
}

void ChannelsForm::enablePlayButtons(const MSomaChannel &channel)
{
    disablePlayButtons();
    ui->playWidget->setVisible(true);

    for (MSomaChannel::PlayListIterator playlist = channel.playlists.begin(); playlist != channel.playlists.end(); ++playlist)
    {
        if (playlist->name == "fastpls")
        {
            if (playlist->format == "mp3") {
                ui->mp3FastButton->setVisible(true);
                signalMapper->setMapping(ui->mp3FastButton, playlist->url);
            } else if (playlist->format == "aacp") {
                ui->aacpFastButton->setVisible(true);
                signalMapper->setMapping(ui->aacpFastButton, playlist->url);
            }
        }
        else if (playlist->name == "slowpls")
        {
            if (playlist->format == "mp3") {
                ui->mp3SlowButton->setVisible(true);
                signalMapper->setMapping(ui->mp3SlowButton, playlist->url);
            } else if (playlist->format == "aacp") {
                ui->aacpSlowButton->setVisible(true);
                signalMapper->setMapping(ui->aacpSlowButton, playlist->url);
            }
        }
    }
}

void ChannelsForm::disablePlayButtons()
{
    ui->playWidget->setVisible(false);
    ui->mp3FastButton->setVisible(false);
    ui->mp3SlowButton->setVisible(false);
    ui->aacpFastButton->setVisible(false);
    ui->aacpSlowButton->setVisible(false);
}

void ChannelsForm::setState(QMediaPlayer::State state)
{
    if (state != playerState) {
        playerState = state;

        switch (state) {
        case QMediaPlayer::StoppedState:
            ui->stopButton->setEnabled(false);
            ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
            nowPlayingUrlTimer->stop();
            break;
        case QMediaPlayer::PlayingState:
            ui->stopButton->setEnabled(true);
            ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPause));
            nowPlayingUrlTimer->start(SONGS_POLL_TIME);
            break;
        case QMediaPlayer::PausedState:
            ui->stopButton->setEnabled(true);
            ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
            nowPlayingUrlTimer->stop();
            break;
        }
    }
}

void ChannelsForm::setMediaStatus(QMediaPlayer::MediaStatus state)
{
    switch (state) {
    case QMediaPlayer::BufferingMedia:
        ui->playerStatusLabel->setText(tr("Buffering..."));
        break;
    case QMediaPlayer::BufferedMedia:
        ui->playerStatusLabel->setText(tr("Complete Buffering."));
        break;
    case QMediaPlayer::LoadingMedia:
        ui->playerStatusLabel->setText(tr("Loading media..."));
        break;
    case QMediaPlayer::StalledMedia:
        ui->playerStatusLabel->setText(tr("Stalled media."));
        break;
    default:
        break;
    }
}

void ChannelsForm::readSettings(QSettings &settings)
{
    channelsFeedUrl = settings.value("feedurl", CHANNELS_FEED_URL).toString();
    ui->splitter->restoreState(settings.value("splitterSizes").toByteArray());
    data = settings.value("channelFeed").toByteArray();
}

void ChannelsForm::writeSettings(QSettings &settings)
{
    settings.setValue("feedurl", channelsFeedUrl);
    settings.setValue("splitterSizes", ui->splitter->saveState());
    settings.setValue("channelFeed", data);
}
