#include "channelplayer.h"
#include "ui_channelplayer.h"
#include "ui_playercontrols.h"
#include "songsreader.h"
#include "playlistreader.h"
#include "constants.h"

#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QTimer>

// qt-mobility
#include <QMediaPlaylist>

ChannelPlayer::ChannelPlayer(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ChannelPlayer),
    showRecentlyPlayedSongs(false)
{
    ui->setupUi(this);

    ui->playerControls->ui->stopButton->setIcon(style()->standardIcon(QStyle::SP_MediaStop));
    ui->playerControls->ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPause));
    ui->playerControls->ui->stopButton->setEnabled(false);
    ui->playerControls->ui->playButton->setEnabled(true);

    ui->songsList->setVisible(false);
    ui->currentSongLabel->setVisible(false);

    playlistNetworkReader = new QNetworkAccessManager(this);
    connect(playlistNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingPlaylist(QNetworkReply*)));

    songsNetworkReader = new QNetworkAccessManager(this);
    connect(songsNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingSongs(QNetworkReply*)));

    songsByteNetworkReader = new QNetworkAccessManager(this);
    connect(songsByteNetworkReader, SIGNAL(finished(QNetworkReply*)), this, SLOT(finishReadingByteSongs(QNetworkReply*)));

    connect(ui->playerControls->ui->stopButton, SIGNAL(clicked()), this, SLOT(stopPlaying()));
    connect(ui->playerControls->ui->playButton, SIGNAL(clicked()), this, SLOT(pausePlaying()));

    player = new QMediaPlayer(this);
    mediaplaylist = new QMediaPlaylist;

// hack to make things work with maemo qt-mobility version
// it's not needed since qt-mobility 1.0.1
// #if defined(Q_WS_MAEMO_5)
//    mediaplaylist->setMediaObject(player);
//#else
    player->setPlaylist(mediaplaylist);
//#endif

    connect(player, SIGNAL(stateChanged(QMediaPlayer::State)),
            this, SLOT(setState(QMediaPlayer::State)));
    connect(player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)),
            this, SLOT(setMediaStatus(QMediaPlayer::MediaStatus)));

    nowPlayingUrlTimer = new QTimer(this);
    connect(nowPlayingUrlTimer, SIGNAL(timeout()), this, SLOT(checkByteSongs()));
}

ChannelPlayer::~ChannelPlayer()
{
    delete ui;
}

void ChannelPlayer::playPlaylist(const QString &url)
{
    setVisible(true);
    enableStopButton();
    qDebug() << "playing " << url;

    startSongsList();

    // starts the async read of the contents of the playlist
    playlistNetworkReader->get(QNetworkRequest(url));
}

void ChannelPlayer::finishReadingPlaylist(QNetworkReply *reply)
{
    QByteArray playlistData = reply->readAll();
    //qDebug() << "playlist info: " << playlistData;
    PlaylistReader playlistReader(playlistData);
    // now get the stream urls of the playlist
    PlaylistReader::StreamUrls streamUrls = playlistReader.getStreamUrls();
    qDebug() << "stream urls: " << streamUrls;

    QString streamUrl = streamUrls[0];
    // we only have one element in the playlist
    mediaplaylist->removeMedia(0);
    mediaplaylist->addMedia(QUrl(streamUrl));
    player->play();

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelPlayer::readSongs()
{
    ui->playerControls->ui->playerStatusLabel->setText(tr("Updating song list"));
    QUrl songsUrl(SONGS_FEED_URL + channel.id + ".xml");
    // starts the async read of the songs of the playlist
    songsNetworkReader->get(QNetworkRequest(songsUrl));
}

void ChannelPlayer::finishReadingSongs(QNetworkReply *reply)
{
    QByteArray songsData = reply->readAll();
    SongsReader songsReader(songsData);
    // now get the songs information
    MSomaSongs songs = songsReader.readSongs();

    ui->songsList->clear();
    ui->songsList->addItems(songs.toStringList());
    if (songs.size())
    {   // avoid pointer exceptions if the list is empty
        ui->currentSongLabel->setText(songs[0].toString());
        ui->currentSongLabel->setCursorPosition(0);
    }

    if (showRecentlyPlayedSongs)
        ui->playerControls->ui->playerStatusLabel->setText(tr("Song list updated"));
    else
        ui->playerControls->ui->playerStatusLabel->setText(tr("Current song updated"));

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelPlayer::checkByteSongs()
{
    QUrl songsUrl(SONGS_POLL_URL + channel.id + ".test.html");
    // starts the async read of the byte song of the playlist
    // to see whether we need to update the songs list
    songsByteNetworkReader->get(QNetworkRequest(songsUrl));
}

void ChannelPlayer::finishReadingByteSongs(QNetworkReply *reply)
{
    QString byteSongs = reply->readAll();

    if (byteSongs != nowPlayingUrlByte)
    {
        nowPlayingUrlByte = byteSongs;
        readSongs();
    }

    // it's our responsibility to delete it, but after the slot!
    reply->deleteLater();
}

void ChannelPlayer::startSongsList()
{
    if (showRecentlyPlayedSongs) {
        ui->songsList->setVisible(true);
        ui->currentSongLabel->setVisible(false);
    } else {
        ui->songsList->setVisible(false);
        ui->currentSongLabel->setVisible(true);
    }
    readSongs();
}

void ChannelPlayer::stopSongsList()
{
    ui->songsList->setVisible(false);
    ui->currentSongLabel->setVisible(false);
}

void ChannelPlayer::stopPlaying()
{
    player->stop();
    //disableStopButton();
    //ui->playWidget->setVisible(true);
    stopSongsList();
    //setVisible(false);
    emit stopPlayer();
}

void ChannelPlayer::pausePlaying()
{
    if (playerState == QMediaPlayer::PlayingState) {
        player->pause();
    } else {
        player->play();
    }
}

void ChannelPlayer::enableStopButton()
{
    ui->playerControls->setVisible(true);
    ui->playerControls->ui->stopButton->setEnabled(true);
}

void ChannelPlayer::disableStopButton()
{
    ui->playerControls->setVisible(false);
}

void ChannelPlayer::setState(QMediaPlayer::State state)
{
    if (state != playerState) {
        playerState = state;

        switch (state) {
        case QMediaPlayer::StoppedState:
            ui->playerControls->ui->stopButton->setEnabled(false);
            ui->playerControls->ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
            nowPlayingUrlTimer->stop();
            break;
        case QMediaPlayer::PlayingState:
            ui->playerControls->ui->stopButton->setEnabled(true);
            ui->playerControls->ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPause));
            nowPlayingUrlTimer->start(SONGS_POLL_TIME);
            break;
        case QMediaPlayer::PausedState:
            ui->playerControls->ui->stopButton->setEnabled(true);
            ui->playerControls->ui->playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
            nowPlayingUrlTimer->stop();
            break;
        }
    }
}

void ChannelPlayer::setMediaStatus(QMediaPlayer::MediaStatus state)
{
    switch (state) {
    case QMediaPlayer::BufferingMedia:
        ui->playerControls->ui->playerStatusLabel->setText(tr("Buffering..."));
        break;
    case QMediaPlayer::BufferedMedia:
        ui->playerControls->ui->playerStatusLabel->setText(tr("Complete Buffering."));
        break;
    case QMediaPlayer::LoadingMedia:
        ui->playerControls->ui->playerStatusLabel->setText(tr("Loading media..."));
        break;
    case QMediaPlayer::StalledMedia:
        ui->playerControls->ui->playerStatusLabel->setText(tr("Stalled media."));
        break;
    default:
        break;
    }
}
