/*
 * Copyright (C) 2008 Tapani Pälli <lemody@c64.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include <dlfcn.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include "loader_tab.h"
#include "player_tab.h"
#include "config_tab.h"
#include "image_tab.h"

#include "plugins/driver_iface.h"
#include "plugins/search_iface.h"
#include "plugins/debug.h"

#ifdef HILDON
#include <hildon/hildon.h>
#endif

#include <sidplay/player.h>
#include <sidplay/fformat.h>
#include <sidplay/myendian.h>

#define PROGRAM_VERSION "0.3.7"
#define PROGRAM_NAME "msid - " PROGRAM_VERSION

#define APP_WINDOW_HEIGHT 400 // ~hildonwindow workspace

static GtkWidget *app_window;
static GtkWidget *player_window;
static GtkWidget *loader_window;
static GtkWidget *image_window;

static GtkWidget *status_text=NULL;

extern GtkWidget *player_song_list;
extern GtkWidget *loader_song_list;
extern GtkWidget *player_play_button;
extern GtkWidget *player_read_button;
extern GtkWidget *player_del_button;
extern GtkWidget *subsong_selector;

extern GtkWidget *config_audio_plugin_button;
extern GtkWidget *config_search_plugin_button;
extern GtkWidget *config_audio_plugin_combo;
extern GtkWidget *config_search_plugin_combo;

extern GtkWidget *hvsc_path_selection_button;
extern GtkWidget *hvsc_path_label;

extern GtkWidget *image_button;
extern GtkWidget *anim_control_button;

// DANGER - this _has_ to point to the current plugin
msid_search_plugin *msid_active_search_plugin;

static GtkWidget *config_dialog = NULL;

static gboolean ignore_subsong_change = FALSE;

static int image_page_num=0;

static int SUBSONG = 1;

char download_path[256];

#define PLUGIN_FILE_PATH "/usr/lib/msid"
#define CONFIG_FILE_PATH "/usr/share/msid"
#define CONFIG_FILE_FULL_PATH CONFIG_FILE_PATH "/msid_config.ini"

/* these settings are used by default *only* if everything else fails */
#define DEFAULT_AUDIO_PLUGIN "libmsid_esd.so"
#define DEFAULT_SEARCH_PLUGIN "libmsid_c64org.so"
static const gchar default_config[] =
  "[msid_config]\naudio_plugin = " DEFAULT_AUDIO_PLUGIN "\n search_plugin = " DEFAULT_SEARCH_PLUGIN "\n";


typedef struct msid_machine
{
  emuEngine engine;
  msid_driver *driver;
  char *song;
  uword song_no;
} msid_machine;

/*
 * states for player thread
 */
enum
  {
    PLAY = 0,
    STOP,
    BUSY,
    READY
  };

G_LOCK_DEFINE_STATIC(msid_state);
static volatile unsigned int msid_state = READY;


typedef struct _msid_plugin_iface
{
  void *dl_handle;
  void *plugin_object;
  create_t* create_plugin;
  destroy_t* destroy_plugin;
} msid_plugin_iface;

// interface to c++ objects
msid_plugin_iface audio_iface;
msid_plugin_iface search_iface;


static void
init_msid_machine (msid_machine *m);

void
update_status_text (const char *text, const char *color);



void *player_thread(void *arg)
{
  unsigned int ready = 0;
  msid_machine *m = (msid_machine*) arg;
  ubyte* buffer;
  int duration = 0;

  init_msid_machine (m); /* init driver */

  buffer = new ubyte[m->driver->bsize()];

  sidTune tune(m->song);
  sidEmuInitializeSong(m->engine, tune, m->song_no);
  m->engine.resetSecondsThisSong();

  while (!ready) {

    G_LOCK(msid_state);
    if (msid_state == STOP) {
      ready = 1;
    }
    G_UNLOCK(msid_state);

    sidEmuFillBuffer (m->engine, tune, buffer, m->driver->bsize());
    m -> driver -> play_stream (buffer, m->driver->bsize());
  }

  delete [] buffer;

  m -> driver -> close();

  G_LOCK(msid_state);
  msid_state = READY;
  G_UNLOCK(msid_state);

  return NULL;
}





/*
 * set fullscreen state for window
 */
static gint
msid_on_key_release_event (GtkWidget *widget, GdkEventKey *event)
{
  static gboolean fullscreen=FALSE;

  switch (event->keyval)
  {
#ifdef HILDON
  case HILDON_HARDKEY_FULLSCREEN:
    if (!fullscreen)
      gtk_window_fullscreen (GTK_WINDOW (widget));
    else
      gtk_window_unfullscreen (GTK_WINDOW (widget));

    fullscreen = !fullscreen;

    return TRUE;
    break;
#endif
  default :
    break;
  }
  return FALSE;
}

static gboolean
set_ready_text_timeout (gpointer data)
{
  gdk_threads_enter();
  update_status_text ("READY.", C64_FG);
  gdk_threads_leave();

  return FALSE;
}

void
update_status_text (const char *text, const char *c)
{
  gchar buffer[256];
  const char *color;

  if (!c)
    color = C64_FG;
  else
    color = c;

  if (!strstr(text, "READY")) {
    g_timeout_add (2500, set_ready_text_timeout, NULL);
  }

  snprintf (buffer, 256, "<span foreground=\"%s\"><b>%s</b></span>", color, text);
  gtk_label_set_markup (GTK_LABEL(status_text), buffer);
}

/*
 * santa's little helper
 */
static void
append_to_vbox (GtkWidget *box, GtkWidget *w,
		gboolean a, gboolean b)
{
  GtkWidget *sep = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(box), sep, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), w, a, b, 0);
}

/*
 * generic init functions to msid c++ plugin object
 * based on http://www.faqs.org/docs/Linux-mini/C++-dlopen.html
 */
static unsigned int
init_msid_plugin (msid_plugin_iface *iface,
		  const char *path)
{
  char *error;
  void *handle;

  handle = dlopen (path, RTLD_LAZY);

  if (!handle) {
    fputs (dlerror(), stderr);
    return 0;
  }

  dlerror();

  iface -> create_plugin  =
    (create_t*) dlsym (handle, "create");

  if ((error = dlerror()) != NULL) {
    fprintf (stderr, "%s\n", error);
    return 0;
  }

  dlerror();

  iface -> destroy_plugin =
    (destroy_t*) dlsym (handle, "destroy");

  if ((error = dlerror()) != NULL) {
    fprintf (stderr, "%s\n", error);
    return 0;
  }

  iface -> plugin_object = iface -> create_plugin();
  iface -> dl_handle = handle;

  return 1;
}

static void
close_msid_plugin (msid_plugin_iface *iface)
{
  iface -> destroy_plugin (iface -> plugin_object);
  dlclose (iface -> dl_handle);
}

/*
 * strange looking configuration function
 */

static void
init_msid_machine (msid_machine *m)
{
  struct emuConfig cfg;

  m -> driver = (msid_driver *) audio_iface.plugin_object;

  m -> engine.getConfig (cfg);
  m -> driver -> set_config (&cfg);
  m -> engine.setConfig (cfg);

  m -> driver->initialize (NULL, 0, 0);
}

static void
load_screenshot (GtkWidget *widget,
		 gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  GtkTreePath *path;
  GtkTreeViewColumn *column;
  gchar *filepath=NULL;
  GError *error = NULL;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(player_song_list));
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    path = gtk_tree_model_get_path(model, &iter);
    if(!path) {
      return;
    }
    column = gtk_tree_view_get_column(GTK_TREE_VIEW(player_song_list), 0);
    gtk_tree_selection_get_selected(selection, &model, &iter);
    gtk_tree_model_get(model, &iter, PLAYER_LIST_SONG_PATH, &filepath, -1);
    gtk_tree_path_free(path);

    if (!filepath) {
      return;
    }

    gtk_widget_set_sensitive (widget, FALSE);

    set_game_screenshot (filepath);
  }
}

static void
select_hvsc_path (GtkWidget *widget,
		  gpointer data)
{
  gchar *dir = ask_for_directory (config_dialog);
  char *markup;

  if (dir)
  {
    g_print ("path: %s\n", dir);

    markup = g_markup_printf_escaped ("<span style=\"italic\">%s</span>", dir);
    gtk_label_set_markup (GTK_LABEL(hvsc_path_label), markup);
    g_free (markup);
    g_free (dir);
  }
}

static void
load_audio_plugin (GtkWidget *widget,
		   gpointer data)
{
  char buffer[256];
  gchar *plugin=NULL;
  msid_driver *driver;

  if ((gtk_combo_box_get_active (GTK_COMBO_BOX(data)) < 1))
    return;

  G_LOCK(msid_state);
  if (msid_state != READY)
  {
    G_UNLOCK(msid_state);
    update_status_text ("cannot update plugin while playing", "#ffaaaa");
    return;
  }
  G_UNLOCK(msid_state);

  plugin = gtk_combo_box_get_active_text (GTK_COMBO_BOX(data));

  if (plugin)
  {
    driver = (msid_driver*) audio_iface.plugin_object;
    snprintf (buffer, 256, "%s/audio/%s", PLUGIN_FILE_PATH, plugin);

    if (g_file_test (buffer, G_FILE_TEST_EXISTS))
    {
      close_msid_plugin (&audio_iface);
      init_msid_plugin (&audio_iface, buffer);

      snprintf (buffer, 256, "plugin [<b>%s</b>] loaded", plugin);
      update_status_text (buffer, "#aaffaa");
    }
    else
    {
      snprintf (buffer, 256, "error loading [<b>%s</b>]", plugin);
      update_status_text (buffer, "#ffaaaa");
    }
    g_free (plugin);
  }
}


static void
load_search_plugin (GtkWidget *widget,
		    gpointer data)
{
  char buffer[256];
  gchar *plugin=NULL;
  msid_search_plugin *driver;

  // FIXME - cannot update plugin while searching!

  if ((gtk_combo_box_get_active (GTK_COMBO_BOX(data)) < 1))
    return;

  plugin = gtk_combo_box_get_active_text (GTK_COMBO_BOX(data));

  if (plugin)
  {
    snprintf (buffer, 256, "%s/search/%s", PLUGIN_FILE_PATH, plugin);

    driver = (msid_search_plugin *) search_iface.plugin_object;

    /*
    search_iface.destroy_plugin (driver);
    dlclose (search_iface.dl_handle);
    */

    if (g_file_test (buffer, G_FILE_TEST_EXISTS))
    {
      close_msid_plugin (&search_iface);
      init_msid_plugin (&search_iface, buffer);

      snprintf (buffer, 256, "plugin [<b>%s</b>] loaded", plugin);
      update_status_text (buffer, "#aaffaa");
    }
    else
    {
      snprintf (buffer, 256, "error loading [<b>%s</b>]", plugin);
      update_status_text (buffer, "#ffaaaa");
    }
    g_free (plugin);

    msid_active_search_plugin = (msid_search_plugin *) search_iface.plugin_object;
  }
}


static void
play_stop_song (GtkWidget *widget,
                gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  GtkTreePath *path;
  GtkTreeViewColumn *column;
  gchar *filepath=NULL;
  GError *error = NULL;

  msid_machine *m;

  m = (msid_machine*) data;

  // switch play|stop button label
  G_LOCK(msid_state);
  switch (msid_state)
  {
  case PLAY :
    msid_state = STOP;
    G_UNLOCK(msid_state);
    update_status_text ("READY.", "#aaaaff");
    gtk_button_set_label(GTK_BUTTON(player_play_button), "PLAY");
    G_UNLOCK(msid_state);
    return;
    break;
  case READY :
    msid_state = PLAY;
    gtk_button_set_label(GTK_BUTTON(player_play_button), "STOP");
    break;
  default :
    G_UNLOCK(msid_state);
    return;
    break;
  }
  G_UNLOCK(msid_state);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(player_song_list));
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
  {
    path = gtk_tree_model_get_path(model, &iter);
    if(!path)
    {
      return;
    }

    column = gtk_tree_view_get_column(GTK_TREE_VIEW(player_song_list), 0);
    gtk_tree_selection_get_selected(selection, &model, &iter);
    gtk_tree_model_get(model, &iter, PLAYER_LIST_SONG_PATH, &filepath, -1);
    gtk_tree_path_free(path);

    if(!filepath)
    {
      return;
    }
    else
    {
      m -> song = filepath;
      m -> song_no = SUBSONG;

      g_thread_create(player_thread, (void*) data, FALSE, &error);
      return;
    }
  }
}

/* FIXME */
static unsigned int
stop_sid_machine()
{
  unsigned int ready = 0, timer = 10, result = 1;
  G_LOCK (msid_state);
  msid_state = STOP;
  G_UNLOCK(msid_state);

  while (!ready)
  {
    usleep (500);

    G_LOCK(msid_state);
    if (msid_state == READY)
    {
      ready=1;
    }
    G_UNLOCK(msid_state);

    /* just in case something goes wrong */
    timer--;
    if (!timer)
    {
      result = 0;
      break;
    }
  }

  return result;
}


static void
quit_msid (GtkMenuItem *menuitem, gpointer data)
{
  stop_sid_machine();

  gtk_main_quit();
}


static void
play_selection (GtkTreeView       *tree_view,
                GtkTreePath       *path,
                GtkTreeViewColumn *column,
                gpointer           user_data)
{
  unsigned int ready = 0, timer = 10;

  G_LOCK (msid_state);
  switch (msid_state)
  {
  case PLAY :
    /* stop machine first and then play */
    msid_state = STOP;
    G_UNLOCK(msid_state);

    stop_sid_machine (); // blocking func

    play_stop_song (NULL, user_data);
    break;

  default :
    G_UNLOCK(msid_state);
    play_stop_song (NULL, user_data);
    break;
  }
}


static unsigned int
get_subsong_amount()
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  unsigned int amount;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(player_song_list));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(player_song_list));

  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, PLAYER_LIST_SONG_AMOUNT, &amount, -1);
    return amount;
  }
  return 0;
}

static void
subsong_changed (GtkComboBox *widget, gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  unsigned int amount;

  amount = get_subsong_amount();

  SUBSONG = gtk_combo_box_get_active (widget);
  if (SUBSONG < 0 || SUBSONG > amount)
    SUBSONG=1;

  if (ignore_subsong_change) {
    ignore_subsong_change = FALSE;
    return;
  }

  if (msid_state != READY) {
    stop_sid_machine ();
    play_stop_song (NULL, data);
  }
}


static void
modify_subsong_selector()
{
  GtkTreeModel *model;
  unsigned int new_amount;
  gint prev_amount;
  char str[8];

  model = gtk_combo_box_get_model (GTK_COMBO_BOX(subsong_selector));
  new_amount = get_subsong_amount();

  if (new_amount <= 0) {
    printf ("[%s] - no subsongs found\n", __func__);
    return;
  }
  prev_amount = gtk_tree_model_iter_n_children (model, NULL);

  if (prev_amount < new_amount) {
    // add new entries if needed
    while (prev_amount < new_amount + 1) {
      sprintf(str, "%d", prev_amount);
      gtk_combo_box_append_text(GTK_COMBO_BOX(subsong_selector), str);
      prev_amount++;
    }

    // set same subsong as was selected before!
    gtk_combo_box_set_active (GTK_COMBO_BOX(subsong_selector), SUBSONG);
  }

  else if (prev_amount > new_amount) {
    // remove entries if needed
    while (prev_amount >= new_amount) {
      gtk_combo_box_remove_text(GTK_COMBO_BOX(subsong_selector),
				prev_amount);
      prev_amount--;
    }

    // TODO - set same subsong if possible!

    // ignore this change as it's program making it
    ignore_subsong_change = TRUE;

    if (SUBSONG < new_amount)
      gtk_combo_box_set_active (GTK_COMBO_BOX(subsong_selector), SUBSONG);
    else
      gtk_combo_box_set_active (GTK_COMBO_BOX(subsong_selector), 0);
  }

}

static void
delete_current_selection (void)
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreePath      *path;
  GtkTreeIter       iter, next;
  gchar *filepath=NULL;
  gchar *name=NULL;
  char buffer[256];

  GtkWidget *dialog;

  model     = gtk_tree_view_get_model (GTK_TREE_VIEW(player_song_list));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(player_song_list));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    GtkWidget *label;
    gtk_tree_model_get (model, &iter, PLAYER_LIST_SONG_PATH, &filepath, -1);
    gtk_tree_model_get (model, &iter, PLAYER_LIST_SONG_NAME, &name, -1);

    snprintf (buffer, 256, "Delete '%s'?", name);

    label = gtk_label_new (buffer);

#ifdef HILDON
    dialog = hildon_note_new_confirmation (GTK_WINDOW(app_window),
					   buffer);
#else
    dialog = gtk_dialog_new_with_buttons ("Delete file",
					  GTK_WINDOW(app_window),
					  (GtkDialogFlags)(GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
					  GTK_STOCK_OK,
					  GTK_RESPONSE_ACCEPT,
					  GTK_STOCK_CANCEL,
					  GTK_RESPONSE_REJECT,
					  NULL);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);
#endif
    gtk_widget_show_all (dialog);

    gint result = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    switch (result)
    {
    case GTK_RESPONSE_ACCEPT:
    case GTK_RESPONSE_OK:
      path = gtk_tree_model_get_path (model, &iter);

      // if there is previous slot
      if (gtk_tree_path_prev (path))
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (player_song_list),
				  path, NULL, FALSE);
      else {
	gtk_tree_path_next (path);
	if (gtk_tree_model_get_iter (model, &next, path))
	  gtk_tree_view_set_cursor (GTK_TREE_VIEW (player_song_list),
				    path, NULL, FALSE);
      }

      gtk_tree_path_free (path);

      gtk_list_store_remove (GTK_LIST_STORE(model), &iter);
      unlink (filepath);

#ifdef HILDON
      hildon_banner_show_information  (app_window, NULL, "File deleted");
#else
      update_status_text ("FILE WAS DELETED", C64_FG);
#endif
    default :
      break;
    }
  }
}

static void
delete_song (GtkWidget *widget,
	     gpointer data)
{
  delete_current_selection();
}

static gboolean
key_on_selection (GtkWidget   *widget,
		  GdkEventKey *event,
		  gpointer     user_data)
{
  switch (event->keyval)
  {
  case GDK_Delete:
  case GDK_BackSpace:
    delete_current_selection();
    break;

  default:
    break;
  }
  return FALSE;
}


static void
select_song (GtkTreeView *treeview,
             gpointer data)
{
  modify_subsong_selector ();

  /* since song changed - we can release image_button for the user again */
  gtk_widget_set_sensitive (image_button, TRUE);

  /* user cannot control animation anymore - song has changed! */
  gtk_widget_set_sensitive (anim_control_button, FALSE);

  // darken_current_animation_image();
}


static void
page_changed (GtkNotebook     *notebook,
	      GtkNotebookPage *new_page,
	      guint            page_num,
	      gpointer         user_data)
{
  if (page_num != image_page_num) {

    stop_screenshot_animation();
    // if (page_num == player_page_num) update_status_text ("playing ...");
  } else {
    // autoplay?
    //
    ; // play_screenshot_animation();    
  }
}

void
program_is_destroyed (gpointer data, GObject *program)
{
  gtk_main_quit();
}



static void
show_from_panel (GtkStatusIcon *status_icon,
		 gpointer       user_data)
{
  /* no - do not let user remove icon if it's still blinking,
     see timeout function below for why not */
  if (gtk_status_icon_get_blinking (status_icon))
    return;
  gtk_widget_show (app_window);
  g_object_unref (status_icon);
}

static gboolean
stop_blinking_timeout (gpointer icon)
{
  gdk_threads_enter();
  /* icon _has_ to still point to valid memory location ... */
  gtk_status_icon_set_blinking (GTK_STATUS_ICON(icon), FALSE);
  gdk_threads_leave();
  return FALSE;
}

static void
hide_to_panel (GtkMenuItem *menuitem, gpointer data)
{
  GtkStatusIcon *icon;

  icon = gtk_status_icon_new_from_file ("/usr/share/pixmaps/msid.png");

  g_signal_connect (G_OBJECT(icon), "activate",
		    G_CALLBACK(show_from_panel), NULL);

  gtk_status_icon_set_tooltip  (icon, PROGRAM_NAME);
  gtk_status_icon_set_visible  (icon, TRUE);
  gtk_status_icon_set_blinking (icon, TRUE);

  g_timeout_add (1500, stop_blinking_timeout, icon);

  gtk_widget_hide (app_window);
}

static inline void
free_args (gpointer data,
	   gpointer user_data)
{
  g_free (data);
}

/*
 * warning ... gnomevfs leaks memory (!)
 */
static void
launch_browser (GtkLinkButton *button,
		const gchar *link_,
		gpointer user_data)
{
  GList *list =NULL;

  GnomeVFSMimeApplication* app =
    gnome_vfs_mime_get_default_application_for_uri
    (link_, "text/html");

  list = g_list_append (list, g_strdup (link_));
  gnome_vfs_mime_application_launch (app, list);

  g_list_foreach (list, free_args, NULL);
  g_list_free (list);

  free(app);
}


GtkWidget *
append_sponsor_links (GtkWidget *dialog)
{
  GtkWidget *c64org = gtk_link_button_new_with_label ("http://www.c64.org", "c64.org");
  GtkWidget *c64com = gtk_link_button_new_with_label ("http://www.c64.com", "c64.com");

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), GTK_WIDGET(c64org));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), GTK_WIDGET(c64com));

  return dialog;
}

void
show_info_dialog (char *title, char *txt1, char *txt2)
{
  GtkWidget *d;

  d = gtk_message_dialog_new_with_markup (GTK_WINDOW(app_window),
                                          GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_MESSAGE_INFO,
                                          GTK_BUTTONS_CLOSE,
                                          txt1);
  gtk_window_set_title(GTK_WINDOW(d), title);

  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(d),
					   txt2);

#ifndef HILDON
  gtk_window_set_position (GTK_WINDOW(d), GTK_WIN_POS_CENTER_ON_PARENT);
#endif

  gtk_dialog_run (GTK_DIALOG(d));
  gtk_widget_destroy(d);
}

void
show_about_dialog (GtkMenuItem *menuitem, gpointer data)
{
  GtkWidget *info;
  GtkWidget *hbox;
  GtkWidget *logo;

  char txt[] = "MSID uses libsidplay1, a SID chip emulator by Michael Schwendt. MSID UI was programmed by lemody@c64.org\n\nMany thanks and respect to the admins of c64.org and c64.com for letting MSID use their resources.";

  GtkWidget *dialog = gtk_dialog_new_with_buttons ("About",
						   GTK_WINDOW(app_window),
						   (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
						   NULL);
  g_signal_connect_swapped (dialog, "response",
			    G_CALLBACK (gtk_widget_destroy),
			    dialog);

  hbox = gtk_hbox_new (FALSE, 8);

  logo = gtk_image_new_from_file ("/usr/share/msid/msidlogo.png");

  info = gtk_label_new ("");
  gtk_label_set_markup (GTK_LABEL(info), txt);
  gtk_label_set_line_wrap (GTK_LABEL(info), TRUE);

  gtk_box_pack_start (GTK_BOX(hbox), logo, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX(hbox), info, TRUE, TRUE, 0);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), hbox);
  
  append_sponsor_links (dialog);

  gtk_widget_show_all (dialog);
  gtk_dialog_run (GTK_DIALOG(dialog));
}

void
show_help_dialog (GtkMenuItem *menuitem, gpointer data)
{
  GtkWidget *info;
  GtkWidget *hbox;

  char txt[] = "Use <span foreground=\"#aaaaff\">Search</span> tab to download sid files, when doubletapped they will appear on <span foreground=\"#aaaaff\">Player</span> tab. When a song is selected in player tab, use <span foreground=\"#aaaaff\">Screenshot</span> tab to display screenshots related to the song, works with most game sid tunes.\n\nEnjoy!";

  GtkWidget *dialog = gtk_dialog_new_with_buttons ("Help!",
						   GTK_WINDOW(app_window),
						   (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
						   NULL);
  g_signal_connect_swapped (dialog, "response",
			    G_CALLBACK (gtk_widget_destroy),
			    dialog);

  hbox = gtk_hbox_new (FALSE, 8);

  info = gtk_label_new ("");
  gtk_label_set_markup (GTK_LABEL(info), txt);
  gtk_label_set_line_wrap (GTK_LABEL(info), TRUE);
  gtk_box_pack_start (GTK_BOX(hbox), info, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), hbox);
 

  gtk_widget_show_all (dialog);
  gtk_dialog_run (GTK_DIALOG(dialog));
}

void
show_config_dialog (GtkMenuItem *menuitem, gpointer data)
{
  GtkWidget *dialog = gtk_dialog_new_with_buttons ("Preferences",
						   GTK_WINDOW(app_window),
						   (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
						   GTK_STOCK_CLOSE,
						   GTK_RESPONSE_ACCEPT,
						   NULL);
#ifndef HILDON
  gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_CENTER_ON_PARENT);
#endif

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), GTK_WIDGET(data));
  gtk_widget_show_all (dialog);

  config_dialog = dialog;

  gtk_dialog_run (GTK_DIALOG(dialog));

  config_dialog = NULL;

  /*
   * unpack config page
   */

  gtk_container_remove (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), GTK_WIDGET(data));
  gtk_widget_destroy (dialog);
}

/*
 * very experimental code
 *
 */
static void
colorize_widget (char *widget)
{
  char rc_buffer[512];

  char BG_COLOR[]   = "\"#111111\"";
  char FG_COLOR[]   = "\"#999999\"";
  char SC_COLOR[]   = "\"#ffffff\"";
  char SCBG_COLOR[] = "\"#444444\"";
  char EVEN_COLOR[] = "\"#111111\"";
  char ODD_COLOR[]  = "\"#222222\"";

  /*
style "window"
  {
    engine "sapwood" {
      image {
       function = BOX
       file        = "../common/images/bg.xmp"
	 }
    }
  }
class "GtkWindow" style "window"
  */

  if (!strcmp (widget, "GtkTreeView"))
    {
#ifdef HILDON
      snprintf (rc_buffer, 512,
		"style \"%s_msid_colors\"\n{engine \"sapwood\"{\n\nbase[INSENSITIVE] = %s\nbase[NORMAL] = %s\nbase[ACTIVE] = %s\nbase[PRELIGHT] = %s\nbase[SELECTED] = %s\nGtkTreeView::even-row-color = %s\nGtkTreeView::odd-row-color  = %s\ntext[NORMAL]   = %s\ntext[ACTIVE]   = %s\ntext[SELECTED] = %s\ntext[INSENSITIVE] = %s\n}\n} class \"%s\" style \"%s_msid_colors\" ",
		widget,
		BG_COLOR, BG_COLOR, SCBG_COLOR, BG_COLOR, SCBG_COLOR,
		EVEN_COLOR, ODD_COLOR,
		FG_COLOR, SC_COLOR, SC_COLOR, BG_COLOR,
		widget, widget);
#else
      snprintf (rc_buffer, 512,
		"style \"%s_msid_colors\"\n{\n\nbase[INSENSITIVE] = %s\nbase[NORMAL] = %s\nbase[ACTIVE] = %s\nbase[PRELIGHT] = %s\nbase[SELECTED] = %s\nGtkTreeView::even-row-color = %s\nGtkTreeView::odd-row-color  = %s\ntext[NORMAL]   = %s\ntext[ACTIVE]   = %s\ntext[SELECTED] = %s\ntext[INSENSITIVE] = %s\n} class \"%s\" style \"%s_msid_colors\" ",
		widget,
		BG_COLOR, BG_COLOR, SCBG_COLOR, BG_COLOR, SCBG_COLOR,
		EVEN_COLOR, ODD_COLOR,
		FG_COLOR, SC_COLOR, SC_COLOR, BG_COLOR,
		widget, widget);
#endif
    }
  else if (!strcmp (widget, "GtkButton"))
    {
      /*
      snprintf (rc_buffer, 512,
		"style \"%s_msid_colors\"\n{\ntext[NORMAL]= %s\ntext[ACTIVE]   = %s\ntext[SELECTED] = %s\n} class \"%s\" style \"%s_msid_colors\" ",
		widget,
		SC_COLOR, SC_COLOR, SC_COLOR,
		widget, widget);
      */
    }

  gtk_rc_parse_string (rc_buffer);
}


static void
show_window (gpointer *window)
{
  printf("show window %p!\n", window);


  gtk_widget_show (GTK_WIDGET(window));
}

int main (int argc, char **argv)
{
  GtkWidget

    *mother_container,
    *main_box,

    *menu_bar,
    *menu,

    *main_menu,
    *item1,
    *item2,
    *item3,
    *item4,

    *player_main_box,
    *status_background,

    *player_label,
    *loader_label,
    *image_label,

    *player_button,
    *loader_button,
    *image_button,

    *player_page,
    *loader_page,
    *config_page,
    *image_page;

  GKeyFile *config_file=NULL;
  gchar *config_value=NULL;
  GError *error=NULL;

  char buffer[256];

#ifdef HILDON
  HildonProgram *program;
#endif

  msid_machine machine;

  msid_driver *driver;
  msid_search_plugin *splugin;

  g_thread_init(NULL);
  gdk_threads_init();
  gdk_threads_enter();

  gtk_init (&argc, &argv);

  gnome_vfs_init ();

  gtk_link_button_set_uri_hook (launch_browser, NULL, NULL);

  /* hildon theming overrides colorization ... buhuu. */
  //#ifndef HILDON
  //colorize_widget ("GtkTreeView");
  //colorize_widget ("GtkButton");
  //#endif

  config_file = g_key_file_new();

  /* TODO - try configuration file first */

  error=NULL;
  g_key_file_load_from_file (config_file,
			     CONFIG_FILE_FULL_PATH,
			     G_KEY_FILE_NONE,
			     &error);
  /* load default configuration */
  if (error)
  {
    error=NULL;
    g_key_file_load_from_data (config_file,
			       default_config,
			       sizeof(default_config),
			       G_KEY_FILE_NONE, &error);
  }

  // AUDIO PLUGIN

  config_value = g_key_file_get_string
    (config_file, "msid_config", "audio_plugin", &error);

  if (config_value)
  {
    snprintf (buffer, 256, "%s/audio/%s", PLUGIN_FILE_PATH, config_value);
    g_free (config_value);

    if (g_file_test (buffer, G_FILE_TEST_EXISTS))
    {
      init_msid_plugin (&audio_iface, buffer);
    }
    else
    {
      fprintf(stderr ,"audio plugin init failed!\n");
      //      goto away;
    }
  }
  else
  {
    fprintf (stderr, "no audio plugin available");
    goto away;
  }

  // SEARCH PLUGIN

  config_value = g_key_file_get_string
    (config_file, "msid_config", "search_plugin", &error);

  if (config_value)
  {
    snprintf (buffer, 256, "%s/search/%s", PLUGIN_FILE_PATH, config_value);
    g_free (config_value);

    if (g_file_test (buffer, G_FILE_TEST_EXISTS))
    {
      init_msid_plugin (&search_iface, buffer);
    }
    else
    {
      goto away;
    }
  }
  else
  {
    fprintf (stderr, "no search plugin available");
    goto away;
  }

  msid_active_search_plugin = (msid_search_plugin *) search_iface.plugin_object;
  msid_active_search_plugin->init();

  g_key_file_free (config_file);

#ifdef HILDON
  program = hildon_program_get_instance();
  app_window = hildon_stackable_window_new();
  hildon_program_add_window (program, HILDON_WINDOW (app_window));
#else
  app_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_position (GTK_WINDOW(app_window), GTK_WIN_POS_CENTER);
#endif

  gtk_widget_set_size_request (app_window, -1, APP_WINDOW_HEIGHT);

  //  g_set_application_name (PROGRAM_NAME);

  snprintf (download_path, 256, "\0");

#ifdef HILDON
  if (!create_sid_path ("/home/user/MyDocs/sidmusic")) {
    if (!create_sid_path ("/media/mmc2/sidmusic")) {
      fprintf (stderr, "error - problems with sid-path!\n");
    }
  }
#else
  snprintf (download_path, 256, "%s/sidmusic", getenv("HOME"));

  if (!create_sid_path (download_path)) {
    fprintf (stderr, "error - problems with sid-path!\n");
  }
#endif

  /*
   * create containers
   */

  main_box = gtk_vbox_new (FALSE, 0);

  player_main_box = gtk_vbox_new (FALSE, 0);

  /*
   * create widgets
   */
  
  player_label = gtk_label_new ("Player");
  loader_label = gtk_label_new ("Search");
  image_label  = gtk_label_new ("Screenshot");

  status_background = gtk_event_box_new();
  status_text       = gtk_label_new("");

  /*
   * pack widgets
   */

  gtk_container_add (GTK_CONTAINER(status_background), status_text);

  /*
   * pack containers
   */

  loader_page = create_loader_page (msid_active_search_plugin, update_status_text);

  /*
   * these are just container boxes, so can be packed in anything.
   * makes it possible to create different UI layouts more easily ...
   */

  player_page = create_player_page (update_status_text);
  image_page  = create_image_page (update_status_text);
  config_page = create_config_page ();

  /* leave reference to page so that it won't get destroyed
   * when unpacked from dialog 
   */
  g_object_ref (config_page);


  /* force notebook for arm device */
  //#ifdef HILDON
#define NOTEBOOK
  //#endif

#ifdef NOTEBOOK
  mother_container = gtk_notebook_new();

  gtk_notebook_append_page (GTK_NOTEBOOK(mother_container),
			    player_page,
			    player_label);
  gtk_notebook_append_page (GTK_NOTEBOOK(mother_container),
			    loader_page,
			    loader_label);
  image_page_num = gtk_notebook_append_page (GTK_NOTEBOOK(mother_container),
					     image_page,
					     image_label);

  g_signal_connect (G_OBJECT (mother_container), "switch-page",
		    G_CALLBACK (page_changed), image_page);
#else

  //#define STACK
#ifdef STACK
  player_button = create_huge_button("Player");
  loader_button = create_huge_button("Loader");
  image_button = create_huge_button("Slideshow");

#define UI_PAD 64
  mother_container = gtk_hbox_new (TRUE, UI_PAD);

  gtk_box_pack_start (GTK_BOX(mother_container), player_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(mother_container), loader_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(mother_container), image_button, FALSE, FALSE, 0);

  player_window = hildon_stackable_window_new();
  loader_window = hildon_stackable_window_new();
  image_window  = hildon_stackable_window_new();

  hildon_program_add_window (program, HILDON_WINDOW (player_window));
  hildon_program_add_window (program, HILDON_WINDOW (loader_window));
  hildon_program_add_window (program, HILDON_WINDOW (image_window));

  gtk_widget_show(player_window);

  printf("window pointer %p!\n", player_window);

  g_signal_connect (G_OBJECT(player_button), "clicked", G_CALLBACK (show_window), (gpointer)player_window);
  g_signal_connect (G_OBJECT(loader_button), "clicked", G_CALLBACK (show_window), (gpointer)loader_window);
  g_signal_connect (G_OBJECT(image_button),  "clicked", G_CALLBACK (show_window), (gpointer)image_window);

  gtk_container_add (GTK_CONTAINER(player_window), player_page);
  gtk_container_add (GTK_CONTAINER(loader_window), loader_page);
  gtk_container_add (GTK_CONTAINER(image_window), image_page);

#endif



#endif

  /* BIG layout for desktop systems - make this look *good* */
  /*
#define UI_PAD 4



  mother_container = gtk_hbox_new (FALSE, UI_PAD);
  gtk_box_pack_start (GTK_BOX(mother_container), player_page, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(mother_container), loader_page, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(mother_container), image_page, FALSE, FALSE, 0);
#endif
  */

  main_menu = gtk_menu_item_new_with_label ("Menu");

  item1 = gtk_menu_item_new_with_label ("Help");
  item2 = gtk_menu_item_new_with_label ("About");
  item3 = gtk_menu_item_new_with_label ("Preferences");
  item4 = gtk_menu_item_new_with_label ("Quit");

  g_signal_connect (G_OBJECT (item1), "activate",
                    G_CALLBACK (show_help_dialog), NULL);
  g_signal_connect (G_OBJECT (item2), "activate",
                    G_CALLBACK (show_about_dialog), NULL);
  g_signal_connect (G_OBJECT (item3), "activate",
                    G_CALLBACK (show_config_dialog), config_page);
  g_signal_connect (G_OBJECT (item4), "activate",
                    G_CALLBACK (quit_msid), NULL);

#ifdef HILDON
  // use hildon menu stuff
  menu_bar = gtk_menu_new();
  gtk_menu_append (GTK_MENU (menu_bar), item1);
  gtk_menu_append (GTK_MENU (menu_bar), item2);
  gtk_menu_append (GTK_MENU (menu_bar), item3);
  gtk_menu_append (GTK_MENU (menu_bar), item4);
  hildon_program_set_common_menu (program, GTK_MENU (menu_bar));

#else
  // use gtkmenubar
  menu_bar = gtk_menu_bar_new ();
  menu = gtk_menu_new();

  gtk_menu_shell_append (GTK_MENU_SHELL(menu), item1);
  gtk_menu_shell_append (GTK_MENU_SHELL(menu), item2);
  gtk_menu_shell_append (GTK_MENU_SHELL(menu), item3);
  gtk_menu_shell_append (GTK_MENU_SHELL(menu), item4);

  gtk_menu_item_set_submenu (GTK_MENU_ITEM(main_menu), menu);
  gtk_menu_shell_append (GTK_MENU_SHELL(menu_bar), main_menu);

  gtk_box_pack_start (GTK_BOX(main_box), menu_bar, FALSE, FALSE, 0);
#endif

  gtk_box_pack_start (GTK_BOX(main_box), mother_container, TRUE, TRUE, 0);
  append_to_vbox (main_box, status_background, FALSE, FALSE);

  gtk_container_add (GTK_CONTAINER(app_window), main_box);



  /*
   * signal-handlers for extern objects
   */

  g_signal_connect (G_OBJECT (player_song_list), "key-release-event",
                    G_CALLBACK (key_on_selection), (gpointer) &machine);

  g_signal_connect (G_OBJECT (player_song_list), "row-activated",
                    G_CALLBACK (play_selection), (gpointer) &machine);
  g_signal_connect (G_OBJECT (player_song_list), "cursor-changed",
                    G_CALLBACK (select_song), NULL);
  g_signal_connect (G_OBJECT (player_play_button), "clicked",
                    G_CALLBACK (play_stop_song), (gpointer) &machine);
  g_signal_connect (G_OBJECT (player_del_button), "clicked",
                    G_CALLBACK (delete_song), (gpointer) &machine);

  g_signal_connect (G_OBJECT (subsong_selector), "changed",
                    G_CALLBACK (subsong_changed), &machine);

  g_signal_connect (G_OBJECT (config_audio_plugin_button), "clicked",
		    G_CALLBACK (load_audio_plugin), config_audio_plugin_combo);
  g_signal_connect (G_OBJECT (config_search_plugin_button), "clicked",
		    G_CALLBACK (load_search_plugin), config_search_plugin_combo);
  g_signal_connect (G_OBJECT (hvsc_path_selection_button), "clicked",
		    G_CALLBACK (select_hvsc_path), NULL);

  g_signal_connect (G_OBJECT (image_button), "clicked",
		    G_CALLBACK (load_screenshot), NULL);


  g_signal_connect (G_OBJECT (app_window), "key-release-event",
                    G_CALLBACK (msid_on_key_release_event), NULL);

  gtk_widget_show_all (app_window);

#ifdef HILDON
  g_object_weak_ref (G_OBJECT (program),
		     (GWeakNotify) program_is_destroyed,
                     NULL);
  g_object_unref (program);
#else
  g_signal_connect (G_OBJECT (app_window),
		    "delete_event",
                    G_CALLBACK (gtk_main_quit),
		    NULL);
#endif

  /**************************************************/
  GdkColor bgcolor;
  gdk_color_parse (C64_BG, &bgcolor);

  gtk_widget_modify_bg (status_background, GTK_STATE_NORMAL, &bgcolor);
  /**************************************************/

  refresh_player_songlist (player_read_button, NULL);

  update_status_text ("READY.", C64_FG);

  driver  = (msid_driver *) audio_iface.plugin_object;
  splugin = (msid_search_plugin *) search_iface.plugin_object;

  gtk_main();
  gdk_threads_leave();


  close_msid_plugin (&audio_iface);
  close_msid_plugin (&search_iface);
  
  snprintf (buffer, 256, "%s/.msid", getenv("HOME"));

  // TODO - save current configuration

  /*
   * have to implement API to get plugin names from plugin objects
   */


  away :

    return 0;
}
