#include "photolistwidget.h"

#include <QtCore/QSignalMapper>
#include <QtCore/QMimeData>
#include <QtCore/QBuffer>
#include <QtCore/QUrl>
#include <QtCore/QFileInfo>
#include <QtGui/QMouseEvent>
#include <QtGui/QMenu>
#include <QtGui/QAction>
#include <QtGui/QIcon>
#include <QtGui/QApplication>

#include "datatypes/photo.h"
#include "widgets/albumselectdialog.h"
#include "servicemgr.h"

typedef QPair<Album, Photo> AlbumPhotoPair;
Q_DECLARE_METATYPE(AlbumPhotoPair)

PhotoListWidget::PhotoListWidget(ServiceMgr* mgr, QWidget *parent) :
    QListWidget(parent), mServiceMgr(mgr)
{
    setLayoutMode(QListWidget::SinglePass);
    setResizeMode(QListWidget::Fixed);
    setSpacing(2);
    setViewMode(QListWidget::IconMode);
    setUniformItemSizes(false);
    setBatchSize(110);

    setIconSize(QSize(100, 100));
    setTextElideMode(Qt::ElideNone);

    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setStyleSheet("QListWidget::item {\n	text-align: bottom;\n	padding: 1px;\n}");

    setDropIndicatorShown(false);
    setDragEnabled(false);
    setDragDropMode(QListWidget::DragOnly);

    setSelectionMode(QAbstractItemView::SingleSelection);

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(QPoint)), SLOT(showPhotoMenu(QPoint)));
}

Qt::DropActions PhotoListWidget::supportedDropActions()
{
    return Qt::MoveAction | Qt::CopyAction;
}

QStringList PhotoListWidget::mimeTypes()
{
    QStringList list;
    list << "application/msa-photo";
    list << "image/png";
    return list;
}

QMimeData* PhotoListWidget::mimeData(const QList<QListWidgetItem*> items) const
{
    QMimeData* data = new QMimeData;
    const QVariant& var = items.first()->data(Qt::UserRole);
    const Photo& photo = var.value<Photo>();

    // Save image data
    QByteArray encodedData;
    QDataStream dataStream(&encodedData, QIODevice::WriteOnly);
    dataStream << photo;
    data->setData("application/msa-photo", encodedData);

    // Save image
    data->setImageData(QVariant(QImage(photo.photo())));

    // Save url
    QFileInfo fileInfo = QFileInfo(photo.photo());
    QString file = fileInfo.canonicalFilePath();
    data->setUrls(QList<QUrl>() << QString("file://%1").arg(file));

    // Save filepath as text
    data->setText(file);

    return data;
}

void PhotoListWidget::mousePressEvent(QMouseEvent* event)
{
    // Ignore right button click
    if (event->button() != Qt::RightButton) {
        QListWidget::mousePressEvent(event);
    }
}

QMenu* PhotoListWidget::createPhotoMenu(const Photo& photo)
{
    QMenu* menu = new QMenu(this);
    QMenu* albumSelectMenu = menu->addMenu(tr("Copy to"));

    AccountList accounts = mServiceMgr->getAccounts();
    QSignalMapper* signalMapper = new QSignalMapper(menu);
    connect(signalMapper, SIGNAL(mapped(QObject*)), SLOT(onMenuAction(QObject*)));

    foreach (Account* account, accounts) {
        const AlbumList& albums = account->getAlbumList();
        foreach (const Album& album, albums) {
            // No need to copy photo to same album
            if (album.albumId() != photo.albumId()) {
                QAction* action = albumSelectMenu->addAction(QIcon(album.icon()), album.title(), signalMapper, SLOT(map()));
                action->setData(QVariant::fromValue(qMakePair(album, photo)));
                signalMapper->setMapping(action, action);
            }
        }
    }
    albumSelectMenu->addSeparator();

    QAction* action = albumSelectMenu->addAction(tr("Select..."), this, SLOT(onSelectAction()));
    action->setData(QVariant::fromValue(photo));

    return menu;
}

void PhotoListWidget::showPhotoMenu(const QPoint& pos)
{
    QModelIndex index = indexAt(pos);
    if (!index.isValid())
        return;

    const Photo& photo = index.data(Qt::UserRole).value<Photo>();

    QMenu* menu = createPhotoMenu(photo);
    menu->exec(mapToGlobal(pos));
    menu->deleteLater();
}

void PhotoListWidget::onMenuAction(QObject* object)
{
    QAction* action = static_cast<QAction*>(object);
    AlbumPhotoPair p = action->data().value<AlbumPhotoPair>();
    mServiceMgr->uploadPhoto(p.first, p.second.photo(), p.second.description());
}

void PhotoListWidget::onSelectAction()
{
    QAction* action = static_cast<QAction*>(sender());
    Photo photo = action->data().value<Photo>();
    AlbumSelectDialog* dialog = new AlbumSelectDialog(mServiceMgr, this);

    int result = dialog->exec();
    if (result == QDialog::Accepted)
        mServiceMgr->uploadPhoto(dialog->selectedAlbum(), photo.photo(), photo.description());

    dialog->deleteLater();
}
