#include "messagebrowser.h"
#include "ui_messagebrowser.h"
#include <QDebug>
#include <QTextBlock>
#include <QHBoxLayout>
#include <QScrollArea>

MessageBrowser::MessageBrowser(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::MessageBrowser)
{
    ui->setupUi(this);
    this->sm = new ServiceMgr;

    init();
}

MessageBrowser::MessageBrowser(ServiceMgr *sm, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::MessageBrowser)
{
    ui->setupUi(this);
    this->sm = sm;

    init();
}

MessageBrowser::MessageBrowser(Message msg, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::MessageBrowser)
{
    ui->setupUi(this);
    this->setMessage(msg);
    this->sm = new ServiceMgr;

    init();
}

void MessageBrowser::init()
{

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    QHBoxLayout *hLayout = new QHBoxLayout;
    hLayout->addWidget(serviceIcon = new QLabel(this));
    hLayout->addWidget(titleLabel = new QLabel(this));
    titleLabel->setWordWrap(true);
    titleLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    ui->gridLayout->addLayout(hLayout, 0, 0, 1, 2, Qt::AlignCenter);
    ui->gridLayout->addWidget(descLabel = new QLabel(this), 1, 0, 1, 2);
    ui->gridLayout->addWidget(textLabel = new QTextEdit(this), 2, 0, 1, 2);

    ui->messageBrowser->setEnabled(false);
    ui->messageBrowser->setMaximumHeight(0);

    textLabel->setReadOnly(true);
    textLabel->setStyleSheet("QTextEdit {background-color: white;}");
    textLabel->setMinimumHeight(200);
#else
    ui->gridLayout->addWidget(titleLabel = new QLabel(tr("Title:"), this), 0, 0);
    ui->gridLayout->addWidget(titleLineEdit = new QLineEdit(this), 0, 1);
    ui->gridLayout->addWidget(serviceIcon = new QLabel(this), 0, 2);
    ui->gridLayout->addWidget(dateLabel = new QLabel(tr("Date:"), this), 1, 0);
    ui->gridLayout->addWidget(dateLineEdit = new QLineEdit(this), 1, 1, 1, 2);
    ui->gridLayout->addWidget(fromLabel = new QLabel(tr("From:"), this), 2, 0);
    ui->gridLayout->addWidget(fromLineEdit = new QLineEdit(this), 2, 1, 1, 2);
    ui->gridLayout->addWidget(toLabel = new QLabel(tr("To:"), this), 3, 0);
    ui->gridLayout->addWidget(toLineEdit = new QLineEdit(this), 3, 1, 1, 2);

    toLabel->setMinimumWidth(40);
    fromLabel->setMinimumWidth(40);
    titleLabel->setMinimumWidth(40);
    dateLabel->setMinimumWidth(40);
#endif

    connect(this->sm, SIGNAL(updateAccounts(QString,AccountList)),
            this, SLOT(updateAccountList(QString,AccountList)));
}

MessageBrowser::~MessageBrowser()
{
    delete ui;
}

QStringList MessageBrowser::to()
{
    QString recepients = this->toLineEdit->text();
    QStringList list = recepients.split(QRegExp("\\s"));
    list.replaceInStrings(",","");

    return list;
}

QString MessageBrowser::from()
{
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    return QString();
#else
    return this->fromLineEdit->text();
#endif
}

QString MessageBrowser::title()
{
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    return this->titleLabel->text();
#else
    return this->titleLineEdit->text();
#endif
}

QString MessageBrowser::message()
{
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    return this->textLabel->toPlainText();
#else
    return this->ui->messageBrowser->toPlainText();
#endif
}

void MessageBrowser::resetMessage()
{

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    this->titleLabel->clear();
    this->textLabel->clear();
    this->descLabel->clear();
#else
    this->toLineEdit->clear();
    this->hideTo(false);
    this->fromLineEdit->clear();
    this->hideFrom(false);
    this->titleLineEdit->clear();
    this->dateLineEdit->clear();
    ui->messageBrowser->clear();
#endif
}

void MessageBrowser::setMessage(Message msg)
{
    QString recepients;
    QString senderId = msg.senderId();

    if (msg.recepientListConst().count() >= 1) {
        for (int i = 0; i < msg.recepientListConst().count(); i++) {
            if (msg.recepientListConst().at(i).id != senderId) {
                if ((msg.recepientListConst().count() - i) == 1) {
                    recepients.append(msg.recepientListConst().at(i).name);
                } else {
                    recepients.append(msg.recepientListConst().at(i).name + ", ");
                }
            } else {
                if ((msg.recepientListConst().count() - i) == 1) {
                    recepients.remove(recepients.length() - 2, 2);
                }
            }
        }
    }

    QIcon sIcon;
    for (int i = 0; i < mAccountList.count(); i++) {
        if (mAccountList.at(i)->accountId() == msg.accountId()) {
            foreach (DriverInfo* driver, sm->getDrivers()) {
                if (!driver->icon.isNull()) {
                    if (driver->serviceName == mAccountList.at(i)->serviceName())
                        sIcon = driver->icon;
                }
            }
            break;
        }
    }

    bool hideFrom = false;
    bool hideTo = false;

    if (mLocalAccountIds.contains(msg.senderId()))
        hideFrom = true;

    foreach (const Recepient& recepient, msg.recepientListConst())
        if (mLocalAccountIds.contains(recepient.id))
            hideTo = true;

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    if (msg.title().isEmpty() && msg.text().isEmpty())
        this->titleLabel->setText(QString::null);
    else
        this->titleLabel->setText("<h4>" + msg.subject() + "</h4>");

    QString desc;
    if (!msg.time().isEmpty()) {
        desc.append("<b>" + tr("Date: ") + "</b>" + msg.time() + "<br>");
    }

    if ((msg.recepientList().length() != 0) && !hideTo) {
        desc.append("<b>" + tr("To: ") + "</b>" + recepients + "<br>");
    }

    if (!msg.senderName().isEmpty() && !hideFrom)
        desc.append("<b>" + tr("From: ") + "</b>" + msg.senderName());

    if (hideFrom && hideTo) {
        desc.append("<br><b>" + tr("From: ") + "</b>" + msg.senderName());
    }

    descLabel->setText(desc);
#else
    if ((msg.recepientListConst().length() == 0) || hideTo) {
        this->hideTo(true);
    } else {
        this->hideTo(false);
        this->toLineEdit->setText(recepients);
        this->toLineEdit->setCursorPosition(0);
    }

    if (msg.senderName().isEmpty() || hideFrom) {
        this->hideFrom(true);
    } else {
        this->hideFrom(false);
        this->fromLineEdit->setText(msg.senderName());
        this->fromLineEdit->setCursorPosition(0);
    }

    if (hideFrom && hideTo) {
        this->hideFrom(false);
        this->fromLineEdit->setText(msg.senderName());
        this->fromLineEdit->setCursorPosition(0);
    }

    this->titleLineEdit->setText(msg.subject());
    this->titleLineEdit->setCursorPosition(0);

    this->dateLineEdit->setText(msg.time());
#endif

    if (!sIcon.isNull())
        serviceIcon->setPixmap(sIcon.pixmap(titleLabel->fontMetrics().height() + 2));

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    textLabel->setText(msg.text());

    QTextDocument *doc = this->textLabel->document()->clone();
    doc->setTextWidth(500);
    QSize commentSize = doc->size().toSize();

    QTextCursor cur = doc->rootFrame()->firstCursorPosition();
    cur.movePosition( QTextCursor::Start );

    bool isNotDone = true;
    int lineCounter = 0;

    while( isNotDone ) {
        isNotDone = cur.movePosition( QTextCursor::Down );
        lineCounter++;
    }

    int newHeight = (lineCounter + 1) * textLabel->fontMetrics().height();

    if (newHeight > 180)
        textLabel->setMinimumHeight(newHeight);
#else
    ui->messageBrowser->setText(msg.text());
#endif
}

void MessageBrowser::updateAccountList(QString, AccountList accountList)
{
    mAccountList = accountList;
    mLocalAccountIds.clear();

    foreach (Account* account, mAccountList) {
        const QString& id = account->getProfile(false).ownerId();
        mLocalAccountIds.insert(id);
    }
}

void MessageBrowser::hideFrom(bool hide)
{
    this->fromLabel->setVisible(!hide);
    this->fromLineEdit->setVisible(!hide);
}

void MessageBrowser::hideTo(bool hide)
{
    this->toLabel->setVisible(!hide);
    this->toLineEdit->setVisible(!hide);
}
