#ifndef FEEDMODEL_H
#define FEEDMODEL_H

#include <QAbstractListModel>
#include <QDateTime>
#include <QList>
#include <QHash>

#include "datatypes/qeventfeed.h"
#include <datatypes/friend.h>

class ServiceMgr;

class FeedModel : public QAbstractListModel
{
    Q_OBJECT

public:
    typedef QList<QEventFeed> FeedItem;
    typedef QList<FeedItem> FeedItemList;

    //! User roles that store data about feed item
    enum ItemRole {
        FeedItemRole = Qt::UserRole, /*!< The data representing one grouped feed item */
        MapRole /*!< Data rectangles */
    };

    explicit FeedModel(ServiceMgr* mgr, QObject *parent = 0);

    int rowCount(const QModelIndex &parent) const;
    Qt::ItemFlags flags(const QModelIndex &index) const;
    QVariant data(const QModelIndex &index, int role) const;
    QVariant headerData(int section, Qt::Orientation orientation, int role) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role);

    //! Get count of all news (not groups)
    int feedCount() const;

    //! Is type enabled
    bool feedTypeState(QEventFeed::FeedType feedType) const;

    /**
      Shows friend's feed

      \param fr The friend data.
      */
    void showFriend(Friend fr);

private:
    void updateFeedItemList(const Friend fr = Friend());

signals:
    //! Count of news changed
    void feedCountChanged(int count);

    //! Emited when new feed type appears in FeedList
    void newFeedType(QEventFeed::FeedType type);

public slots:
    void setFeed(QEventFeedList list, QEventFeed::FeedType type, bool isLastUpdate);

    //! Enable or disable displaying of specific feed type
    void setFeedTypeState(QEventFeed::FeedType, bool isActive);

    //! Recalculate feed count
    void updateFeedCount();

private:
    struct FeedTypeState
    {
        FeedTypeState();
        QEventFeedList list;
        bool active;
    };

    ServiceMgr* mServiceMgr;

    FeedItemList mGroupedFeed;
    int mFeedCount;
    bool mGroupItems;
    QHash<QEventFeed::FeedType, FeedTypeState> mFeedByType;
    QHash<int, QVariant> mRegions;
};

Q_DECLARE_METATYPE(FeedModel::FeedItem)

#endif // FEEDMODEL_H
