#ifndef FEEDDELEGATE_H
#define FEEDDELEGATE_H

#include <QStyledItemDelegate>
#include <QPen>
#include <QHash>
#include <QMutex>

#include "datatypes/friend.h"
#include "datatypes/account.h"
#include "widgets/feed/regionmap.h"

class QEventFeed;
class ServiceMgr;

class FeedDelegate : public QStyledItemDelegate
{
    Q_OBJECT
    static const int AVATAR_SIZE;
    static const int PHOTO_SIZE;
    static const char* newlineRegex;
    static const int BETWEEN_PHOTO_DISTANCE;

public:
    explicit FeedDelegate(ServiceMgr* serviceMgr, QObject* parent = 0);

    void paint(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const;
    QSize sizeHint(const QStyleOptionViewItem& option, const QModelIndex& index) const;

public slots:
    //! Update friends cache
    void setFriends(FriendList friendList);

    //! Update local accounts cache
    void setAccounts(QString, AccountList accountList);

private:
    int drawFeedItem(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const;
    int drawTextFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;
    QSize drawPhotoFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;
    QRect drawPhotoFeedList(QPainter* painter, QList<QEventFeed>& feedList, const QStyleOptionViewItem& option, const QPoint& topLeft) const;
    int drawLinkFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;

    void drawFrame(QPainter* painter, const QStyleOptionViewItem& option, const QRect& rect) const;

    //! Draw header of the news item

    //! \param painter QPainter object, can be 0, then function will only return size of header
    //! \param feed News item
    //! \param option
    //! \param topLeft Top-left point of header
    //! \return Rectangle containing header
    QRect drawHeader(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;

    //! Draw avatar

    //! \param painter QPainter object, can be 0, then function will only return size of avatar
    //! \param feed News item
    //! \param option
    //! \param topLeft Top-left point of avatar
    //! \return Rectangle containing avatar
    QRect drawAvatar(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;

    //! Draw one news item text

    //! \param painter QPainter object, can be 0, then function will only return size of text block
    //! \param feed News item
    //! \param option
    //! \param topLeft Top-left point of text block
    //! \return Rectangle containing text block
    QRect drawFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const;

    //! Get friend from cache

    //! \param id Friend's identifier
    //! \param fr Out parameter
    //! \retval true if found
    //! \retval false otherwise
    bool findFriend(const QString& id, Friend& fr) const;

private:
    ServiceMgr* mServiceMgr;
    QHash<QString, Friend> mFriendCache;
    QHash<QString, Friend> mLocalFriendCache;
    mutable QMutex mLock;

    //! This variables doesn't represent state of object,
    //! and can be defined with 'mutable' modifier
    //! They valid only when FeedDelegate::paint executes
    mutable QModelIndex mCurrentIndex;
    mutable QRect mCurrentRect;
    mutable RegionMap mCurrentRegions;
    mutable bool mIsStoringRegions;

private:
    // Region storing functions
    void beforeRendering(const QModelIndex&, const QStyleOptionViewItem& option) const;
    void addActionRegion(const QRect&, const QEventFeed&) const;
    void afterRendering(const QModelIndex&) const;
};

#endif // FEEDDELEGATE_H
