#include "feeddelegate.h"

#include <cassert>

#include <QPainter>
#include <QLinearGradient>
#include <QAbstractScrollArea>

#include "widgets/feed/feedmodel.h"
#include "servicemgr.h"
#include "datatypes/friend.h"
#include "config.h"
#include "utils/utils.h"

const int FeedDelegate::AVATAR_SIZE = ITEM_HEIGHT;
const int FeedDelegate::PHOTO_SIZE = 128;
const int FeedDelegate::BETWEEN_PHOTO_DISTANCE = 6;
const char* FeedDelegate::newlineRegex = "<br>";

FeedDelegate::FeedDelegate(ServiceMgr* mgr, QObject *parent) :
    QStyledItemDelegate(parent), mServiceMgr(mgr), mIsStoringRegions(false)
{
    setFriends(mServiceMgr->getFriends(false, false));
    setAccounts(QString(), mServiceMgr->getAccounts());

    connect(mServiceMgr, SIGNAL(updateFriends(FriendList,bool)), this, SLOT(setFriends(FriendList)));
    connect(mServiceMgr, SIGNAL(updateAccounts(QString,AccountList)), this, SLOT(setAccounts(QString,AccountList)));
}

int FeedDelegate::drawFeedItem(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const
{
    FeedModel::FeedItem item = index.data(FeedModel::FeedItemRole).value<FeedModel::FeedItem>();
    QEventFeed firstFeed = item.first();
    QRect rect = option.rect;

    // Draw background without selection
    if (painter)
        painter->fillRect(rect, option.palette.base());

    QRect headerRect = drawHeader(painter, firstFeed, option, rect.topLeft());
    QRect avatarRect = drawAvatar(painter, firstFeed, option, headerRect.bottomLeft());

    QList<QEventFeed> photos;
    QList<QEventFeed> otherFeeds;
    foreach (const QEventFeed& feed, item) {
        if (feed.attach().type() == QAttachment::ImageAttach) {
            photos.append(feed);
        } else {
            otherFeeds.append(feed);
        }
    }

    QPoint nextFeedPos = avatarRect.topRight();

    // Draw photo rows until photo list empty
    while (!photos.isEmpty()) {
        QRect r = drawPhotoFeedList(painter, photos, option, nextFeedPos + QPoint(4, 4));
        QRect frameRect = r.adjusted(-2, -2, 2, 2);
        frameRect.setRight(rect.right() - 2);
        drawFrame(painter, option, frameRect);
        nextFeedPos += QPoint(0, r.height() + 7);
    }

    // Draw all feeds but photo
    foreach (const QEventFeed& feed, otherFeeds) {
        QRect r = drawFeed(painter, feed, option, nextFeedPos + QPoint(4, 4));
        QRect frameRect = r.adjusted(-2, -2, 2, 2);
        frameRect.setRight(rect.right() - 2);
        drawFrame(painter, option, frameRect);
        nextFeedPos += QPoint(0, r.height() + 7);
    }

    return qMax(nextFeedPos.y() - option.rect.top(), avatarRect.height());
}

QRect FeedDelegate::drawHeader(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    QRect rect(topLeft, QSize(option.rect.width(), option.fontMetrics.height()));

    if (!painter)
        return rect;

    painter->save();

    // Set bold font
    QFont headerFont = option.font;
    headerFont.setBold(true);
    painter->setFont(headerFont);

    // Draw header
    QString authorName = feed.ownerName();
    QString date = feed.created().date().toString();
    painter->drawText(rect, Qt::AlignVCenter, QString("%1 %3 %2").arg(authorName, date, QChar(0x2014 /* m-dash */)));

    painter->restore();
    return rect;
}

QRect FeedDelegate::drawAvatar(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    Q_UNUSED(option);

    Friend fr;
    // TODO: replace this code with Utils::adjustImage
    if (findFriend(feed.ownerId(), fr)) {
        QImage avatar(fr.icon());

        if (avatar.height() > avatar.width())
            avatar = avatar.copy(0, (avatar.height() - avatar.width()) / 2, avatar.width(), avatar.width());
        else if (avatar.width() > avatar.height())
            avatar = avatar.copy((avatar.width() - avatar.height()) / 2, 0, avatar.height(), avatar.height());

        avatar = avatar.scaled(AVATAR_SIZE, AVATAR_SIZE, Qt::KeepAspectRatio, Qt::SmoothTransformation);

        if (painter)
            painter->drawImage(topLeft, avatar);
        return QRect(topLeft, QSize(AVATAR_SIZE + 3, AVATAR_SIZE + 3));
    } else {
        return QRect(topLeft, QSize(3, 3));
    }
}


void FeedDelegate::drawFrame(QPainter* painter, const QStyleOptionViewItem& option, const QRect& frameRect) const
{
    Q_UNUSED(option);

    if (painter) {
        // Draw frame
        painter->save();
        QPen framePen(Qt::DotLine);
        framePen.setColor(QColor(200, 200, 200));
        framePen.setWidth(1);
        painter->setPen(framePen);
        painter->drawRect(frameRect);
        painter->restore();
    }
}

QRect FeedDelegate::drawFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    int height = 0;
    height += option.fontMetrics.height(); // Time label's height

    if (!feed.text().isEmpty()) {
        height += drawTextFeed(painter, feed, option, topLeft + QPoint(0, height)) + 3;
    }

    switch (feed.attach().type()) {
    case QAttachment::LinkAttach:
        height += drawLinkFeed(painter, feed, option, topLeft + QPoint(0, height));
        break;

    default:
        ;
    }

    QRect contentRect(topLeft, QPoint(option.rect.right(), topLeft.y() + height));

    if (painter) {
        // Draw time
        painter->save();
        painter->setPen(QColor(Qt::lightGray));
        painter->drawText(contentRect, Qt::AlignTop, feed.created().time().toString());
        painter->restore();
    }

    return contentRect;
}

void FeedDelegate::paint(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const
{
    QMutexLocker lock(&mLock);
    Q_UNUSED(lock);

    beforeRendering(index, option);
    drawFeedItem(painter, option, index);
    afterRendering(index);
}

QSize FeedDelegate::sizeHint(const QStyleOptionViewItem& option_, const QModelIndex& index) const
{
    QStyleOptionViewItem option = option_;
    QAbstractScrollArea* scrollArea = qobject_cast<QAbstractScrollArea*>(parent());

    assert(scrollArea != 0);
    if (scrollArea != 0)
        option.rect.setWidth(scrollArea->viewport()->width());

    int height = drawFeedItem(0, option, index);
    return QSize(option.rect.width(), height);
}

void FeedDelegate::setFriends(FriendList friendList)
{
    mFriendCache.clear();
    foreach (const Friend& fr, friendList)
        mFriendCache[fr.ownerId()] = fr;
}

void FeedDelegate::setAccounts(QString, AccountList accountList)
{
    mLocalFriendCache.clear();
    foreach (Account* account, accountList) {
        const Friend& localFriend = account->getProfile(false);
        mLocalFriendCache[localFriend.ownerId()] = localFriend;
    }
}

bool FeedDelegate::findFriend(const QString& id, Friend& fr) const
{
    // Try find id in friend's cache
    QHash<QString, Friend>::const_iterator iter = mFriendCache.find(id);

    // If not found, search it in local friend's cache
    if (iter == mFriendCache.constEnd())
        iter = mLocalFriendCache.find(id);

    if (iter == mLocalFriendCache.constEnd())
        return false;

    fr = *iter;
    return true;
}

int FeedDelegate::drawTextFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    QString text = Utils::decodeString(feed.text());
    text.replace(QRegExp(newlineRegex), "\n");

    QSize size(option.rect.right() - topLeft.x(), 0);


    QRect textRect = option.fontMetrics.boundingRect(QRect(topLeft, size),
                                                     Qt::TextWordWrap,
                                                     text);

    if (painter) {
        painter->drawText(textRect, Qt::AlignTop | Qt::TextWordWrap, text);
    }

    return textRect.height();
}

QSize FeedDelegate::drawPhotoFeed(QPainter* painter, const QEventFeed& feed, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    Q_UNUSED(option);

    const QAttachment& attachment = feed.attach();
    QImage photo(attachment.icon());

    if (photo.isNull())
        return QSize();

    if (photo.height() <= PHOTO_SIZE && photo.width() <= PHOTO_SIZE) {
        if (painter) {
            painter->drawImage(topLeft, photo);
            addActionRegion(QRect(topLeft, photo.size()), feed);
        }
        return photo.size();
    } else {
        int maxSize = qMax(photo.height(), photo.width());
        qreal factor = qreal(PHOTO_SIZE) / maxSize;
        if (painter) {
            QRect photoRect(topLeft, QSize(int(photo.width() * factor), int(photo.height() * factor)));
            painter->drawImage(photoRect, photo);
            addActionRegion(photoRect, feed);

        }
        return (QSizeF(photo.size()) * factor).toSize();
    }
}

int FeedDelegate::drawLinkFeed(QPainter *painter, const QEventFeed &feed, const QStyleOptionViewItem& option, const QPoint &topLeft) const
{
    if (painter) {
        painter->save();
        painter->setPen(QPen(option.palette.link(), 1));
        QFont linkFont = option.font;
        linkFont.setUnderline(true);
        painter->setFont(linkFont);
    }

    QString text = feed.attach().objectUrl();
    QSize size(option.rect.right() - topLeft.x(), 0);


    QRect textRect = option.fontMetrics.boundingRect(QRect(topLeft, size),
                                                     Qt::TextWordWrap,
                                                     text);

    if (painter) {
        painter->drawText(textRect, Qt::AlignTop | Qt::TextWordWrap, text);
        painter->restore();
        // Add link region
        addActionRegion(textRect, feed);
    }

    return textRect.height();
}

QRect FeedDelegate::drawPhotoFeedList(QPainter* painter, QList<QEventFeed>& feedList, const QStyleOptionViewItem& option, const QPoint& topLeft) const
{
    int remainingWidth = option.rect.width() - topLeft.x();
    int maxHeight = -1;

    int i = 0;
    while (remainingWidth >= PHOTO_SIZE && !feedList.isEmpty()) {
        const QEventFeed& feedToDraw = feedList.first();
        QSize photoSize = drawPhotoFeed(painter, feedToDraw, option,
                                   QPoint(option.rect.width() - remainingWidth, topLeft.y()));

        if (photoSize.height() > maxHeight)
            maxHeight = photoSize.height();

        feedList.removeFirst();
        remainingWidth -= photoSize.width() + BETWEEN_PHOTO_DISTANCE;
        i++;
    }

    return QRect(topLeft, QSize(option.rect.width() - topLeft.x(), maxHeight));
}

void FeedDelegate::beforeRendering(const QModelIndex& index, const QStyleOptionViewItem& option) const
{
    mIsStoringRegions = true;
    mCurrentIndex = index;
    mCurrentRect = option.rect;
    mCurrentRegions.clear();
}

void FeedDelegate::afterRendering(const QModelIndex& index) const
{
    mIsStoringRegions = false;
    // XXX: replace this hack if possible
    const_cast<QAbstractItemModel*>(index.model())->setData(index, QVariant::fromValue(mCurrentRegions), FeedModel::MapRole);
    mCurrentIndex = QModelIndex();
}

void FeedDelegate::addActionRegion(const QRect& rect, const QEventFeed& value) const
{
    if (mIsStoringRegions) {
        // Store rectangle relative to upper-left corner of current rectangle (option.rect)
        mCurrentRegions.addRegion(rect.translated(-mCurrentRect.topLeft()), QVariant::fromValue(value));
    }
}
