#include "qmessagelistmodel.h"

#include <QStringList>

int QMessageListModel::rowCount(const QModelIndex &) const
{
    return shownMessagesList.isEmpty() ? 0 : shownMessagesList.count();
}

int QMessageListModel::columnCount(const QModelIndex &) const
{
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    return 1;
#else
    return 3;
#endif
}

QVariant QMessageListModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (index.row() >= shownMessagesList.size())
        return QVariant();

    if (role == Qt::DisplayRole) {
        QStringList data;

        // TODO: why QSharedPointer ???
        QSharedPointer<Message> msg = QSharedPointer<Message>(new Message(shownMessagesList.at(index.row())));
        data << msg->subject() << msg->time();

        if (index.column() < data.count())
            return data.at(index.column());
    }

    if (role == Qt::UserRole) {
        QVariant v;
        v.setValue(shownMessagesList.at(index.row()));
        return v;
    }

    if (role == SortRole) {
        if (index.column() == 0) { // Sort by message title
            return shownMessagesList.at(index.row()).subject();
        } else if (index.column() == 1) { // Sort by date
            return QDateTime::fromString(shownMessagesList.at(index.row()).time(), "dd.MM.yyyy hh:mm:ss");
        }
    }

    return QVariant();
}

QVariant QMessageListModel::headerData(int section, Qt::Orientation orientation,
                                       int role) const
{
    if (role != Qt::DisplayRole)
        return QVariant();

    switch (section) {
    case 0:
        return tr("Title");
    case 1:
        return tr("Date");
    case 2:
        return tr("From/To");
    default:
        break;
    }

    return QVariant();
}

void QMessageListModel::updateList(const MessageList &list)
{
    msgList = list;

    beginResetModel();
    shownMessagesList = list;
    endResetModel();

    emit messageCountChanged(list.size());
}

void QMessageListModel::readMessage(const QString &accountId, const QString &messageId)
{
    for (int i = 0; i < this->msgList.length(); i++)
        if (this->msgList.at(i).messageId().compare(messageId) == 0 &&
                this->msgList.at(i).accountId().compare(accountId) == 0)
        {
            this->msgList[i].setReaded();
            break;
        }

    for (int i = 0; i < this->shownMessagesList.length(); i++)
        if (this->shownMessagesList.at(i).messageId().compare(messageId) == 0 &&
                this->shownMessagesList.at(i).accountId().compare(accountId) == 0)
        {
            this->shownMessagesList[i].setReaded();
            emit dataChanged(this->index(i), this->index(i));
            break;
        }

}

void QMessageListModel::setMessagesList(MessageList &list)
{
    this->shownMessagesList = list;
    emit this->dataChanged(this->index(0), this->index(this->shownMessagesList.length()));
    emit messageCountChanged(list.size());
}

QModelIndex QMessageListModel::getIndex(const Message &msg) const
{
    return this->getIndex(msg.accountId(), msg.messageId());
}

QModelIndex QMessageListModel::getIndex(const QString& accountId, const QString& messageId) const
{
    for (int i = 0; i < this->shownMessagesList.length(); i++)
    {
        if (accountId == this->shownMessagesList.at(i).accountId() &&
                messageId == this->shownMessagesList.at(i).messageId())
        {
            return this->index(i);
        }
    }

    return QModelIndex();
}

QModelIndex QMessageListModel::getIndex(int column, const Message &msg) const
{
    QString accountId = msg.accountId();
    QString messageId = msg.messageId();
    for (int i = 0; i < this->shownMessagesList.length(); i++) {
        if (accountId == this->shownMessagesList.at(i).accountId() &&
                messageId == this->shownMessagesList.at(i).messageId())
        {
            return this->index(i, column);
        }
    }

    return QModelIndex();
}
