#include "feedmodel.h"

#include <QHash>

#include "servicemgr.h"

struct FeedKey
{
    QString ownerId;
    QDate date;
};

static uint qHash(const FeedKey& key)
{
    return qHash(key.ownerId) ^ qHash(QDate(0, 0, 0).daysTo(key.date));
}

static bool operator==(const FeedKey& key1, const FeedKey& key2)
{
    return key1.date == key2.date && key1.ownerId == key2.ownerId;
}

struct CompareFeedsByTime {
    bool operator()(const QEventFeed& feed1, const QEventFeed& feed2)
    {
        return feed1.created() < feed2.created();
    }
};

struct CompareFeedListsByTime {
    bool operator()(const QList<QEventFeed>& feedList1, const QList<QEventFeed>& feedList2)
    {
        QDateTime dateTime1 = feedList1.first().created();
        QDateTime dateTime2 = feedList2.first().created();
        return dateTime1 < dateTime2;
    }
};

FeedModel::FeedModel(ServiceMgr* mgr, QObject *parent) :
    QAbstractListModel(parent), mServiceMgr(mgr), mFeedCount(0), mGroupItems(false)
{

    if (mServiceMgr->settings->value(SETTINGS_GROUP_FEEDS).toBool())
        mGroupItems = true;
}

void FeedModel::setFeed(QEventFeedList list, QEventFeed::FeedType feedType, bool isLastUpdate)
{
    Q_UNUSED(isLastUpdate);
    if (list.empty())
        return;

    bool isNewType = false;

    QHash<QEventFeed::FeedType, FeedTypeState>::Iterator iter = mFeedByType.find(feedType);
    if (iter == mFeedByType.constEnd()) {
        iter = mFeedByType.insert(feedType, FeedTypeState());
        isNewType = true;
    }

    iter->list = list;
    updateFeedItemList();

    if (isNewType)
        emit newFeedType(feedType);

    if (isLastUpdate) {
        emit feedCountChanged(mFeedCount);
    }
}

int FeedModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);

    return mGroupedFeed.count();
}

Qt::ItemFlags FeedModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    else
        return QAbstractItemModel::flags(index);
}

QVariant FeedModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || index.row() >= mGroupedFeed.size())
        return QVariant();

    switch (role) {
    case FeedModel::FeedItemRole:
        return QVariant::fromValue(mGroupedFeed.at(index.row()));
        break;

    case Qt::DisplayRole:
        return QString("Not implemented");
        break;

    case FeedModel::MapRole:
        if (mRegions.contains(index.row()))
            return mRegions.value(index.row());
        else
            return QVariant();
        break;

    default:
        return QVariant();
    }
}

bool FeedModel::setData(const QModelIndex& index, const QVariant& value, int role)
{
    if (role == FeedModel::MapRole) {
        mRegions.insert(index.row(), value);
        return true;
    }

    return false;
}

QVariant FeedModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (role != Qt::DisplayRole)
        return QVariant();
    else if (orientation == Qt::Horizontal)
        return "Feed";
    else
        return QString("%1").arg(section);
}

int FeedModel::feedCount() const
{
    return mFeedCount;
}

void FeedModel::updateFeedItemList()
{
    beginResetModel();

    if (mGroupItems) {
        QHash<FeedKey, QList<QEventFeed> > hash;

        // Group feeds by author and date
        foreach (const FeedTypeState& feedTypeState, mFeedByType.values()) {
            if (feedTypeState.active) {
                const QEventFeedList& list = feedTypeState.list;
                foreach (const QEventFeed& feed, list) {
                    FeedKey key;
                    key.ownerId = feed.ownerId();
                    key.date = feed.created().date();
                    hash[key].append(feed);
                }
            }
        }
        mGroupedFeed = hash.values();

        // Sort each group by time
        QMutableListIterator<QList<QEventFeed> > iter(mGroupedFeed);
        while(iter.hasNext()) {
            QList<QEventFeed>& feedList = iter.next();
            qSort(feedList.begin(), feedList.end(), CompareFeedsByTime());
        }
    } else {
        mGroupedFeed.clear();
        foreach (const FeedTypeState& feedTypeState, mFeedByType.values()) {
            if (feedTypeState.active) {
                const QEventFeedList& list = feedTypeState.list;
                foreach (const QEventFeed& feed, list)
                    mGroupedFeed.append(FeedItem() << feed);
            }
        }
    }

    // Sort groups by date
    qSort(mGroupedFeed.begin(), mGroupedFeed.end(), CompareFeedListsByTime());
    mRegions.clear();
    endResetModel();

    updateFeedCount();
}

FeedModel::FeedTypeState::FeedTypeState()
    : active(true)
{
}

void FeedModel::setFeedTypeState(QEventFeed::FeedType feedType, bool state)
{
    QHash<QEventFeed::FeedType, FeedTypeState>::Iterator iter = mFeedByType.find(feedType);
    if (iter != mFeedByType.constEnd()) {
        if (iter->active != state) {
            iter->active = state;
            updateFeedItemList();
        }
    }
}

bool FeedModel::feedTypeState(QEventFeed::FeedType feedType) const
{
    return mFeedByType.value(feedType).active;
}

void FeedModel::updateFeedCount()
{
    mFeedCount = 0;
    foreach (const FeedItem& item, mGroupedFeed) {
        mFeedCount += item.size();
    }
}
