#include "messagebrowserpanel.h"
#include "ui_messagebrowserpanel.h"

#include <QToolBar>

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"
#include "filter/filtermessagetype.h"

#include "config.h"

MessageBrowserPanel::MessageBrowserPanel(QFont uf, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::MessageBrowserPanel),
    currentFriend(Friend()),
    isPortraitMode(false),
    state(WindowMessageList),
    msgState(MessagesInbox)
{
    this->usedFont = uf;
    this->sm = new ServiceMgr;

    ui->setupUi(this);
    init();

    this->setState();
}

MessageBrowserPanel::MessageBrowserPanel(QFont uf, ServiceMgr *mgr, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::MessageBrowserPanel),
        currentFriend(Friend()),
        isPortraitMode(false),
        state(WindowMessageList),
        msgState(MessagesOutbox)
{
    this->usedFont = uf;

    if (mgr != NULL)
        this->sm = mgr;
    else
        this->sm = new ServiceMgr(this);

    ui->setupUi(this);
    init();

    this->setState();
}

MessageBrowserPanel::~MessageBrowserPanel()
{
    delete ui;
}

void MessageBrowserPanel::init()
{
    // Title labels
    QLabel* messagesLabel = new QLabel();
    messagesLabel->setPixmap(QIcon(":/res/messages.png").pixmap(ICON_SIZE));
    messagesLabel->setAlignment(Qt::AlignRight|Qt::AlignVCenter);
    ui->layoutMessagesTop->addWidget(messagesLabel);

    messagesCountLabel = new QLabel(tr("Messages (%1)").arg(0));
    ui->layoutMessagesTop->addWidget(messagesCountLabel);


    qDebug() << __FILE__ << ":" << __LINE__ << " - " << __FUNCTION__;
    mb = new MessageBrowser(this->sm, this);
    ui->layoutMessageBrowserBottom->addWidget(mb, 0);

    buttonBack = new QPushButton(this);
    buttonBack->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonBack->setIcon(QIcon::fromTheme("general_back",
                                         QIcon(":/res/general_back.png")));
    buttonBack->setFlat(TRUE);
    ui->layoutMessageBrowserTop->addWidget(buttonBack, 0);
    buttonBack->setVisible(FALSE);

    ui->layoutMessageBrowserTop->addWidget(buttonNewMsg = new QPushButton(this), 0);
    buttonNewMsg->setFlat(TRUE);
    buttonNewMsg->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonNewMsg->setIcon(QIcon(":/res/msg_new.png"));

    ui->layoutMessageBrowserTop->addWidget(buttonReply = new QPushButton(this), 0);
    buttonReply->setFlat(TRUE);
    buttonReply->setEnabled(FALSE);
    buttonReply->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonReply->setIcon(QIcon(":/res/msg_reply.png"));

    ui->layoutMessageBrowserTop->addWidget(buttonEditDraft = new QPushButton(this), 0);
    buttonEditDraft->setFlat(TRUE);
    buttonEditDraft->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonEditDraft->setIcon(QIcon(":/res/msg_edit.png"));
    buttonEditDraft->setEnabled(FALSE);

    ui->layoutMessageBrowserTop->addWidget(buttonDeleteMsg = new QPushButton(this), 0);
    buttonDeleteMsg->setFlat(TRUE);
    buttonDeleteMsg->setEnabled(FALSE);
    buttonDeleteMsg->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonDeleteMsg->setIcon(QIcon(":/res/msg_delete.png"));

    ui->layoutMessagesTop->addWidget(buttonRefreshMessages = new QPushButton(this), 0);
    buttonRefreshMessages->setFlat(TRUE);
    buttonRefreshMessages->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    buttonRefreshMessages->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    // message list
    this->messagesPanel = new QMessageListWidget(this->usedFont, sm, this);
    this->ui->layoutMessages->addWidget(this->messagesPanel);

    // Filters
    QList<QActionGroup*> actionsList = sm->filterManager()->createActions<Message>();
    foreach (QActionGroup* actions, actionsList) {
#if !defined(Q_WS_MAEMO_5) && !defined(MEEGO_EDITION_HARMATTAN)
        messagesPanel->filtersToolBar()->addSeparator();
#endif
        messagesPanel->filtersToolBar()->addActions(actions->actions());
    }

    connect(buttonBack, SIGNAL(clicked()), this, SLOT(hideMessage()));
    connect(buttonDeleteMsg, SIGNAL(clicked()), this, SLOT(deleteMessageClicked()));
    connect(buttonEditDraft, SIGNAL(clicked()), this, SLOT(editDraftMessage()));
    connect(buttonNewMsg, SIGNAL(clicked()), this, SLOT(newMessageClicked()));
    connect(buttonReply, SIGNAL(clicked()), this, SLOT(replyMessageClicked()));
    connect(this->messagesPanel, SIGNAL(ownerSelected(Message)), this,
            SLOT(ownrSelected(Message)));
    connect(buttonRefreshMessages, SIGNAL(clicked()), this,
            SLOT(refreshListMessages()));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    this->isFriendMessages = false;
    this->isMessagesUpdate = false;

    // connect to signals from Service Manager
    connect(sm, SIGNAL(updateMessageList(MessageList, bool)), this,
            SLOT(updateListMessages(MessageList, bool)));
    connect(sm, SIGNAL(updateAccounts(QString, AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));

    // Connect to message count update
    connect(messagesPanel, SIGNAL(messageCountChanged(int)), this, SLOT(setMessageCount(int)));

    // check that one acount must be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshListMessages(false);
    }

    orientationChanged();
}

void MessageBrowserPanel::updateAccounts(QString, AccountList)
{
    if (this->sm->getAccounts().length() != 0)
            refreshListMessages(false);
}

void MessageBrowserPanel::updateListMessages(MessageList list, bool isLastUpdate)
{
    Q_UNUSED(list);

    Message curMsg = this->messagesPanel->getSelectedMessage();

    if (isLastUpdate) {
        this->isMessagesUpdate = false;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh",
                                                        QIcon(":/res/general_refresh.png")));
    } else {
        this->isMessagesUpdate = true;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_stop",
                                                        QIcon(":/res/general_stop.png")));
    }

    setMessageCount(list.count());

    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;
    if (isMultiWindow)
        this->updateMessage(curMsg);
    else {
        this->setState(WindowMessageList);
    }
}

void MessageBrowserPanel::ownrSelected(Message msg, bool isNeedUpdateState)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": msgId=" << msg.messageId();

    updateMessage(msg);
    emit openMessage(msg);

    if (this->sm->canSendMessage(msg.accountId()))
        this->buttonReply->setEnabled(true);
    else
        this->buttonReply->setEnabled(false);

    if (isNeedUpdateState)
        this->setState(WindowMessage);
}

void MessageBrowserPanel::updateMessage(Message message)
{
    Message curMsg = this->messagesPanel->getSelectedMessage();

    // recepient for VKontakte service
    if (message.recepientListConst().isEmpty()) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == message.accountId()) {
                Friend profile = list.at(i)->getProfile(false);

                Recepient rec;
                rec.name = profile.name();
                rec.id = profile.ownerId();
                message.recepientList().append(rec);
            }
        }
    }

    // sender for VKontakte service
    if (message.senderName().isEmpty()) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == message.accountId()) {
                Friend profile = list.at(i)->getProfile(false);

                message.setSenderId(profile.ownerId());
                message.setSenderName(profile.name());
            }
        }
    }

    if (message.messageId() == curMsg.messageId()) {
        mb->setMessage(message);
    }

    this->buttonDeleteMsg->setEnabled(TRUE);

    if (!message.time().isEmpty()) {
        this->buttonReply->setEnabled(TRUE);
        this->buttonEditDraft->setEnabled(FALSE);
    } else {
        if (!message.messageId().isEmpty()) {
            this->buttonReply->setEnabled(FALSE);
            this->buttonEditDraft->setEnabled(TRUE);
        } else {
            this->buttonReply->setEnabled(FALSE);
            this->buttonEditDraft->setEnabled(FALSE);
        }
    }

    this->setState(WindowMessage);
}

void MessageBrowserPanel::ownrSelected(Message item)
{
    this->ownrSelected(item, true);
}

void MessageBrowserPanel::refreshListMessages()
{
    this->refreshListMessages(true);
}

void MessageBrowserPanel::refreshListMessages(bool isNeedUpdate)
{
    if (!this->isMessagesUpdate) {
        this->isMessagesUpdate = true;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        this->mb->resetMessage();
        sm->getMessages(isNeedUpdate, true);
    } else {
        // TODO: add stopping messages update
    }
}

void MessageBrowserPanel::deleteMessageClicked()
{
    Message curMsg = this->messagesPanel->getSelectedMessage();

    // delete draft message
    if (curMsg.time().isEmpty()) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == curMsg.accountId()) {
                MessageList list2 = list.at(i)->getDraftMessages();

                for (int j = 0; j < list2.count(); j++) {
                    if (list2.at(j).messageId() == curMsg.messageId())
                        list2.removeAt(j);
                }
                list.at(i)->setDraftMessages(list2);
                emit deleteMessage(curMsg);
                refreshListMessages();
                break;
            }
        }
    } else {
        // delete general message
        this->sm->deleteMessage(curMsg);
    }
}

void MessageBrowserPanel::newMessageClicked()
{
    emit newMessage();
}

void MessageBrowserPanel::replyMessageClicked()
{
     Message curMsg = this->messagesPanel->getSelectedMessage();
     emit replyMessage(curMsg);
}

void MessageBrowserPanel::editDraftMessage()
{
    Message curMsg = this->messagesPanel->getSelectedMessage();
    emit editDraftMessage(curMsg);
}

void MessageBrowserPanel::setMessagesState(MessagesState state)
{
    this->msgState = state;    
}

void MessageBrowserPanel::setState()
{
    this->setState(this->state);
}

void MessageBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    this->messagesPanel->setMultipanelEnabled(isMultiWindow);

    switch(this->state) {
    case WindowMessageList:
        setMessagesVisible(true);

        if (isMultiWindow) {
            showAll(ui->layoutMessageBrowser);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutMessageBrowser);
        }
        break;
    case WindowMessage:
        showAll(ui->layoutMessageBrowser);

        if (isMultiWindow) {
            setMessagesVisible(true);
            buttonBack->setVisible(FALSE);
        } else {
            setMessagesVisible(false);
            buttonBack->setVisible(TRUE);
        }
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void MessageBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void MessageBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void MessageBrowserPanel::hideMessage()
{
    this->setState(WindowMessageList);
}

void MessageBrowserPanel::showFriendMessages(Friend fd)
{
    this->mb->resetMessage();
    this->messagesPanel->showFriendMessages(fd);
}

void MessageBrowserPanel::setMessagesVisible(const bool isShow)
{
    if (isShow) {
        this->showAll(this->ui->layoutMessages);
    } else {
        this->hideAll(this->ui->layoutMessages);
    }
}

void MessageBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    foreach (QAction* action, messagesPanel->filtersToolBar()->actions()) {
        if ((action->isSeparator())
                || (action->data().value<FilterMessageType::MessageType>() == FilterMessageType::ALL)) {
            if (!isPortraitMode)
                action->setVisible(false);
            else
                action->setVisible(true);
        }
    }
#endif

}

void MessageBrowserPanel::setMessageCount(int count)
{
    messagesCountLabel->setText(tr("Messages (%1)").arg(count));
}
