#include "qmessagelistmodel.h"

#include <QStringList>

inline bool titleAscending(const Message & msg1 , const Message & msg2)
{
    QSharedPointer<Message> m1 = QSharedPointer<Message>(new Message(msg1));
    QSharedPointer<Message> m2 = QSharedPointer<Message>(new Message(msg2));
    return m1->subject() < m2->subject();
}

inline bool titleDescending(const Message & msg1 , const Message & msg2)
{
    QSharedPointer<Message> m1 = QSharedPointer<Message>(new Message(msg1));
    QSharedPointer<Message> m2 = QSharedPointer<Message>(new Message(msg2));
    return m1->subject() > m2->subject();
}

inline bool dateAscending(const Message & msg1 , const Message & msg2)
{
    QDateTime dt1 = QDateTime::fromString(msg1.time(), "dd.MM.yyyy hh:mm:ss");
    QDateTime dt2 = QDateTime::fromString(msg2.time(), "dd.MM.yyyy hh:mm:ss");
    return dt1 < dt2;
}

inline bool dateDescending(const Message & msg1 , const Message & msg2)
{
    QDateTime dt1 = QDateTime::fromString(msg1.time(), "dd.MM.yyyy hh:mm:ss");
    QDateTime dt2 = QDateTime::fromString(msg2.time(), "dd.MM.yyyy hh:mm:ss");
    return dt1 > dt2;
}

int QMessageListModel::rowCount(const QModelIndex &) const
 {
     return shownMessagesList.isEmpty() ? 0 : shownMessagesList.count();
 }

int QMessageListModel::columnCount(const QModelIndex &) const
{
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    return 1;
#else
    return 3;
#endif
}

QVariant QMessageListModel::data(const QModelIndex &index, int role) const
 {
     if (!index.isValid())
         return QVariant();

     if (index.row() >= shownMessagesList.size())
         return QVariant();

     if (role == Qt::DisplayRole) {
         QStringList data;
         QSharedPointer<Message> msg = QSharedPointer<Message>(new Message(shownMessagesList.at(index.row())));
         data << msg->subject() << msg->time();

         if (index.column() < data.count())
             return data.at(index.column());
     }

     if (role == Qt::UserRole) {
         QVariant v;
         v.setValue(shownMessagesList.at(index.row()));
         return v;
     }

     return QVariant();
 }

QVariant QMessageListModel::headerData(int section, Qt::Orientation orientation,
                                      int role) const
 {
     if (role != Qt::DisplayRole)
         return QVariant();

     switch (section) {
     case 0:
         return tr("Title");
     case 1:
         return tr("Date");
     case 2:
         return tr("From/To");
     default:
         break;
     }

     return QVariant();
 }

void QMessageListModel::setNameFilter(const QString &str)
{
    this->nameFilter = str;
    this->updateShownMessagesList();

    // as default sort list by date
    sort(1, Qt::DescendingOrder);
}

void QMessageListModel::updateList(const MessageList &list)
{
    this->shownMessagesList.clear();
    MessageList newMessages(list);
    MessageList ret = msgList;
    QList<int> dels;

    // search of old and new messages
    for (int i = 0; i < msgList.length(); i++) {
        Message oldMsg = msgList.at(i);
        bool isFound = false;

        int removedIndex = -1;

        for (int j = 0; j < newMessages.length(); j++) {
            Message newMsg = newMessages.at(j);

            // this message is already in message list, so skip it
            if ((oldMsg.accountId() == newMsg.accountId()) &&
                    (oldMsg.messageId() == newMsg.messageId())) {
                removedIndex = j;
                isFound = true;
                break;
            }
        }

        if (isFound && removedIndex >= 0)
            newMessages.removeAt(removedIndex);

        if (!isFound)
            dels.append(i);
    }

    // delete old messages from list and update its rows
    for (int i = 0; i < dels.length(); i++) {
        ret.removeAt(dels.at(i) - i);

        // update row with this message
        QModelIndex idx = this->index(dels.at(i));
        emit dataChanged(idx, idx);
    }

    // move new messages to end of message list and update this part of list
    ret.append(newMessages);

    this->msgList = ret;
    this->shownMessagesList = ret;

    emit dataChanged(this->index(ret.length() - newMessages.length()), this->index(ret.length()));

    // as default sort list by date
    sort(1, Qt::DescendingOrder);
}

void QMessageListModel::readMessage(const QString &accountId, const QString &messageId)
{
    for (int i = 0; i < this->msgList.length(); i++)
        if (this->msgList.at(i).messageId().compare(messageId) == 0 &&
                this->msgList.at(i).accountId().compare(accountId) == 0)
        {
            this->msgList[i].setReaded();
            break;
        }

    for (int i = 0; i < this->shownMessagesList.length(); i++)
        if (this->shownMessagesList.at(i).messageId().compare(messageId) == 0 &&
                this->shownMessagesList.at(i).accountId().compare(accountId) == 0)
        {
            this->shownMessagesList[i].setReaded();
            emit dataChanged(this->index(i), this->index(i));
            break;
        }

}

void QMessageListModel::updateShownMessagesList()
{
    this->shownMessagesList.clear();

    for(int i = 0; i < this->msgList.length(); i++)
    {
        Message *msg = new Message(msgList.at(i));
        if (msg->subject().indexOf(this->nameFilter, 0, Qt::CaseInsensitive) >= 0)
            this->shownMessagesList.append(this->msgList.at(i));
    }

    emit this->dataChanged(this->index(0), this->index(this->shownMessagesList.length()));
}

void QMessageListModel::sort(int column, Qt::SortOrder order)
{
    emit layoutAboutToBeChanged();

    switch (column) {
        // sort by message title
        case 0:
            if (order == Qt::AscendingOrder)
                qSort(shownMessagesList.begin(), shownMessagesList.end(), titleAscending);
            else
                qSort(shownMessagesList.begin(), shownMessagesList.end(), titleDescending);
            break;
        // sort by date
        case 1:
            if (order == Qt::AscendingOrder)
                qSort(shownMessagesList.begin(), shownMessagesList.end(), dateAscending);
            else
                qSort(shownMessagesList.begin(), shownMessagesList.end(), dateDescending);
            break;
        // sort by sender/recepient
        case 2: {
//            if (order == Qt::AscendingOrder)
//                qSort(shownMessagesList.begin(), shownMessagesList.end(), authorAscending);
//            else
//                qSort(shownMessagesList.begin(), shownMessagesList.end(), authorDescending);
            break;
        }
        default:
            break;
    }

    emit layoutChanged();
}

void QMessageListModel::setMessagesList(MessageList &list)
{
    this->shownMessagesList = list;
    emit this->dataChanged(this->index(0), this->index(this->shownMessagesList.length()));
}

QModelIndex QMessageListModel::getIndex(const Message &msg) const
{
    return this->getIndex(msg.accountId(), msg.messageId());
}

QModelIndex QMessageListModel::getIndex(const QString& accountId, const QString& messageId) const
{
    for (int i = 0; i < this->shownMessagesList.length(); i++)
    {
        if (accountId == this->shownMessagesList.at(i).accountId() &&
                messageId == this->shownMessagesList.at(i).messageId())
        {
            return this->index(i);
        }
    }

    return QModelIndex();
}

QModelIndex QMessageListModel::getIndex(int column, const Message &msg) const
{
    QString accountId = msg.accountId();
    QString messageId = msg.messageId();
    for (int i = 0; i < this->shownMessagesList.length(); i++) {
        if (accountId == this->shownMessagesList.at(i).accountId() &&
                messageId == this->shownMessagesList.at(i).messageId())
        {
            return this->index(i, column);
        }
    }

    return QModelIndex();
}
