/**
  \file messagebrowserpanel.h
  \brief MessageBrowserPanel class description
  \author MySocials team
  \date 2011-03-23
  */
#ifndef MESSAGEBROWSERPANEL_H
#define MESSAGEBROWSERPANEL_H

#include <QWidget>
#include <QListWidget>
#include <QMessageBox>
#include <QTreeWidget>
#include <QPushButton>
#include <QRadioButton>
#include <QDesktopWidget>

#include "datatypes/message.h"
#include "datatypes/friend.h"

#include "widgets/messagebrowser.h"
#include "widgets/uiaccountsettings.h"
#include "widgets/messages/qmessagelistwidget.h"

#include "servicemgr.h"

namespace Ui {
    class MessageBrowserPanel;
}

/*!
 \class MessageBrowserPanel
 \brief Panel for messaging with friends.

 This class is based on QWidget. It allows browsing inbox and sent messages,
 deleting, sending, creating new messages and saving drafts. Message is shown
 by Message Browser (see \a MessageBrowser).
 The list of inbox or sent messages gets from Service Manager (see \a ServiceMgr).

 \sa ServiceMgr
 */
class MessageBrowserPanel : public QWidget
{
    Q_OBJECT
public:
    /*!
      A panel constructor.

      \param parent Pointer to parent widget.
     */
    explicit MessageBrowserPanel(QFont usedFont, QWidget *parent = 0);

    /*!
      A panel constructor.

      \param mgr Pointer to Service Manager.

      \param parent Pointer to parent widget.
     */
    explicit MessageBrowserPanel(QFont usedFont, ServiceMgr *mgr, QWidget *parent = 0);

    //! A destructor.
    ~MessageBrowserPanel();

    /*!
     \brief Initialization function.

     Initializes UI elements of panel and connects signals to slots.
     */
    void init();

    /*!
     Hides specified layout and all layouts contained in it.

     \param layout Pointer to layout which must be hidden.
     */
    void hideAll(QLayout *layout);

    /*!
     Shows specified layout and all layouts contained in it.

     \param layout Pointer to layout which must be shown.
     */
    void showAll(QLayout *layout);

    /*!
     \brief States of window.

     It is using for single-panel mode for defining which layout must be shown
     at current moment.
     */
    enum WindowState {
        WindowMessageList,  /*!< State, when list of messages is shown */
        WindowMessage       /*!< State, when selected message is shown */
    };

    /*!
     \brief Sets state of window.

     \param state
     */
    void setState(WindowState state);

    /*!
     \brief Updates window corresponding current state.
     */
    void setState();

    /*!
     \brief States of message list.

     It is using for defining current state of message list widget.
     */
    enum MessagesState {
        MessagesInbox,  /*!< Inbox messages state */
        MessagesOutbox, /*!< Outbox messages state */
        MessagesDrafts  /*!< Drafts messages state */
    };

    /*!
     \brief Sets current state of message list widget.

     \param state
     */
    void setMessagesState(MessagesState state);

    void setMessagesVisible(const bool isShow);

signals:
    /*!
     This signal is emitted when user selects a message.

     \param msg Selected message
     */
    void openMessage(Message msg);

    /*!
     This signal is emitted when user tries to delete message.

     \param msg Current message
     */
    void deleteMessage(Message msg);

    /*!
      This signal is emitted when user tries to create new message.
     */
    void newMessage();

    /*!
      This signal is emitted when user tries to reply on current message.
     */
    void replyMessage(Message msg);

    /*!
      This signal is emitted when user tries to edit draft of message.
     */
    void editDraftMessage(Message msg);
public slots:
    /*!
     \brief Filters friend messages.

     This function is used for filtering friend messages.

     \param fd Current friend.
     */
    void showFriendMessages(Friend fd);

private:
    Ui::MessageBrowserPanel *ui;
    ServiceMgr *sm;
    MessageBrowser *mb;
    QMessageListWidget *messagesPanel;

    Friend currentFriend;

    bool isFriendMessages;
    bool isPortraitMode;
    bool isMessagesUpdate;

    // current state for window
    WindowState state;

    MessagesState msgState;

    QPushButton *buttonRefreshMessages;
    QPushButton *buttonInbox;
    QPushButton *buttonOutbox;    
    QPushButton *buttonDrafts;
    QPushButton *buttonNewMsg;
    QPushButton *buttonSaveDraft;
    QPushButton *buttonEditDraft;
    QPushButton *buttonReply;
    QPushButton *buttonDeleteMsg;
    QPushButton *buttonBack;
    QFont usedFont;

    QLabel* messagesCountLabel;

    void ownrSelected(Message msg, bool isNeedUpdateState);

private slots:
    void deleteMessageClicked();
    void newMessageClicked();
    void replyMessageClicked();
    void editDraftMessage();

    void orientationChanged();
    void ownrSelected(Message msg);

    void hideMessage();

    void refreshListMessages();
    void refreshListMessages(bool isNeedUpdate);

    void updateListMessages(MessageList list, bool isLastUpdate);
    void updateMessage(Message message);
    void updateAccounts(QString accoutId, AccountList list);
};

#endif // MESSAGEBROWSERPANEL_H
