#include "imagebrowserpanel.h"
#include "ui_imagebrowserpanel.h"

#include <QTimer>
#include <QToolBar>

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"
#include "widgets/profilebrowser.h"
#include "widgets/albuminfowidget.h"

#include "config.h"

ImageBrowserPanel::ImageBrowserPanel(QFont usedFont, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::ImageBrowserPanel), state(WindowFriendAlbum), isPortraitMode(false)
{
    this->usedFont = usedFont;
    this->sm = new ServiceMgr(this);
    ui->setupUi(this);
    init();

    this->isFriendsUpdate = false;
    this->isAlbumsUpdate = false;
    this->isPhotosUpdate = false;

    this->isPhotosWidthSet = false;
    photosWidth = 0;

    //connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)),
            this, SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateAlbumList(Friend,AlbumList,bool)),
            this, SLOT(updateListAlbums(Friend,AlbumList, bool)));
    connect(sm, SIGNAL(updatePhotoList(QString,QString,QString,PhotoList,bool)),
            this, SLOT(updateListPhotos(QString,QString,QString,PhotoList, bool)));
    connect(sm, SIGNAL(errorOccured(QString,QTransport::Action,bool)),
            SLOT(gotError(QString, QTransport::Action, bool)));
    connect(sm, SIGNAL(updateAccounts(QString,AccountList)), this,
            SLOT(updateAccounts(QString,AccountList)));

    // check that one acount mast be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshFriends(false);
        this->refreshAlbums(false);
//        this->refreshPhotos(false);
    }

    this->setState();
}

ImageBrowserPanel::ImageBrowserPanel(QFont usedFont, ServiceMgr *mgr, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ImageBrowserPanel), state(WindowFriendAlbum), isPortraitMode(false)
{
    this->usedFont = usedFont;
    this->sm = mgr;
    ui->setupUi(this);
    init();

    this->isFriendsUpdate = false;
    this->isAlbumsUpdate = false;
    this->isPhotosUpdate = false;

    this->isPhotosWidthSet = false;
    photosWidth = 0;

    //connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)),
            this, SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateAlbumList(Friend,AlbumList, bool)),
            this, SLOT(updateListAlbums(Friend,AlbumList, bool)));
    connect(sm, SIGNAL(updatePhotoList(QString,QString,QString,PhotoList,bool)),
            this, SLOT(updateListPhotos(QString,QString,QString,PhotoList, bool)));
    connect(sm, SIGNAL(errorOccured(QString,QTransport::Action,bool)),
            SLOT(gotError(QString, QTransport::Action, bool)));
    connect(sm, SIGNAL(updateAccounts(QString, AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));

    // check that one acount mast be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshFriends(false);
        this->refreshAlbums(false);
//        this->refreshPhotos(false);
    }

    this->setState();
}

ImageBrowserPanel::~ImageBrowserPanel()
{
    delete ui;
}

void ImageBrowserPanel::init()
{
    // Title labels
    QLabel* friendLabel = new QLabel();
    friendLabel->setPixmap(QIcon(":/res/general_contacts.png").pixmap(ICON_SIZE));
    ui->layoutFriendsTop->addWidget(friendLabel);
    friendLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    friendsCountLabel = new QLabel(tr("People (%1)").arg(0));
    ui->layoutFriendsTop->addWidget(friendsCountLabel);

    // friends list
    QIcon icon;
    ui->layoutFriendsTop->addWidget(buttonRefreshFriends = new QPushButton(this), 0);
    buttonRefreshFriends->setFlat(TRUE);
    buttonRefreshFriends->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                   QIcon(":/res/general_refresh.png")));
    buttonRefreshFriends->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    this->friendsPanel = new QFriendListWidget(this->usedFont, sm, this);
    this->ui->layoutFriends->addWidget(this->friendsPanel);

    // albums
    this->albumsTopPanel = new QTopPanel(this);
    this->ui->layoutAlbums->insertWidget(0, this->albumsTopPanel);

    icon = QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png"));
    this->albumsTopPanel->setPixmap(icon.pixmap(ICON_SIZE, QIcon::Normal, QIcon::On));

    this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh",
                                                          QIcon(":/res/general_refresh.png")));
    this->albumsTopPanel->setBackIcon(QIcon::fromTheme("general_back",
                                                       QIcon(":/res/general_back.png")));

    connect(this->albumsTopPanel, SIGNAL(clicked()), this, SLOT(showFriendInfo()));

    this->ui->listAlbums->setFont(this->usedFont);
    this->albumsTopPanel->setFont(this->usedFont);

    // Profile info widget (ProfileBrowser) for albums top panel
    this->profileInfoWidget = new ProfileBrowser(this->sm, false);
    this->profileInfoWidget->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
    this->albumsTopPanel->setInfoWidget(this->profileInfoWidget);
    connect(this, SIGNAL(showFriendInfo(Friend)), this->profileInfoWidget, SLOT(setProfile(Friend)));

    // photos
    this->photosTopPanel = new QTopPanel(this);
    this->ui->layoutPhotos->insertWidget(0, this->photosTopPanel);

    icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
    this->photosTopPanel->setPixmap(icon.pixmap(ICON_SIZE, QIcon::Normal, QIcon::On));

    this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh",
                                                          QIcon(":/res/general_refresh.png")));
    this->photosTopPanel->setBackIcon(QIcon::fromTheme("general_back",
                                                       QIcon(":/res/general_back.png")));

    ui->listPhotos->setDragDropMode(QAbstractItemView::NoDragDrop);
    this->ui->listPhotos->setFont(this->usedFont);
    this->photosTopPanel->setFont(this->usedFont);

    // Album info widget for photos top panel
    this->albumInfoWidget = new AlbumInfoWidget(this->photosTopPanel);
    this->photosTopPanel->setInfoWidget(this->albumInfoWidget);
    connect(this, SIGNAL(showAlbumInfo(Album)), this->albumInfoWidget, SLOT(setAlbum(Album)));

    // Filters
    QList<QActionGroup*> actionsList = sm->filterManager()->createActions<Album>();
    foreach (QActionGroup* actions, actionsList) {
#if !defined(Q_WS_MAEMO_5) && !defined(MEEGO_EDITION_HARMATTAN)
        friendsPanel->filtersToolBar()->addSeparator();
#endif
        friendsPanel->filtersToolBar()->addActions(actions->actions());
    }

    // signals
    connect(this->friendsPanel, SIGNAL(ownerSelected(Friend)),
            this, SLOT(ownrSelected(Friend)));
    connect(ui->listAlbums, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(albmSelected(QListWidgetItem*)));
    connect(ui->listPhotos, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(photoSelected(QListWidgetItem*)));
    connect(buttonRefreshFriends, SIGNAL(clicked()),
            this, SLOT(refreshFriends()));
    connect(this->albumsTopPanel, SIGNAL(refreshPressed()),
            this, SLOT(refreshAlbums()));
    connect(this->albumsTopPanel, SIGNAL(backPressed()),
            this, SLOT(hideAlbums()));

    connect(this->photosTopPanel, SIGNAL(refreshPressed()),
            this, SLOT(refreshPhotos()));
    connect(this->photosTopPanel, SIGNAL(backPressed()),
            this, SLOT(hidePhotos()));

    // enable sorting
    this->ui->listAlbums->setSortingEnabled(true);

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

void ImageBrowserPanel::updateAccounts(QString, AccountList )
{
    if (this->sm->getAccounts().length() != 0) {
        this->refreshFriends(false);
        this->refreshAlbums(false);
        this->refreshPhotos(false);
    }
}

QListWidgetItem * ImageBrowserPanel::createFriendWidget(Friend fr)
{
    QListWidgetItem *item = new QListWidgetItem(fr.name());

    QImage img = QImage(fr.icon());
    if (!img.isNull()) {
        if (img.height() > img.width())
            img = img.copy(0, (img.height() - img.width()) / 2, img.width(), img.width());

        if (img.width() > img.height())
            img = img.copy((img.width() - img.height()) / 2, 0, img.height(), img.height());

        item->setIcon(QIcon(QPixmap::fromImage(img)));
    }

    if (item->icon().isNull())
        item->setIcon(QIcon::fromTheme("general_default_avatar", QIcon(":/res/general_default_avatar.png")));

    QVariant v;
    v.setValue(fr);
    item->setData(Qt::UserRole, v);

    return item;
}

void ImageBrowserPanel::ownrSelected(Friend fd, bool isNeedUpdateState)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": Get albums for " << fd.ownerId();

    if (fd.isEmpty())
        this->sm->getAlbums(false, true);
    else
        this->sm->getAlbums(fd, false, true);
    this->setAlbumHeader(fd);

    if (isNeedUpdateState)
        this->setState(WindowAlbumFriend);
}

void ImageBrowserPanel::ownrSelected(Friend item)
{
    this->ownrSelected(item, true);
}

void ImageBrowserPanel::showFriendAlbums(Friend fd)
{
    this->friendsPanel->selectFriend(fd);
    this->ownrSelected(fd, true);
}

void ImageBrowserPanel::albmSelected(QListWidgetItem *item)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":"<< __FUNCTION__<< "(): Selected album name: " << item->text();
    this->setState(WindowPhotoAlbum);

    if (!isPhotosWidthSet) {
        photosWidth = ui->listPhotos->viewport()->rect().width() - 18;
        this->isPhotosWidthSet = true;
    }

    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<Album>()) {
        Album fd = v.value<Album>();
        qDebug() << "Get photos for " << fd.albumId();
        this->sm->getPhotos(fd, false, true, true);
        this->setPhotoHeader(fd);
    } else {
        this->setPhotoHeader();
    }

}

QString ImageBrowserPanel::albumItem(const Album& al)
{
    QString name;
    if (al.title().isEmpty())
        name = tr("Untitled album");
    else
        name = Utils::decodeString(al.title());

    QString str = name + "\n";
    str += tr("%1 photo(s)", "", al.size()).arg(al.size()) + ".\n";
    if (al.size() != 0 && !al.time_update().isEmpty())
        str += tr(" Updated: ") + al.time_update();

    return str;
}

QPixmap ImageBrowserPanel::albumIcon(const Album& al)
{
    QPixmap pm(al.icon());
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png"));
        pm = icon.pixmap(IMAGE_SIZE, QIcon::Normal, QIcon::On);
        if (pm.width() < IMAGE_SIZE)
            pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    } else {
        if (pm.width() > pm.height())
            pm = pm.copy((pm.width() - pm.height()) / 2, 0, pm.height(), pm.height());

        if (pm.height() > pm.width())
            pm = pm.copy(0, (pm.height() - pm.width()) / 2, pm.width(), pm.width());

        pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    }
    return pm;
}

void ImageBrowserPanel::setAlbumHeader()
{
    this->albumsTopPanel->setText(tr("Album(s):"));
    this->albumsTopPanel->setPixmap(QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png")).pixmap(48, QIcon::Normal, QIcon::On));
}

void ImageBrowserPanel::setAlbumHeader(const Friend& curFr)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() fr.isEmpty()=" << (curFr.isEmpty() ? "true" : "false");
    Friend fd = curFr;
    if (fd.isEmpty())
    {
        fd = sm->getMyProfile(false);
    }
    this->albumsTopPanel->setText(Utils::decodeString(fd.name()));
    QPixmap pm(fd.icon());
    if (!pm.isNull()) {
        if (pm.width() > pm.height())
            pm = pm.copy((pm.width() - pm.height()) / 2, 0, pm.height(), pm.height());

        if (pm.height() > pm.width())
            pm = pm.copy(0, (pm.height() - pm.width()) / 2, pm.width(), pm.width());

        pm = pm.scaled(QSize(ICON_SIZE, ICON_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
        this->albumsTopPanel->setPixmap(pm);
    } else {
        this->albumsTopPanel->setPixmap(QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png")).pixmap(48, QIcon::Normal, QIcon::On));
    }
}

QPixmap ImageBrowserPanel::photoIcon(const Photo& al)
{
    QPixmap pm(al.icon());
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
        pm = icon.pixmap(IMAGE_SIZE, QIcon::Normal, QIcon::On);
        if (pm.width() < IMAGE_SIZE)
            pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    } else {
        if (pm.width() > pm.height())
            pm = pm.copy((pm.width() - pm.height()) / 2, 0, pm.height(), pm.height());

        if (pm.height() > pm.width())
            pm = pm.copy(0, (pm.height() - pm.width()) / 2, pm.width(), pm.width());

        pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    }

    return pm;
}

QPixmap ImageBrowserPanel::photoIcon(const QAttachment& at)
{
    QPixmap pm(at.icon());
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
        pm = icon.pixmap(IMAGE_SIZE, QIcon::Normal, QIcon::On);
        if (pm.width() < IMAGE_SIZE)
            pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    } else {
        if (pm.width() > pm.height())
            pm = pm.copy((pm.width() - pm.height()) / 2, 0, pm.height(), pm.height());

        if (pm.height() > pm.width())
            pm = pm.copy(0, (pm.height() - pm.width()) / 2, pm.width(), pm.width());

        pm = pm.scaled(QSize(IMAGE_SIZE, IMAGE_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);
    }
    return pm;
}

void ImageBrowserPanel::setPhotoHeader()
{
    QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
    this->photosTopPanel->setPixmap(icon.pixmap(ICON_SIZE, QIcon::Normal, QIcon::On));
    this->photosTopPanel->setText(tr("Photo(s):"));
}

void ImageBrowserPanel::setPhotoHeader(Album al)
{
    al.setTitle(cutHeader(al.title()));
    al.setDescription(cutHeader(al.description()));

    this->photosTopPanel->setText(Utils::decodeString(al.title()) +
                         (al.description().isNull() ? "": "\n" +
                          Utils::decodeString(al.description())) +
                         "\n" + tr("%1 photo(s):", "", al.size()).arg(al.size()));

    QPixmap pm(al.icon());
    if (pm.isNull()) {
        this->photosTopPanel->setPixmap( QIcon::fromTheme("general_image", QIcon(":/res/general_image.png")).pixmap(48, QIcon::Normal, QIcon::On));
    } else {
        if (pm.width() > pm.height())
            pm = pm.copy((pm.width() - pm.height()) / 2, 0, pm.height(), pm.height());

        if (pm.height() > pm.width())
            pm = pm.copy(0, (pm.height() - pm.width()) / 2, pm.width(), pm.width());

        pm = pm.scaled(QSize(ICON_SIZE, ICON_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation);

        this->photosTopPanel->setPixmap(pm);
    }

    // Update album info
    emit showAlbumInfo(al);
}

void ImageBrowserPanel::photoSelected(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<Photo>()) {
        Photo fd = v.value<Photo>();
        emit this->openImage(fd);
    }
}

void ImageBrowserPanel::updateListFriends(FriendList list, bool isLastUpdate)
{
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): list.length()="<<list.length() << "; isLastUpdate=" << isLastUpdate;

    if (isLastUpdate) {
        this->isFriendsUpdate = false;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

    this->friendsCountLabel->setText(tr("People (%1)").arg(list.length()));


    Friend curFr = this->friendsPanel->getSelectedFriend();

    this->setAlbumHeader(curFr);
}

void ImageBrowserPanel::updateListAlbums(Friend owner, AlbumList list, bool isLastUpdate)
{
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): owner=" << owner.ownerId() << ";list.length()="<<list.length() << "; isLastUpdate=" << isLastUpdate;
    bool isFoundSelectedFriend = false;

    AccountList accs = this->sm->getAccounts();
    Friend fd = this->friendsPanel->getSelectedFriend();

    if (fd.hasProfile(owner)) {
        isFoundSelectedFriend = true;
    }

    // check my albums
    if (!isFoundSelectedFriend && fd.isEmpty())
    {
        for (int i = 0; i < accs.length(); i++) {
            if (accs.at(i)->getProfile(false).hasProfile(owner))
            {
                isFoundSelectedFriend = true;
                break;
            }
        }
    }

    if (!isFoundSelectedFriend)
    {
        qDebug() << __FILE__ <<":"<<__LINE__<<":"<< __FUNCTION__ << " unused album list with owner=" << owner.ownerId();
        return;
    }
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): check passed";
    if (isLastUpdate) {
        this->isAlbumsUpdate = false;
        this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isAlbumsUpdate = true;
        this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

    if (list.isEmpty()) {
        qDebug() << "Empty album list for " << owner.ownerId() << "!!!!!!";
        ui->listAlbums->clear();
        if (isLastUpdate)
            new QListWidgetItem(tr("No albums"), ui->listAlbums); //QListWidgetItem *item2 = new
        else
            new QListWidgetItem(tr("Download albums"), this->ui->listAlbums);
        ui->listAlbums->setDisabled(true);
        return;
    }

    ui->listAlbums->setDisabled(false);

    QList<QListWidgetItem *> delItems;
    AlbumList newAlbums;
    newAlbums.append(list);

    // search new albums in list of widgets
    for (int i = 0; i < this->ui->listAlbums->count(); i++) {
        QListWidgetItem *item = this->ui->listAlbums->item(i);
        QVariant v = item->data(Qt::UserRole);

        bool isFound = false;
        int removedVal = -1;

        if (v.canConvert<Album>()) {
            Album oldAlbum = v.value<Album>();

            for (int j = 0; j < newAlbums.length(); j++) {
                Album curAlbum = newAlbums.at(j);
                if (oldAlbum.albumId() == curAlbum.albumId() &&
                    oldAlbum.accountId() == curAlbum.accountId()) {
                    removedVal = j;
                    item->setIcon(QIcon(this->albumIcon(curAlbum)));
                    item->setData(Qt::DisplayRole, this->albumItem(curAlbum));

                    v.setValue(curAlbum);
                    item->setData(Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newAlbums.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove album with id " << oldAlbum.albumId() << " from " << oldAlbum.accountId();
            }
        }

        if (!isFound)
            delItems.append(item);
    }

    qDebug() << "Will be add " << newAlbums.length() << " albums and removed " << delItems.length() << " records. Total widgets: " << ui->listAlbums->count();

    // add new albums
    for (int i = 0; i < newAlbums.length(); i++) {
        //QListWidgetItem *item = new QListWidgetItem(str, ui->listAlbums);
        //item->setIcon(QIcon(list[i]->icon));
        //if (item->icon().isNull())
        //    item->setIcon(QPixmap(":/res/ms.png"));
        QListWidgetItem *albumitem = new QListWidgetItem(ui->listAlbums);
        albumitem->setIcon(QIcon(this->albumIcon(newAlbums.at(i))));

        albumitem->setData(Qt::DisplayRole, this->albumItem(newAlbums.at(i)));

        QVariant v;
        v.setValue(newAlbums[i]);
        albumitem->setData(Qt::UserRole, v);
    }

    // del old albums
    for (int i = 0; i < delItems.length(); i++) {
        if (delItems.at(i)->isSelected())
        {
            qDebug() << __FILE__<< __LINE__ << __FUNCTION__ << "(): delete items for owner" << owner.ownerId() << "; new length=" << list.length();
            this->hidePhotos();
        }
        delete delItems.at(i);
    }

    // update photo header
    if (this->ui->listAlbums->selectedItems().length() > 0) {
        QVariant v = this->ui->listAlbums->selectedItems().at(0)->data(Qt::UserRole);
        Album al = v.value<Album>();
        this->setPhotoHeader(al);
    }
}

void ImageBrowserPanel::updateListPhotos(QString accountId, QString ownerId, QString albumId, PhotoList list, bool isLastUpdate)
{
    Q_UNUSED(accountId);
    Q_UNUSED(ownerId);

    // check correctness that list was return for selected photo list
    bool isFoundSelectedAlbum = false;
    QList<QListWidgetItem *> selItems = ui->listAlbums->selectedItems();
    for (int i = 0; i < selItems.length(); i++) {
        QVariant v = selItems[i]->data(Qt::UserRole);
        if (v.canConvert<Album>()) {
            Album fd = v.value<Album>();
            if (albumId == fd.albumId()) {
                isFoundSelectedAlbum = true;
                break;
            }
        }
    }
    if (!isFoundSelectedAlbum)
        return;

    if (isLastUpdate) {
        this->isPhotosUpdate = false;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isPhotosUpdate = true;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop",
                                                              QIcon(":/res/general_stop.png")));
    }

    if (list.isEmpty()) {
        this->ui->listPhotos->clear();
        if (isLastUpdate)
            new QListWidgetItem(tr("No images"), ui->listPhotos);
        else
            new QListWidgetItem(tr("Download images"), this->ui->listPhotos);
        ui->listPhotos->setDisabled(true);
        return;
    }

    ui->listPhotos->setDisabled(false);

    QList<QListWidgetItem *> delItems;
    PhotoList newPhoto;
    newPhoto.append(list);

    // search new images in list of widgets
    for (int i = 0; i < this->ui->listPhotos->count(); i++) {
        QListWidgetItem *item = this->ui->listPhotos->item(i);
        QVariant v = item->data(Qt::UserRole);

        bool isFound = false;
        int removedVal = -1;

        if (v.canConvert<Photo>()) {
            Photo oldPhoto = v.value<Photo>();

            for (int j = 0; j < newPhoto.length(); j++) {
                Photo curPhoto = newPhoto.at(j);
                if (oldPhoto.photoId() == curPhoto.photoId() &&
                    oldPhoto.accountId() == curPhoto.accountId()) {
                    removedVal = j;

                    QPixmap icon = this->photoIcon(curPhoto);
                    item->setIcon(QIcon(icon));
                    //((QAbstractItemView*)this->ui->listPhotos)->setIconSize(icon.size());

                    v.setValue(curPhoto);
                    item->setData(Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newPhoto.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove photo with id " << oldPhoto.photoId() << " from " << oldPhoto.accountId();
            }
        }

        if (!isFound)
            delItems.append(item);
    }
    this->ui->listPhotos->setUpdatesEnabled(false);

    // add new photos
    for (int i = 0; i < newPhoto.length(); i++) {
        qDebug() << "Show photo_thumb: " << newPhoto.at(i).photoId();
        QListWidgetItem *item = new QListWidgetItem(NULL, ui->listPhotos);
        QPixmap icon = this->photoIcon(newPhoto.at(i));
        item->setIcon(QIcon(icon));
        QVariant v;
        v.setValue(newPhoto.at(i));
        item->setData(Qt::UserRole, v);
    }

    // del old photos
    for (int i = 0; i < delItems.length(); i++)
        delete delItems.at(i);

    ((QAbstractItemView*)this->ui->listPhotos)->setIconSize(QSize(IMAGE_SIZE, IMAGE_SIZE));
    this->ui->listPhotos->setUpdatesEnabled(true);
    this->ui->listPhotos->repaint();
}

void ImageBrowserPanel::refreshFriends()
{
    this->refreshFriends(true);
}

void ImageBrowserPanel::refreshFriends(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":" << __FUNCTION__<<"()";

    // friends refresh
    if (!this->isFriendsUpdate) {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        sm->getFriends(isNeedUpdate, true);
    } else {
        //TODO: add stopping friends update
    }
}

void ImageBrowserPanel::refreshAlbums()
{
    this->refreshAlbums(true);
}

void ImageBrowserPanel::refreshAlbums(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): isNeedUpdate=" << isNeedUpdate << "; albumUpdate=" << this->isAlbumsUpdate;

    if (!this->isAlbumsUpdate) {
        Friend fd = this->friendsPanel->getSelectedFriend();
        if (!fd.isEmpty()) {
            qDebug() << "Get albums for " << fd.ownerId();
            this->sm->getAlbums(fd, isNeedUpdate, true);
        } else {
            qDebug("Try to load my albums");
            this->sm->getAlbums(isNeedUpdate, true);
        }
    } else {
        //TODO: add stopping albums update
    }
}

void ImageBrowserPanel::refreshPhotos()
{
    this->refreshPhotos(true);
}

void ImageBrowserPanel::refreshPhotos(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":" << __FUNCTION__<<"()";
    QListWidgetItem *item = this->ui->listAlbums->currentItem();
    if (item == NULL) {
        qWarning() << __FILE__ <<":"<< __LINE__ << ":" << "unknown selected item";
        return;
    }

    if (!this->isPhotosUpdate) {
        this->isPhotosUpdate = true;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        qDebug() << "Update photos for " << item->text();
        QVariant v = item->data(Qt::UserRole);
        if (v.canConvert<Album>()) {
            Album fd = v.value<Album>();
            qDebug() << "Get photos for " << fd.albumId();
            this->sm->getPhotos(fd, isNeedUpdate, true, true);
        }
    }
}

void ImageBrowserPanel::hidePhotos()
{
    this->setState(WindowAlbumPhoto);
}

void ImageBrowserPanel::hideAlbums()
{
    this->setState(WindowFriendAlbum);
}

void ImageBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void ImageBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void ImageBrowserPanel::setAlbumsVisible(const bool isShow)
{
    if (isShow)
        this->showAll(this->ui->layoutAlbums);
    else
        this->hideAll(this->ui->layoutAlbums);
}

void ImageBrowserPanel::setImagesVisible(const bool isShow)
{
    if (isShow)
        this->showAll(this->ui->layoutPhotos);
    else
        this->hideAll(this->ui->layoutPhotos);
}

void ImageBrowserPanel::setFriendsVisible(const bool isShow)
{
    if (isShow)
    {
        this->showAll(this->ui->layoutFriendsTop);
        this->showFriendPanel();
    } else {
        this->hideAll(this->ui->layoutFriends);
    }
}

void ImageBrowserPanel::showFriendPanel()
{
    this->friendsPanel->show();
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;
    setAlbumsVisible(isMultiWindow);
    if (this->isFriendsUpdate)
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    else
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
}

void ImageBrowserPanel::setState()
{
    this->setState(this->state);
}

void ImageBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): multiwindow=" << isMultiWindow;

    switch(this->state) {
    case WindowAlbumFriend:
        this->setAlbumsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->albumsTopPanel->setBackVisible(false);
            this->setFriendsVisible(true);
        } else {
            this->albumsTopPanel->setBackVisible(true);
            this->setFriendsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show albums from friends";
        break;
    case WindowAlbumPhoto: // from photo to album
        this->setAlbumsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->albumsTopPanel->setBackVisible(false);
            this->photosTopPanel->setBackVisible(false);
            this->setFriendsVisible(true);
        } else {
            this->albumsTopPanel->setBackVisible(true);
            this->photosTopPanel->setBackVisible(true);
            this->setFriendsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show albums from photos";
        break;
    case WindowFriendAlbum:
        this->setFriendsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->albumsTopPanel->setBackVisible(false);
            this->setAlbumsVisible(true);
        } else {
            this->setAlbumsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show friends";
        break;
    case WindowPhotoAlbum:
        this->setFriendsVisible(false);
        this->setImagesVisible(true);

        if (isMultiWindow) {
            this->albumsTopPanel->setBackVisible(true);
            this->photosTopPanel->setBackVisible(false);
            this->setAlbumsVisible(true);
        } else {
            this->photosTopPanel->setBackVisible(true);
            this->setAlbumsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show photos";
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void ImageBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;
    this->setState();
    this->ui->listPhotos->repaint();
}

void ImageBrowserPanel::showFriendInfo()
{
    Friend fd = this->friendsPanel->getSelectedFriend();
    if (fd.isEmpty())
        fd = sm->getMyProfile(false);
    qDebug() << __FILE__ << ":" << __LINE__ << ": emit showFriendInfo for " << fd.accountId();
    emit showFriendInfo(fd);
}

QString ImageBrowserPanel::cutHeader(QString text)
{
    if (text.isEmpty()) {
        return text;
    } else if (text.length() <= 40) {
        return text;
    } else if (text.length() > 40) {
        text.truncate(30);
        text.truncate(text.lastIndexOf(QRegExp("\\s")));
        text = text.trimmed();
        text.append(QString("..."));
        return text;
    }

    return QString::null;
}

void ImageBrowserPanel::gotError(QString, QTransport::Action action, bool)
{
    switch(action)
    {
    case QTransport::getListFriendsAction:

        if (this->sm->getNumThreadsFriendUpdate() == 0)
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                               QIcon(":/res/general_refresh.png")));
        else
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));

        break;
    case QTransport::getListAlbumsAction:
        if (this->sm->getNumThreadsAlbumUpdate() == 0)
        {
            this->isAlbumsUpdate = false;
            this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
            if (ui->listAlbums->count() == 0 || ui->listAlbums->item(0)->text().compare(tr("Download albums")) == 0) {
                ui->listAlbums->clear();
                new QListWidgetItem(tr("No albums"), ui->listAlbums); //QListWidgetItem *item2 = new
                ui->listAlbums->setDisabled(true);
            }
        } else {
            this->isAlbumsUpdate = true;
            this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        }
        break;
    case QTransport::getListPhotosAction:
        if (this->sm->getNumThreadsPhotoUpdate() == 0)
        {
            this->isPhotosUpdate = false;
            this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
        } else {
            this->isPhotosUpdate = true;
            this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        }
        break;
    default:
        break;
    }
}
