/*******************
 * Interface builder
 *******************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <hildon/hildon-gtk.h>
#include <gtk/gtk.h>
#include <pango/pango.h>

#include <hildon-extras/he-fullscreen-button.h>
#include <hildon-extras/he-helper.h>
#include <libraw/libraw.h>

#include "appdata.h"
#include "interface.h"
#include "callbacks.h"
#include "helpers.h"
#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

/*
 * Main application's window
 */
void create_app(app_data_t *myapp) {
	HildonAppMenu *appmenu;
	GtkWidget *openitem;
	GtkWidget *imgdetitem;
	GtkWidget *fullscreenitem;
	GtkWidget *aboutitem;
	GtkWidget *panarea;
    GtkWidget *eventbox;
    HeFullscreenButton *fsbutton;
    osso_return_t ossores;

    /* Enable accelerometers */
    he_helper_accelerometer_enable(myapp->ossocont);

    /* Create the hildon program and setup the title */
    myapp->app = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("RAW Viewer");

    /* Create HildonWindow and set it to HildonProgram */
    myapp->mainwin = HILDON_WINDOW (hildon_window_new());
    gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_get_application_name());
    hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin), HILDON_PORTRAIT_MODE_SUPPORT);
    hildon_program_add_window(myapp->app, myapp->mainwin);

    /* Create the main menu */
    appmenu = HILDON_APP_MENU(hildon_app_menu_new());

	/* Add menu buttons */
    openitem = gtk_button_new_with_label("Open image");
    g_signal_connect_after(G_OBJECT (openitem),"clicked",G_CALLBACK (open_image_file),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(openitem),2);

    imgdetitem = gtk_button_new_with_label((myapp->showimgparam ? "Hide details" : "View details"));
    g_signal_connect_after(G_OBJECT (imgdetitem),"clicked",G_CALLBACK (toggle_image_details),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON (imgdetitem),3);

    fullscreenitem = gtk_button_new_with_label("Go fullscreen");
    g_signal_connect_after(G_OBJECT (fullscreenitem),"clicked",G_CALLBACK (toggle_fullscreen_mode),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON (fullscreenitem),4);

    aboutitem = gtk_button_new_with_label("About");
    g_signal_connect_after(G_OBJECT (aboutitem),"clicked",G_CALLBACK (create_about),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(aboutitem),5);

    /* Add the main manu to the app itself */
    gtk_widget_show_all(GTK_WIDGET (appmenu));
    hildon_window_set_app_menu(HILDON_WINDOW (myapp->mainwin),appmenu);

	/* Add a pannable area with the image container inside */
    panarea = GTK_WIDGET (hildon_pannable_area_new());
    g_object_set(G_OBJECT (panarea),"mov-mode",HILDON_MOVEMENT_MODE_BOTH,NULL);
    gtk_widget_show(GTK_WIDGET (panarea));
    gtk_container_add(GTK_CONTAINER (myapp->mainwin),panarea);
    eventbox = gtk_event_box_new();
    gtk_widget_show(eventbox);
    gtk_widget_set_events(eventbox,GDK_BUTTON_PRESS | GDK_BUTTON_RELEASE | GDK_KEY_PRESS);
    myapp->image = GTK_WIDGET (gtk_image_new());
    gtk_widget_show(GTK_WIDGET (myapp->image));
    gtk_container_add(GTK_CONTAINER (eventbox),myapp->image);
    hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (panarea),eventbox);

    /* Show everything */
    gtk_widget_show_all(GTK_WIDGET (myapp->mainwin));

    /* Add Fullscreen button */
    fsbutton = he_fullscreen_button_new(GTK_WINDOW (myapp->mainwin));
    g_signal_connect(G_OBJECT (fsbutton),"clicked",G_CALLBACK (toggle_fullscreen_mode),myapp);

    /* Add previous / next arrow buttons */
    myapp->prevarrowbut = arrow_button_new(GTK_WINDOW (myapp->mainwin),ARROW_LEFT);
    myapp->nextarrowbut = arrow_button_new(GTK_WINDOW (myapp->mainwin),ARROW_RIGHT);
    g_signal_connect(G_OBJECT (myapp->prevarrowbut),"clicked",G_CALLBACK (load_prev_image),myapp);
    g_signal_connect(G_OBJECT (myapp->nextarrowbut),"clicked",G_CALLBACK (load_next_image),myapp);

    /* Create image parameters window */
    myapp->imgparamwin = imgparam_window_new(GTK_WINDOW (myapp->mainwin),NULL);
    if (myapp->showimgparam) g_object_set(G_OBJECT (myapp->imgparamwin),"enabled",TRUE,NULL);

    /* Save list of known image files in current directory */
	store_filenames_in_dir(myapp);

    /* Ungrab volume keys */
    ungrab_volume_keys(GTK_WIDGET (myapp->mainwin));

    /* Connect some signals */
    g_signal_connect(G_OBJECT (myapp->mainwin),"destroy",
    		G_CALLBACK (destroy_app),myapp);
    g_signal_connect_after(G_OBJECT (myapp->mainwin),"size-allocate",
    		G_CALLBACK (detect_main_window_changes),myapp);
    g_signal_connect(G_OBJECT(myapp->mainwin),"key-press-event",
			G_CALLBACK (detect_hardware_key_press),myapp);
    g_signal_connect(G_OBJECT(myapp->mainwin),"notify::is-topmost",
			G_CALLBACK (on_is_topmost_changed),myapp);

    /* Connect system bus D-Bus events handler */
	myapp->sysbus = dbus_bus_get(DBUS_BUS_SYSTEM,NULL);
	if (myapp->sysbus) {
		dbus_bus_add_match(myapp->sysbus,MCE_SIGNAL_MATCH,NULL);
		dbus_connection_add_filter(myapp->sysbus,(DBusHandleMessageFunction)detect_mce_signal,myapp,NULL);
	}

	/* Connect session bus D-Bus events handler */
	ossores = osso_rpc_set_cb_f(myapp->ossocont,APP_DBUS_SERVICE,APP_DBUS_OBJECT_PATH,APP_DBUS_INTERFACE,
								(osso_rpc_cb_f *)detect_dbus_message,myapp);
	if (ossores != OSSO_OK) {
		g_print("Error setting session D-Bus default callback (%d)",ossores);
	}

}

/*
 * Create the About dialog
 */
void create_about(GtkWidget *callerobj,app_data_t *myapp) {
	GtkWidget *abdialog;
	GtkWidget *appicon;
	GtkWidget *copyright;
	GtkWidget *separator;
	GtkWidget *panarea;
	GtkWidget *dispmessage;
	gchar *message;
	const char **cameralist;
	gint i;

	abdialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW (abdialog),g_strconcat("About ",PACKAGE," ",VERSION,NULL));
	gtk_window_set_default_size(GTK_WINDOW (abdialog),-1,400);
	appicon = GTK_WIDGET (gtk_image_new_from_file(g_strconcat(HILDONPIXMAPDIR,"/",PACKAGE,".png",NULL)));
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),appicon,FALSE,FALSE,10);
	copyright = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL (copyright),"<span size=\"small\" style=\"italic\">(C) 2009 Luca Donaggio (donaggio@gmail.com)</span>");
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),copyright,FALSE,FALSE,10);
	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),separator,FALSE,FALSE,10);
	panarea = GTK_WIDGET (hildon_pannable_area_new());
	dispmessage = gtk_label_new(NULL);
	gtk_label_set_line_wrap(GTK_LABEL (dispmessage),TRUE);
	message = g_strconcat("<b>",PACKAGE,"</b> is an image viewer for the Maemo platform supporting RAW image file formats as those produced by modern DSLRs.\n\n"
						  "For an exahustive list of supported cameras please see below.\n\n",
						  "<b>",PACKAGE,"</b> is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.\n\n",
						  "<b>",PACKAGE,"</b> is based on LibRaw ",libraw_version()," which in turn is (C) 2008-2009 LibRaw LLC (info@libraw.org)\n\n",
						  "Supported cameras:\n\n<small>",NULL);
	cameralist = libraw_cameraList();
	for(i = 0;i < libraw_cameraCount();i++) message = g_strconcat(message,"\t",cameralist[i],"\n",NULL);
	message = g_strconcat(message,"</small>",NULL);
	gtk_label_set_markup(GTK_LABEL (dispmessage),message);
	hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (panarea),dispmessage);
	gtk_box_pack_end_defaults(GTK_BOX (GTK_DIALOG (abdialog)->vbox),panarea);
	gtk_widget_show_all(abdialog);
	gtk_dialog_run(GTK_DIALOG (abdialog));
	gtk_widget_destroy(abdialog);
}
