// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "moviepainter.h"

#include "data/movie.h"
#include "ratingprovider.h"
#include "ui/styleutils.h"

#include <QPainter>
#include <QStyleOptionViewItem>

MoviePainter::MoviePainter()
    : _rating_height(RatingProvider::INSTANCE()->GetRating(0.0).height())
{
}

void MoviePainter::Paint(QPainter *painter, const QStyleOptionViewItem &option,
                         const Movie *movie) const
{
    painter->setFont(option.font);
    painter->setPen(StyleUtils::INSTANCE()->GetColor(option.palette, StyleUtils::DefaultTextColor));
    painter->drawText(0, painter->fontMetrics().ascent(), movie->GetName());
    int y = painter->fontMetrics().height() + 2;
    int h = option.font.pointSizeF() * 0.7;
    if (_rating_height > h) {
        h = _rating_height;
    }
    painter->setPen(StyleUtils::INSTANCE()->GetColor(option.palette, StyleUtils::SecondaryTextColor));
    QRect bounding_rect;
    bool drawn = false;
    if (movie->GetDuration().isValid()) {
        QFont font2(option.font);
        font2.setPointSizeF(font2.pointSizeF() * 0.7);
        painter->setFont(font2);
        int minutes = movie->GetDuration().hour() * 60 + movie->GetDuration().minute();
        painter->drawText(0, y, 1000, painter->fontMetrics().height(),
                          0 /* Qt::AlignBottom */, QString(QObject::tr("%1min", "movie duration")).arg(minutes), &bounding_rect);
        drawn = true;
    }
    if (!movie->GetComment().isEmpty()) {
        QFont font2(option.font);
        font2.setPointSizeF(font2.pointSizeF() * 0.7);
        painter->setFont(font2);
        painter->drawText(bounding_rect.right(), y, 1000, painter->fontMetrics().height(),
                          0 /* Qt::AlignBottom */, QString(drawn ? ", %1" : "%1").arg(movie->GetComment()), &bounding_rect);
        drawn = true;
    }
    if (movie->GetRate() >= 0.0) {
        QPixmap p = RatingProvider::INSTANCE()->GetRating(movie->GetRate());
        painter->drawPixmap(bounding_rect.right() + (drawn ? 8 : 0), y + h - p.height(), p);
        drawn = true;
    }
}

QSize MoviePainter::GetSizeHint(const QStyleOptionViewItem &option) const
{
    int y0 = option.fontMetrics.height();
    int y1 = option.fontMetrics.height() * 0.7;
    if (_rating_height > y1) {
        y1 = _rating_height;
    }
    return QSize(0, y0 + 2 + y1 + 4);
}
