// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#ifndef CINEMASCHEDULE_H
#define CINEMASCHEDULE_H

#include "location.h"
#include "cinemakey.h"
#include "moviekey.h"
#include "scheduleentrykey.h"
#include "scheduleentry.h"
#include "utils/assertedlock.h"

#include <QHash>
#include <QSet>
#include <QReadWriteLock>

class Cinema;
class Movie;

class CinemaSchedule
{
    Q_DISABLE_COPY(CinemaSchedule);
public:
    typedef QHash<CinemaKey, Cinema *> Cinemas;
    typedef QHash<MovieKey, Movie *> Movies;
    typedef QSet<QDate> ScheduleDates;
    typedef QSet<ScheduleEntry> Schedules;
    typedef QSet<ScheduleEntryKey> ScheduleKeys;
public:
    CinemaSchedule();

    AssertedLock *GetLock() const { return const_cast<AssertedLock *>(&_lock); }

    void Clear();

    bool AreAllCinemasLoaded() const { _lock.AssertLockedForRead(); return _all_cinemas_loaded; }
    void SetAllCinemasLoaded(bool all_cinemas_loaded) { _lock.AssertLockedForWrite(); _all_cinemas_loaded = all_cinemas_loaded; }
    const Cinemas GetCinemas() const { _lock.AssertLockedForRead(); return _cinemas; }
    Cinema *FindCinema(const CinemaKey &);
    const Cinema *FindCinema(const CinemaKey &) const;
    Cinema *AddCinema(const CinemaKey &);

    bool AreAllMoviesLoaded() const { _lock.AssertLockedForRead(); return _all_movies_loaded; }
    void SetAllMoviesLoaded(bool all_movies_loaded) { _lock.AssertLockedForWrite(); _all_movies_loaded = all_movies_loaded; }
    const Movies GetMovies() const { _lock.AssertLockedForRead(); return _movies; }
    Movie *FindMovie(const MovieKey &);
    const Movie *FindMovie(const MovieKey &) const;
    Movie *AddMovie(const MovieKey &);

    const ScheduleDates GetScheduleDates() const;
    const ScheduleDates GetScheduleDates(const Cinema *) const;
    const ScheduleDates GetScheduleDates(const Movie *) const;
    ScheduleEntry FindScheduleEntry(const ScheduleEntryKey &) const;
    const Schedules GetSchedules() const { _lock.AssertLockedForRead(); return _schedules; }
    const Schedules GetSchedules(const Cinema *) const;
    const Schedules GetSchedules(const Movie *) const;
    const ScheduleKeys GetScheduleKeys(const CinemaKey &) const;
    const ScheduleKeys GetScheduleKeys(const MovieKey &) const;
    void AddSchedule(const Cinema *, const Movie *, const QTime &start_time, const QDate &date);

private:
    AssertedLock _lock;
    bool _all_cinemas_loaded;
    Cinemas _cinemas;
    bool _all_movies_loaded;
    Movies _movies;
    Schedules _schedules;
};

inline int qHash(const QDate &date) { return qHash(date.toJulianDay()); }

#endif // CINEMASCHEDULE_H
