// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "abstractsearchclient.h"

#include "utils/timeutils.h"

#include <QMutexLocker>
#include <iostream>

AbstractSearchClient::AbstractSearchClient(QObject *parent)
    : QObject(parent),
    _network(new QNetworkAccessManager(this)),
    _search_task_id(INVALID_SEARCH_TASK_ID),
    _start(0)
{
    connect(_network, SIGNAL(finished(QNetworkReply *)),
            this, SLOT(ReplyFinished(QNetworkReply*)));
    {
        QMutexLocker locker(&_next_search_task_id_mutex);
        _search_task_id = _next_search_task_id++;
    }
}

AbstractSearchClient::~AbstractSearchClient()
{
}

void AbstractSearchClient::Search(const QUrl &url, int start)
{
    _start = start;
    if (start == 0) {
        emit SearchStarted(_search_task_id);
    }
    QNetworkRequest request;
    QUrl xurl(url);
    if (_start != 0 && !url.hasQueryItem("start")) {
        xurl.addQueryItem("start", QString::number(_start));
    }
    FixLocation(&xurl);
    //std::cout << "URL: " << qPrintable(QString(xurl.toEncoded())) << std::endl;
    request.setUrl(xurl);
    //request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.2.8) Gecko/20100723 Ubuntu/9.10 (karmic) Firefox/3.6.8");
    request.setRawHeader("User-Agent", "Mozilla/5.0");
    //request.setRawHeader("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8");
    request.setRawHeader("Accept", "application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8");
    request.setRawHeader("Accept-Language", "en-gb;q=1.0,en;q=0.9,de-de;q=0.5,de;q=0.3");
    //request.setRawHeader("Accept-Language", "en-us,en;q=0.8,de-de;q=0.5,de;q=0.3");
    //request.setRawHeader("Accept-Encoding", "gzip,deflate");
    request.setRawHeader("Accept-Charset", "ISO-8859-1,utf-8;q=0.7,*;q=0.7");
    request.setRawHeader("Keep-Alive", "115");
    request.setRawHeader("Connection", "keep-alive");
    request.setRawHeader("Cache-Control", "max-age=0");
    QNetworkReply *reply = _network->get(request);
    connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(DownloadProgress(qint64,qint64)));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(NetworkError(QNetworkReply::NetworkError)));
}

void AbstractSearchClient::SearchEncodedUrl(const QString &encoded_url, int start)
{
    QUrl url = QUrl::fromEncoded((QString("http://www.google.com") + encoded_url).toAscii());
    Search(url, start);
}

void AbstractSearchClient::DownloadProgress(qint64 a,qint64 b)
{
    //std::cout << "Search Progress of " << qPrintable(objectName()) << " - " << a << ", " << b << std::endl;
    //sleep(1);
    emit Progress(_search_task_id, a, b);
}

void AbstractSearchClient::NetworkError(QNetworkReply::NetworkError error)
{
    emit SearchFinished(_search_task_id, false);
    std::cout << "ERROR: " << error << std::endl;
    emit Error(_search_task_id);
    sender()->deleteLater();
    deleteLater();
}

QList<QTime> AbstractSearchClient::TimesFromString(const QList<QString> &time_strings)
{
    QList<QTime> schedule_times;
    bool am_pm = false;
    Q_FOREACH(const QString time_str, time_strings) {
        if (time_str.endsWith("am", Qt::CaseInsensitive) || time_str.endsWith("pm", Qt::CaseInsensitive)) {
            am_pm = true;
        }
    }
    if (am_pm) {
        int i = 0;
        bool am = true;
        while (i < time_strings.length()) {
            int j = i;
            while (i < time_strings.length()) {
                if (time_strings[i].endsWith("am", Qt::CaseInsensitive)) {
                    am = true;
                    break;
                } else if (time_strings[i].endsWith("pm", Qt::CaseInsensitive)) {
                    am = false;
                    break;
                }
                ++i;
            }
            while (j < i) {
                QString time_str = time_strings[j] + (am ? "am" : "pm");
                QTime time = TimeUtils::FromTimeString(time_str);
                if (time.isValid()) {
                    schedule_times.append(time);
                } else {
                    std::cout << "ERROR: time " << qPrintable(time_str) << " is invalid." << std::endl;
                }
                ++j;
            }
            if (i < time_strings.length()) {
                QString time_str = time_strings[i];
                QTime time = TimeUtils::FromTimeString(time_str);
                if (time.isValid()) {
                    schedule_times.append(time);
                } else {
                    std::cout << "ERROR: time " << qPrintable(time_str) << " is invalid." << std::endl;
                }
                schedule_times.append(time);
            }
            ++i;
        }
    } else {
        Q_FOREACH(const QString time_str, time_strings) {
            QTime time = TimeUtils::FromTimeString(time_str);
            if (time.isValid()) {
                schedule_times.append(time);
            } else {
                std::cout << "ERROR: time " << qPrintable(time_str) << " is invalid." << std::endl;
            }
        }
    }
    return schedule_times;
}

void AbstractSearchClient::FixLocation(QUrl *url)
{
#if 0
    // Try to fix the Google url which returns
    // wrong locations in some links
    if (_location.isNull()) {
        // Fetch location from url on first call (which is still correct)
        if (url->hasQueryItem("loc")) {
            _location = url->queryItemValue("loc");
        } else if (url->hasQueryItem("near")) {
            _location = url->queryItemValue("near");
        } else if (url->hasQueryItem("defaultloc")) {
            _location = url->queryItemValue("defaultloc");
        }
    } else {
        // Replace with fetched location in later calls
        if (url->hasQueryItem("loc")) {
            url->removeAllQueryItems("loc");
            url->addQueryItem("loc", _location);
        }
        if (url->hasQueryItem("near")) {
            url->removeAllQueryItems("near");
            url->addQueryItem("near", _location);
        }
        if (url->hasQueryItem("defaultloc")) {
            url->removeAllQueryItems("defaultloc");
            url->addQueryItem("defaultloc", _location);
        }
    }
#else
    Q_UNUSED(url);
#endif
}

QMutex AbstractSearchClient::_next_search_task_id_mutex;
int AbstractSearchClient::_next_search_task_id = 1;
