// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#ifndef ASSERTEDLOCK_H
#define ASSERTEDLOCK_H

#include <QMutex>
#include <QReadWriteLock>
#include <QMap>

class QThread;

class AssertedLock
{
public:
    enum LockMode { READ, WRITE };

#ifndef NDEBUG
private:
    struct LockData {
        LockData() : _lock_mode(READ), _lock_counter(0) {}
        LockMode _lock_mode;
        int _lock_counter;
    };

    typedef QMap<QThread*, LockData> ThreadData;
#endif

public:
    AssertedLock();
    ~AssertedLock();

    void Lock(LockMode mode);
    void LockForRead() { Lock(READ); }
    void LockForWrite() { Lock(WRITE); }

    void Unlock();

    void AssertLockedForRead() const
    {
#ifndef DEBUG
        AssertLockedForMode(READ);
#endif
    }

    void AssertLockedForWrite() const
    {
#ifndef NDEBUG
        AssertLockedForMode(WRITE);
#endif
    }

    void AssertUnlocked() const
    {
#ifndef NDEBUG
        AssertNotLocked();
#endif
    }

#ifndef NDEBUG
private:
    LockMode GetLockMode() const;
    void SetLockMode(LockMode mode);
    int GetLockCounter() const;
    void IncLockCounter();
    void DecLockCounter();
    void AssertLockedForMode(LockMode mode) const;
    void AssertNotLocked() const;
#endif

private:
#ifndef NDEBUG
    int _lock_counter;
    ThreadData _thread_data;
    mutable QMutex _mutex;
#endif
    QReadWriteLock _lock;
};

#endif // ASSERTEDLOCK_H
