// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "movieschedulemodel.h"

#include "scheduleentryitem.h"
#include "data/cinemaschedule.h"
#include "utils/assertedlocker.h"

#include <iostream>

MovieScheduleModel::MovieScheduleModel(const CinemaSchedule *cinema_schedule, const MovieKey &movie_key, QObject *parent)
    : QAbstractListModel(parent),
    _cinema_schedule(cinema_schedule),
    _movie_key(movie_key)
{
}

void MovieScheduleModel::Update()
{
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    const Movie *movie = _cinema_schedule->FindMovie(_movie_key);
    if (movie != 0) {
        _schedule_dates = _cinema_schedule->GetScheduleDates(movie).toList();
        _schedule_keys = _cinema_schedule->GetScheduleKeys(movie->GetKey()).toList();
    }
}

int MovieScheduleModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);
    return _schedule_dates.size() + _schedule_keys.size();
}

QVariant MovieScheduleModel::data(const QModelIndex &index, int role) const
{
    if (!(index.isValid() && index.row() >= 0 && index.row() < _schedule_dates.size() + _schedule_keys.size())) {
        std::cout << "MovieScheduleModel: invalid index" << std::endl;
        return QVariant();
    }
    if (index.row() < _schedule_dates.size()) {
        QDate date = _schedule_dates[index.row()];
        switch (role) {
        case Qt::DisplayRole:
            return qVariantFromValue(ScheduleEntryItem(date));
        case SortRole:
            return QVariant(date.toString("yyyyMMdd"));
        }
    } else {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        const ScheduleEntryKey &schedule_entry_key = _schedule_keys.operator [](index.row() - _schedule_dates.size());
        const ScheduleEntry schedule_entry = _cinema_schedule->FindScheduleEntry(schedule_entry_key);
        if (schedule_entry.IsValid()) {
            switch (role) {
            case Qt::DisplayRole:
            case MovieScheduleItemRole:
                return qVariantFromValue(ScheduleEntryItem(schedule_entry));
            case MovieScheduleKeyRole:
                return qVariantFromValue(schedule_entry_key);
            case SortRole:
                return QVariant(schedule_entry.GetDate().toString("yyyyMMdd") + schedule_entry.GetStartTime().toString("hhmm")
                                + schedule_entry.GetCinema()->GetName());
            }
        }
    }
    return QVariant();
}

