// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "theatermodel.h"
#include "theaterdelegate.h"
#include "moviemodel.h"
#include "moviedelegate.h"
#include "locationdialog.h"
#include "data/cinemaschedule.h"
#include "data/cinema.h"
#include "data/movie.h"
#include "data/cinemakey.h"
#include "data/moviekey.h"
#include "data/cinemakey.h"
#include "utils/assertedlocker.h"

#include <QCursor>
#include <QtGui>
#ifdef Q_WS_MAEMO_5
#include <QAbstractKineticScroller>
#endif
#include <iostream>

MainWindow::MainWindow(const CinemaSchedule *cinema_schedule, QWidget *parent) :
        AbstractMainWindow(parent),
        ui(new Ui::MainWindow),
        _cinema_schedule(cinema_schedule),
        _theater_context_menu(0),
        _movie_context_menu(0),
        _theater_phone_call(0),
        _theater_find_route(0),
        _theater_search_web(0),
        _movie_search_web(0),
        _theater_model(0),
        _movie_model(0)
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    QStackedLayout *layout = dynamic_cast<QStackedLayout*>(ui->MainStack->layout());
    if (layout != 0) {
        layout->setStackingMode(QStackedLayout::StackAll);
    }
    ui->Theaters->setItemDelegate(new TheaterDelegate(this));
    ui->Theaters->setProperty("FingerScrollable", true);
    ui->Theaters->setContextMenuPolicy(Qt::CustomContextMenu);
    ui->Movies->setItemDelegate(new MovieDelegate(this));
    ui->Movies->setProperty("FingerScrollable", true);
    ui->Movies->setContextMenuPolicy(Qt::CustomContextMenu);
    QFont f(font());
    f.setPointSizeF(f.pointSizeF() * 1.5);
    ui->ErrorMessage->setFont(f);
    connect(ui->Theaters, SIGNAL(clicked(QModelIndex)), this, SLOT(TheaterClicked(QModelIndex)));
    connect(ui->Theaters, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(TheaterContextMenuRequested(QPoint)));
    connect(ui->Movies, SIGNAL(clicked(QModelIndex)), this, SLOT(MovieClicked(QModelIndex)));
    connect(ui->Movies, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(MovieContextMenuRequested(QPoint)));
    ui->menubar->addAction(tr("Movies"), this, SIGNAL(SearchMovies()));
    ui->menubar->addAction(tr("Theaters"), this, SIGNAL(SearchTheaters()));
    ui->menubar->addAction(tr("Location"), this, SIGNAL(OpenLocationDialog()));
    ui->menubar->addAction(tr("Options"), this, SIGNAL(OpenOptionsDialog()));
    ui->menubar->addAction(tr("About"), this, SIGNAL(OpenAboutDialog()));
    _theater_context_menu = new QMenu(this);
    _theater_phone_call = _theater_context_menu->addAction(tr("Call By Phone"), this, SLOT(CallTheaterByPhone()));
    _theater_find_route = _theater_context_menu->addAction(tr("Find Route"), this, SLOT(FindRouteToTheater()));
    _theater_search_web = _theater_context_menu->addAction(tr("Search In Web"), this, SLOT(SearchTheaterInWeb()));
    _movie_context_menu = new QMenu(this);
    _movie_search_web = _movie_context_menu->addAction(tr("Search In Web"), this, SLOT(SearchMovieInWeb()));
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::SetOrientation(Settings::OrientationMode orientation)
{
#ifdef Q_WS_MAEMO_5
    switch (orientation) {
    case Settings::LANDSCAPE:
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        break;
    case Settings::PORTRAIT:
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        break;
    case Settings::AUTOROTATION:
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        break;
    }
#else
    Q_UNUSED(orientation);
#endif
}

void MainWindow::SetLocation(Location location)
{
    _location = location;
    if (_location.IsValid()) {
        ui->Background->SetLabelText(_location.GetLocationName());
    } else {
        ui->Background->SetLabelText("");
    }
}

void MainWindow::SetTheaterModel(QAbstractItemModel *theater_model)
{
    _theater_model = theater_model;
    setWindowTitle(tr("Theaters"));
    ui->Theaters->setModel(theater_model);
    ui->MainPages->setCurrentWidget(ui->TheatersPage);
}

void MainWindow::SetMovieModel(QAbstractItemModel *movie_model)
{
    _movie_model = movie_model;
    setWindowTitle(tr("Movies"));
    ui->Movies->setModel(movie_model);
    ui->MainPages->setCurrentWidget(ui->MoviesPage);
}

void MainWindow::SetError(QString error_text)
{
    setWindowTitle(tr("MovieSchedule"));
    ui->ErrorMessage->setText(error_text);
    ui->MainPages->setCurrentWidget(ui->ErrorPage);
}

void MainWindow::TheaterClicked(QModelIndex index)
{
    emit TheaterSelected(GetCinema(index));
}

void MainWindow::TheaterContextMenuRequested(const QPoint &pos)
{
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    const Cinema *cinema = _cinema_schedule->FindCinema(GetCinema(ui->Theaters->currentIndex()));
    if (cinema != 0) {
        ShowContextMenu(cinema, ui->Theaters->viewport()->mapToGlobal(pos));
    }
}

void MainWindow::MovieClicked(QModelIndex index)
{
    emit MovieSelected(GetMovie(index));
}

void MainWindow::MovieContextMenuRequested(const QPoint &pos)
{
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    const Movie *movie = _cinema_schedule->FindMovie(GetMovie(ui->Movies->currentIndex()));
    if (movie != 0) {
        ShowContextMenu(movie, ui->Movies->viewport()->mapToGlobal(pos));
    }
}

void MainWindow::CallTheaterByPhone()
{
    emit CallTheaterByPhone(GetCinema(ui->Theaters->currentIndex()));
}

void MainWindow::FindRouteToTheater()
{
    emit FindRouteToTheater(GetCinema(ui->Theaters->currentIndex()));
}

void MainWindow::SearchTheaterInWeb()
{
    emit SearchTheaterInWeb(GetCinema(ui->Theaters->currentIndex()));
}

void MainWindow::SearchMovieInWeb()
{
    emit SearchMovieInWeb(GetMovie(ui->Movies->currentIndex()));
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

CinemaKey MainWindow::GetCinema(const QModelIndex &index)
{
    QVariant var = _theater_model->data(index, TheaterModel::TheaterKeyRole);
    CinemaKey cinema_key = qVariantValue<CinemaKey>(var);
    return cinema_key;
}

MovieKey MainWindow::GetMovie(const QModelIndex &index)
{
    QVariant var = _movie_model->data(index, MovieModel::MovieKeyRole);
    MovieKey movie_key = qVariantValue<MovieKey>(var);
    return movie_key;
}

void MainWindow::ShowContextMenu(const Cinema *cinema, const QPoint &pos)
{
    _theater_phone_call->setVisible(!cinema->GetTelephone().isEmpty());
    // TODO enable if find-route works
    _theater_find_route->setVisible(false && !cinema->GetAddress().isEmpty());
    _theater_search_web->setVisible(true);
    _theater_context_menu->popup(pos);
}

void MainWindow::ShowContextMenu(const Movie *movie, const QPoint &pos)
{
    Q_UNUSED(movie);
    _movie_search_web->setVisible(true);
    _movie_context_menu->popup(pos);
}
