// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "theatersearchclient.h"

#include "data/cinemakey.h"
#include "data/cinema.h"
#include "data/cinemaschedule.h"
#include "utils/assertedlocker.h"

#include <QXmlStreamReader>
#include <iostream>

TheaterSearchClient::TheaterSearchClient(CinemaSchedule *cinema_schedule, QObject *parent)
    : AbstractSearchClient(parent),
    _cinema_schedule(cinema_schedule)
{
}

void TheaterSearchClient::SearchTheater(const QString &town)
{
    _semaphore.Activate(GetSearchTaskId());
    setObjectName(QString("TheaterSearchClient:%1").arg(town));
    _town = town;
    Search(0);
}

void TheaterSearchClient::CancelAllRunningSearchs()
{
    _semaphore.CancelAll();
}

void TheaterSearchClient::Search(int start)
{
    QUrl url("http://www.google.com/m/movies");
    url.addQueryItem("loc", _town);
    url.addQueryItem("sort", QString::number(0));
    AbstractSearchClient::Search(url, start);
}

enum State {
    PARSE_HTML,
    PARSE_THEATER_LINK,
    PARSE_THEATER_DIV,
    PARSE_THEATER_BR,
    PARSE_THEATER_SPAN,
    PARSE_THEATER_PHONE
};

void TheaterSearchClient::ReplyFinished(QNetworkReply *reply)
{
    QXmlStreamReader xml(reply);
    State state = PARSE_HTML;
    int found = 0;
    QString theater_name;
    QString theater_address;
    QString theater_movies_url;
    QString theater_phone;
    while (!xml.atEnd()) {
        QXmlStreamReader::TokenType token = xml.readNext();
        if (token == QXmlStreamReader::StartElement) {
            QString attr_href = xml.attributes().value("href").toString();
            //std::cout << "name: " << qPrintable(xml.name().toString()) << ", href: " << qPrintable(attr_href) << std::endl;
            if (state == PARSE_HTML && xml.name() == "a" && attr_href.startsWith("/m/movies")) {
                QUrl url = QUrl::fromEncoded(QString("http://www.google.com" + attr_href).toAscii(), QUrl::TolerantMode);
                //std::cout << "LINK " << qPrintable(attr_href) << std::endl;
                if (url.hasQueryItem("tid")) {
                    theater_name = "";
                    theater_address = "";
                    theater_movies_url = attr_href;
                    theater_phone = "";
                    state = PARSE_THEATER_LINK;
                } else {
                    state = PARSE_HTML;
                }
            } else if (state == PARSE_THEATER_DIV && xml.name() == "br") {
                state = PARSE_THEATER_BR;
            } else if (state == PARSE_THEATER_DIV && xml.name() == "span") {
                state = PARSE_THEATER_SPAN;
            } else if (state == PARSE_THEATER_DIV && xml.name() == "a" && attr_href.startsWith("wtai:")) {
                state = PARSE_THEATER_PHONE;
            } else if (state == PARSE_THEATER_DIV && xml.name() == "a") {
                state = PARSE_THEATER_BR;
            } else {
                state = PARSE_HTML;
            }
        } else if (token == QXmlStreamReader::EndElement) {
            if (state == PARSE_THEATER_LINK) {
                state = PARSE_THEATER_DIV;
            } else if (state == PARSE_THEATER_BR) {
                state = PARSE_THEATER_DIV;
            } else if (state == PARSE_THEATER_SPAN) {
                state = PARSE_THEATER_DIV;
            } else if (state == PARSE_THEATER_PHONE) {
                state = PARSE_THEATER_DIV;
            } else if (state == PARSE_THEATER_DIV) {
                if (!theater_name.isEmpty()) {
                    AssertedWriteLocker locker(_cinema_schedule->GetLock());
                    if (!_semaphore.IsActive(GetSearchTaskId())) {
                        break;
                    }
                    ++found;
                    CinemaKey key(theater_name, theater_address);
                    Cinema *cinema = _cinema_schedule->FindCinema(key);
                    if (cinema == 0) {
                        cinema = _cinema_schedule->AddCinema(key);
                    }
                    if (!theater_movies_url.isEmpty()) {
                        cinema->SetMoviesUrl(theater_movies_url);
                    }
                    if (!theater_phone.isEmpty()) {
                        cinema->SetTelephone(theater_phone);
                    }
                }
                state = PARSE_HTML;
            }
        } else if (token == QXmlStreamReader::Characters) {
            if (state == PARSE_THEATER_LINK) {
                theater_name = xml.text().toString();
            } else if (state == PARSE_THEATER_PHONE) {
                theater_phone = xml.text().toString();
            } else if (state == PARSE_THEATER_SPAN) {
                theater_address = xml.text().toString();
            }
        }
    }
    if (xml.hasError()) {
        emit SearchFinished(GetSearchTaskId(), false);
        std::cout << "xml error (" << xml.lineNumber() << "/" << xml.columnNumber() << "): " << qPrintable(xml.errorString()) << std::endl;
        emit Error(GetSearchTaskId());
        deleteLater();
    } else if (!_semaphore.IsActive(GetSearchTaskId())) {
        emit Cancelled(GetSearchTaskId());
        emit SearchFinished(GetSearchTaskId(), false);
        deleteLater();
    } else {
        if (found > 0) {
            emit Reply(GetSearchTaskId(), true);
            Search(GetStartIndex() + found);
        } else {
            emit Reply(GetSearchTaskId(), false);
            emit SearchFinished(GetSearchTaskId(), true);
            deleteLater();
        }
    }
    reply->deleteLater();
}

SearchClientSemaphore TheaterSearchClient::_semaphore;
