// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "movieschedule.h"

#include "data/settings.h"
#include "data/cinemaschedule.h"
#include "control/settingscontroller.h"
#include "control/maincontroller.h"
#include "control/theatercontroller.h"
#include "control/moviecontroller.h"
#include "control/locationcontroller.h"
#include "control/actioncontroller.h"
#include "control/itemmodelsortcontroller.h"
#include "ui/mainwindow.h"
#include "ui/moviemodel.h"
#include "ui/moviewindow.h"
#include "ui/theatermodel.h"
#include "ui/theaterwindow.h"
#include "ui/ratingprovider.h"
#include "ui/styleutils.h"

#include <QApplication>
#include <QTranslator>
#include <QLocale>
#include <QThread>
#include <iostream>

#ifdef MAEMO_SDK
#include <glib-object.h>
#endif

#define _STRINGIZE_(x) #x
#define _STRINGIZE(x) _STRINGIZE_(x)

class ControllerThread : public QThread
{
public:
    void run()
    {
        exec();
    }

    void quit()
    {
        QThread::quit();
        wait();
    }
};

MovieSchedule::MovieSchedule()
{
}

int MovieSchedule::exec(int argc, char *argv[])
{
    QApplication app(argc, argv);

#ifdef MAEMO_SDK
    g_type_init();
#endif

    qRegisterMetaType<Location>();

    StyleUtilsSingleton style_utils_singleton;
    (void) style_utils_singleton;

    QString app_dir;
#ifdef DEFAULT_PREFIX
    app_dir = QApplication::applicationDirPath();
#else
    app_dir = _STRINGIZE(DATADIR) "/movie-schedule";
#endif
    //std::cout << "app_dir = " << qPrintable(app_dir) << std::endl;

    QString locale = QLocale::system().name();
    //std::cout << "locale: " << qPrintable(locale) << std::endl;

#if 1
    QTranslator translator;
    translator.load(app_dir + "/translations/movie-schedule_" + locale);
    app.installTranslator(&translator);
#endif

    RatingSingleton rating_singleton;
    (void) rating_singleton;

    CinemaSchedule cinema_schedule;

    MainWindow main_window(&cinema_schedule);
    main_window.show();

    Settings settings;
    SettingsController settings_controller(&settings);
    settings_controller.Load();

    ActionController action_controller(&cinema_schedule);

    ControllerThread sort_worker;
    sort_worker.start();

    ControllerThread search_worker;
    search_worker.start();

    ItemModelSortController sort_controller(&cinema_schedule);
    sort_controller.moveToThread(&sort_worker);

    TheaterWindow theater_window(&cinema_schedule, &main_window);
    TheaterController theater_controller(&theater_window, &cinema_schedule, &action_controller,
                                         &sort_controller, &search_worker);

    MovieWindow movie_window(&cinema_schedule, &main_window);
    MovieController movie_controller(&movie_window, &cinema_schedule, &action_controller,
                                     &sort_controller, &search_worker);

    LocationController location_controller(&main_window, &settings);

    MainController main_controller(&main_window, &settings, &cinema_schedule, &theater_controller,
                                   &movie_controller, &location_controller, &action_controller,
                                   &sort_controller, &search_worker);

    main_controller.Run();

    int rc = app.exec();

    if (rc == 0) {
        settings_controller.Save();
    }

    search_worker.quit();

    sort_worker.quit();

    return rc;
}
