#include <QMaemo5InformationBox>
#include <QStringBuilder>

#include "mouho.h"
#include "networkhandler.h"

Mouho::Mouho()
    : QObject(0)
{
    m_config = new Config(this);
    m_handler = new NetworkHandler(QUrl("http://ztesch.fi/mouh/feed.php"));

    connect(m_handler, SIGNAL(channelsReceived(const ChannelList&)), this, SLOT(setChannels(const ChannelList&)));
    connect(m_handler, SIGNAL(postImageCompleted(QSharedPointer<Image>)), this, SLOT(postCompleted(QSharedPointer<Image>)));

    // Error handlers
    connect(m_handler, SIGNAL(channelListRequestFailed(const QString&)), this, SLOT(channelListFailed(QString)));
    connect(m_handler, SIGNAL(postImageFailed(const QString&)), this, SLOT(postFailed(QString)));

    // Initialize empty channel list, request for real ones
    m_channels = ChannelList(new QList<QSharedPointer<Channel> >());
}

/**
 * Requests the channel list
 */
void Mouho::initChannels()
{
    m_handler->listChannels();
}

/**
 * Sets the list of channels
 */
void Mouho::setChannels(const ChannelList &channels)
{
    m_channels = channels;
    emit channelListReceived(true);
}

void Mouho::channelListFailed(const QString &value)
{
    showBanner("Channel list retrieval failed with error code " % value);
    emit channelListReceived(false);
}

/**
 * Send the image to be posted, wait for result
 */
void Mouho::post(QSharedPointer<Image> image)
{
    emit postStarted();
    ulong datalen = m_handler->xmlTemplateSize() + image->imageData->length() + image->imageSummary.length() + image->postTitle.length();
    showBanner("Posting " % QString::number(datalen/1024) % " kilobytes to channel " % image->channel->name);
    m_handler->postImage(image);
}

/**
 * Post completed!
 */
void Mouho::postCompleted(QSharedPointer<Image> image)
{
    emit postFinished();
    showBanner("Successfully sent " % image->imageTitle % " to " % image->channel->name);
}

/**
 * Handler for network errors
 */
void Mouho::postFailed(const QString &value)
{
    emit postFinished();
    showBanner("Posting failed with error code " % value);
}

/**
 * Show the Maemo5 banner
 */
void Mouho::showBanner(const QString &value) const
{
    QMaemo5InformationBox::information(0, value, QMaemo5InformationBox::DefaultTimeout);
}
