/* 
 * MORTAR
 * 
 * -- screen access functions for SDL library
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms specified in the GNU Public Licence (GPL).
 *
 * Copyright (C) 2001 by Eero Tamminen
 *
 * NOTES
 * - Uses 8-bit buffer like internal one in Mortar and
 *   lets SDL do format conversion for the screen.
 */

#include <time.h>
#include "mortar.h"
#include <SDL/SDL.h>
#include <SDL/SDL_keysym.h>


static SDL_Surface *sdl_screen;


int win_init(int *wd, int *ht, int fullscreen)
{
	int i;
	m_uchar *map;
	unsigned int flags = 0;

	if (SDL_Init(SDL_INIT_VIDEO) != 0) {
		fprintf(stderr,	"SDL error: %s\n", SDL_GetError());
		return 0;
	}

	/* default values */
	if (!*wd || !*ht) {
		*wd = 640;
		*ht = 480;
	}
	if (fullscreen) {
		flags |= SDL_FULLSCREEN;
	}
	
	sdl_screen = SDL_SetVideoMode(*wd, *ht, 8, flags);
	if (sdl_screen == NULL) {
		fprintf(stderr,	"SDL error: %s\n", SDL_GetError());
		return 0;
	}

	if (!screen_init(*wd, *ht, 256)) {
		SDL_Quit();
		return 0;
	}
	
	SDL_WM_SetCaption("SDL-Mortar", "SDL-Mortar");

	/* use direct mapping */
	map = map_get();
	for (i = 0; i < 256; i++) {
		map[i] = i;
	}

	/* set key repeat (for cannon turning) */
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	
	return 1;
}


void win_exit(void)
{
	SDL_Quit();
}


void win_changecolor(int idx, m_rgb_t *rgb)
{
	SDL_Color col;

	col.r = rgb->r;
	col.g = rgb->g;
	col.b = rgb->b;

	SDL_SetColors(sdl_screen, &col, idx, 1);
}


/* set / map window palette.  idx 0 is left for transparency.
 * should be called only once at program start
 */
int win_setpalette(int colors, m_rgb_t *pal)
{
	SDL_Color col[--colors];
	int idx;

	pal++;
	/* covert to SDL structs */
	for (idx = 0; idx < colors; idx++) {
		col[idx].r = pal[idx].r;
		col[idx].g = pal[idx].g;
		col[idx].b = pal[idx].b;
	}
	SDL_SetColors(sdl_screen, col, 1, colors);
	return 1;
}


void win_sync(void)
{
	int rects, x, y, w, h, i, soff, doff;
	m_uchar *src, *dst;
	rect_t *rect;

	rects = screen_rects(&rect);
	snd_flush();

	soff = Screen->wd;
	doff = sdl_screen->pitch;
	while (rects--) {
		x = rect->x1;
		y = rect->y1;
		w = rect->x2 - rect->x1;
		h = rect->y2 - rect->y1;
		src = Screen->data + Screen->wd * y + x;
		dst = sdl_screen->pixels;
		dst += sdl_screen->pitch * y + x;

		/* copy internal buffer to SDL surface using
		 * (gcc inline) memcpy()
		 */
		i = h;
		while (i-- > 0) {
			memcpy(dst, src, w);
			dst += doff;
			src += soff;
		}
		/* blit SDL surface to screen */
		SDL_UpdateRect(sdl_screen, x, y, w, h);
		
		rect++;
	}
}


int win_getkey(long timeout)
{
	struct timeval tval;
	SDL_Event ev;
	char *name;
	int key = 0;

	/* no time */
	tval.tv_sec = timeout / 1000;
	tval.tv_usec = (timeout % 1000) * 1000L;

	win_sync();

	/* wait until frame time has expired */
	select(0, NULL, NULL, NULL, &tval);

	/* handle and empty event queue */
	while (SDL_PollEvent(&ev)) {

		switch (ev.type) {

		case SDL_QUIT:
			return KEY_EXIT;

		case SDL_KEYDOWN:
			switch(ev.key.keysym.sym) {

			/* special keys */
			case SDLK_KP8:
			case SDLK_UP:       return KEY_UP;
			case SDLK_KP2:
			case SDLK_DOWN:     return KEY_DOWN;
			case SDLK_KP4:
			case SDLK_LEFT:     return KEY_LEFT;
			case SDLK_KP6:
			case SDLK_RIGHT:    return KEY_RIGHT;
			case SDLK_RETURN:
			case SDLK_KP_ENTER: return KEY_ACCEPT;
			case SDLK_SPACE:    return KEY_NEXT;
#if MAEMO
			case SDLK_F4:       return KEY_PREV;	/* menu */
			case SDLK_F7:       return KEY_PREV;	/* - */
			case SDLK_ESCAPE:   return KEY_NEXT;
			case SDLK_F8:       return KEY_NEXT;	/* + */
#else
			case SDLK_HOME:     return KEY_PREV;
			case SDLK_PAGEUP:   return KEY_PREV;
			case SDLK_END:      return KEY_NEXT;
			case SDLK_PAGEDOWN: return KEY_NEXT;
			case SDLK_ESCAPE:   return KEY_EXIT;
#endif
#if MAEMO
			case SDLK_F6:  /* fullscreen key */
#endif
			case SDLK_f:
				/* toggle fullscreen */
				SDL_WM_ToggleFullScreen(sdl_screen);
				break;

			case SDLK_s:
				/* snapshot saving */
				name = get_string("snapshot");
				if (name) {
					ppm_write(name, Screen);
				}
				break;
				
			default:
				key = ev.key.keysym.sym;
			}
		}
	}
	
	/* no recognizable key to return */
	return key;
}

