/* 
 * MORTAR
 * 
 * -- double-buffer / 'dirty rectangle' updating
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms specified in the GNU Public Licence (GPL).
 *
 * Copyright (C) 1998-1999,2005,2008 by Eero Tamminen
 * 
 * 
 * Ways to handle updates:
 * - single rectangle is expanded to contain all updates
 *   (may do lot of redundant screen area updates)
 * - all updates have separate rectangles
 *   (can do a lot of overlapping screen updates)
 * - if update overlaps with previous one, it's merged with it,
 *   otherwise it's separate (has lots of iffing which can be
 *   slow if branching is slow on your CPU)
 *
 * TODO
 * - Adding code for Screen offsets into screen_update(), we could have
 *   (scrollable) larger than screen playing areas.
 */

#include <assert.h>
#include "mortar.h"

/* depends on how many updates can happen at the same time,
 * see POINTS in intro.c
 */
#define MAX_RECTS 256

static int Rects;
static rect_t Rect[MAX_RECTS];
static update_t UpdateMethod = SINGLE_RECT;


void screen_set_update_mode(update_t mode)
{
	assert(!Rects);
	fprintf(stderr, "UPDATE MODE: %d\n", mode);
	UpdateMethod = mode;
}


int screen_init(int wd, int ht, int colors)
{
	Screen = bm_alloc(wd, ht, colors);
	if (!Screen) {
		return 0;
	}
	Rects = 0;
	return 1;
}


/* align new update to 16-pixels in horizontal direction and check
 * whether it overlaps the previous update.  If there's overlap, expand
 * the previous rectangle (if neccessary) and return true,
 * otherwise return false
 */
static inline int join(rect_t *r, int x1, int y1, int x2, int y2)
{
	x1 &= ~15;
	x2 = (x2 + 15) & ~15;
	/* if co-ordinates overlap with the previous rectangle, join it */
	if ((r->x1 >= x1 && r->x1 <= x2 && r->y1 >= y1 && r->y1 <= y2) ||
	    (r->x2 >= x1 && r->x2 <= x2 && r->y2 >= y1 && r->y2 <= y2) ||
	    (x1 >= r->x1 && x1 <= r->x2 && y1 >= r->y1 && y1 <= r->y2) ||
	    (x2 >= r->x1 && x2 <= r->x2 && y2 >= r->y1 && y2 <= r->y2)) {
		if (x1 < r->x1) {
			r->x1 = x1;
		}
		if (x2 > r->x2) {
			r->x2 = x2;
		}
		if (y1 < r->y1) {
			r->y1 = y1;
		}
		if (y2 > r->y2) {
			r->y2 = y2;
		}
		return 1;
	}
	return 0;
}


/* set which part of screen was drawn into
 */
void screen_dirty(int x, int y, int w, int h)
{
	w += x;
	h += y;

	/* completely outside screen? */
	if (w < 0 || h < 0 || x >= Screen->wd || y >= Screen->ht) {
		return;
	}

	/* Single dirty rect update */
	if (UpdateMethod == SINGLE_RECT) {
		if (!Rects) {
			Rect->x1 = x;
			Rect->y1 = y;
			Rect->x2 = w;
			Rect->y2 = h;
			Rects = 1;
			return;
		}
		/* will be limited within screen in screen_rects() */
		if (x < Rect->x1) {
			Rect->x1 = x;
		}
		if (y < Rect->y1) {
			Rect->y1 = y;
		}
		if (w > Rect->x2) {
			Rect->x2 = w;
		}
		if (h > Rect->y2) {
			Rect->y2 = h;
		}
		return;
	}

	/* Multiple dirty rects update */

	/* limit inside screen */
	if (x < 0) {
		x = 0;
	}
	if (y < 0) {
		y = 0;
	}
	if (w >= Screen->wd) {
		w = Screen->wd - 1;
	}
	if (h >= Screen->ht) {
		h = Screen->ht - 1;
	}
	if (Rects) {
		if (!join(&(Rect[Rects-1]), x, y, w, h)) {
			Rect[Rects].x1 = x;
			Rect[Rects].y1 = y;
			Rect[Rects].x2 = w;
			Rect[Rects].y2 = h;
			Rects++;
			/* make sure we have enough space for rects */
			assert(Rects < MAX_RECTS);
		}	
	} else {
		Rect->x1 = x & ~15;
		Rect->y1 = y;
		Rect->x2 = (w + 15) & ~15;
		Rect->y2 = h;
		Rects = 1;
	}
}

/* set rect to first rectangle and return number of update rectangles */
int screen_rects(rect_t **rect)
{
	int ret;

	if (UpdateMethod == SINGLE_RECT) {
		static rect_t dirty;
		
		/* limit inside screen */
		dirty = Rect[0];
		dirty.x1 = (dirty.x1 < 0 ? 0 : dirty.x1);
		dirty.y1 = (dirty.y1 < 0 ? 0 : dirty.y1);
		dirty.x2 = (dirty.x2 < Screen->wd ? dirty.x2 : Screen->wd - 1);
		dirty.y2 = (dirty.y2 < Screen->ht ? dirty.y2 : Screen->ht - 1);
		*rect = &dirty;
	} else {
		*rect = Rect;
	}
	ret = Rects;
#ifdef DEBUG
	if (ret) {
		int i;
		for (i = 0; i < ret; i++) {
			fprintf(stderr, "%d: %dx%d @ (%d,%d)\n", i,
				Rect[i].x2 - Rect[i].x1,
				Rect[i].y2 - Rect[i].y1,
				Rect[i].x1, Rect[i].y1
			);
		}
	}
#endif
	Rects = 0;
	return ret;
}


#if 0		/* currrently not needed */

/* if given co-ordinates would need clipping, return true,
 * else (everything completely inside Screen) return false.
 */
int screen_clip(int x, int y, int w, int h)
{
	w += x;
	h += y;

	if (x < 0 || y < 0 || w >= Screen->wd || h >= Screen->ht) {
		return 1;
	}
	return 0;
}

#include <string.h>

/* currently expects 'dest' to be same size as 'Screen'...
 */
void screen_update(m_image_t *dest)
{
	int rects, w, h, line;
	m_uchar *src, *dst;
	rect_t *rect;
	long offset;
 
	rects = screen_rects(&rect);
	
	while (rects--) {
		line = Screen->wd;
		offset = line * rect->y1 + rect->x1;
		src = Screen->data + offset;
		dst = dest->data + offset;
		
		w = rect->x2 - rect->x1;
		h = rect->y2 - rect->y1;
		while (h--) {
			memcpy(dst, src, w);
			dst += line;
			src += line;
		}
		rect++;
	}
}
#endif

/* draw a box with given color */
void screen_box(int x, int y, int wd, int ht, int color)
{
	m_uchar *ptr;
	int swd, off;

	screen_dirty(x, y, wd, ht);

	swd = Screen->wd;
	ptr = Screen->data + y * swd + x;
	off = swd - wd;

	while (ht-- > 0) {
		x = wd;
		while (x-- > 0) {
			*ptr++ = color;
		}
		ptr += off;
	}
}

