/*
 * MORTAR
 *
 * -- emulates required unix functions missing from Amiga
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms specified in the GNU Public Licence (GPL).
 *
 * Copyright (C) 1999 by Frank Wille <frank@phoenix.owl.de>
 *
 * NOTES
 * - implemented function:
 *   + usleep()
 *   + sleep()
 *   + chdir()
 *   + stat()
 * - init_amiga() has to be called by main(), because there are other
 *   functions, which need TimerBase as well, e.g. GetSysTime().
 *   exit_amiga() will be called through atexit(), though.
 */

#include <exec/devices.h>
#include <dos/dos.h>
#include <devices/timer.h>
#include <proto/exec.h>
#include <proto/dos.h>
#include <proto/timer.h>

#include "amiga.h"

struct Library *TimerBase = NULL;  /* timer.device */
static struct timerequest *timerio;
static struct MsgPort *timerport;
static BPTR origlock = 0;  /* original work directory's lock */


/* remove timer etc */
static int exit_amiga(void)
{
  BPTR old;

  if (TimerBase) {
    if (!CheckIO((struct IORequest *)timerio)) {
      AbortIO((struct IORequest *)timerio);
      WaitIO((struct IORequest *)timerio);
    }
    CloseDevice((struct IORequest *)timerio);
    DeleteMsgPort(timerport);
    DeleteIORequest((struct IORequest *)timerio);
  }

  if (origlock) {
    /* restore original work directory */
    if (old = CurrentDir(origlock)) {
      UnLock(old);
    }
  }
  return 1;
}


/*  initialize timer etc */
int init_amiga(void)
{
  /* initialize timer i/o */
  if (timerport = CreateMsgPort()) {
    if (timerio = (struct timerequest *)
                   CreateIORequest(timerport,sizeof(struct timerequest))) {
      if (OpenDevice(TIMERNAME,UNIT_MICROHZ,
                     (struct IORequest *)timerio,0) == 0) {
        TimerBase = (struct Library *)timerio->tr_node.io_Device;
      }
      else {
        DeleteIORequest((struct IORequest *)timerio);
        DeleteMsgPort(timerport);
      }
    }
    else {
      DeleteMsgPort(timerport);
    }
  }
  if (!TimerBase) {
    fprintf(stderr,"Can't open timer.device!\n");
    return 0;
  }
  atexit((void (*))exit_amiga);
  return 1;
}


void usleep(unsigned long timeout)
{
  timerio->tr_node.io_Command = TR_ADDREQUEST;
  timerio->tr_time.tv_secs = timeout / 1000000;
  timerio->tr_time.tv_micro = timeout % 1000000;
  SendIO((struct IORequest *)timerio);
  WaitIO((struct IORequest *)timerio);
}


unsigned int sleep(unsigned int seconds)
{
  Delay(seconds * 50);
  return 0;
}


int stat(char *name,struct stat *st)
{
  static struct FileInfoBlock fib;  /* longword aligned! */
  int rc = -1;
  BPTR lck;

  if (lck = Lock((STRPTR)name,ACCESS_READ)) {
    if (Examine(lck,&fib)) {
      st->st_size = (size_t)fib.fib_Size;
      rc = 0;
    }
    UnLock(lck);
  }
  return rc;
}


int chdir(char *path)
{
  BPTR newlock,oldlock;

  if (newlock = Lock((STRPTR)path,ACCESS_READ)) {
    if (oldlock = CurrentDir(newlock)) {
      if (!origlock) {
        origlock = oldlock;
      } else {
        UnLock(oldlock);
      }
    }
  }
  return 0;
}

