/*
 * monorail.c
 *
 * Copyright (C) 2009-2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib.h>

#include <telepathy-glib/debug.h>

#include <unique/unique.h>

#include <hildon/hildon-fm.h>

#include "approve.h"
#include "contact-chooser.h"
#include "debug.h"
#include "incoming.h"
#include "monorail.h"
#include "transfer.h"

/* Monorail can send files either when asked from its main window or when it
 * receives a list of files to send from its D-Bus Send interface. Any of the
 * those two methods will use a MonorailSendContext to store the context.
 */
typedef struct
{
  enum {
    CONTEXT_MAIN,
    CONTEXT_DBUS
  } context_type;
  GtkWidget *dialog;
  GtkWidget *contact_view;
  GtkWidget *live_search;
  GtkWidget *file_window;
  GFile *selected_file;
  /* list of GFile */
  GSList *selected_files;
  OssoABookContact *selected_master_contact;
  OssoABookContact *selected_contact;
  MonorailData *monorail_data;
} MonorailSendContext;

static gboolean
no_transfers_timeout (gpointer data)
{
  DEBUG ("No remaining transfers; exiting");
  gtk_main_quit ();
  return FALSE;
}

void
monorail_transfers_inc (MonorailData *data,
    gboolean fire_signal)
{
  data->transfers++;

  if (fire_signal)
    monorail_notify_update_num_transfers (data->notify,
        ++data->active_transfers);

  if (!data->no_ui)
    return;

  if (data->quit_timeout_id > 0)
    {
      DEBUG ("Removing quit timeout");
      g_source_remove (data->quit_timeout_id);
      data->quit_timeout_id = 0;
    }
}

void
monorail_transfers_dec (MonorailData *data,
    gboolean fire_signal)
{
  /* So we can call this in main */
  if (data->transfers > 0)
    {
      data->transfers--;

      if (fire_signal)
        monorail_notify_update_num_transfers (data->notify,
            --data->active_transfers);
    }

  if (!data->no_ui)
    return;

  if (data->transfers > 0)
    return;

  DEBUG ("Setting quit timeout");
  data->quit_timeout_id = g_timeout_add_seconds (
      15, no_transfers_timeout, NULL);
}

#define COMMAND_SHOW 1

static UniqueResponse
unique_app_message_cb (UniqueApp *unique_app,
    gint command,
    UniqueMessageData *message_data,
    guint timestamp,
    gpointer user_data)
{
  MonorailData *data = user_data;

  if (command != COMMAND_SHOW)
    return UNIQUE_RESPONSE_PASSTHROUGH;

  DEBUG ("Other instance launched, presenting the main window. "
      "Command=%d, timestamp %u", command, timestamp);

  if (data->quit_timeout_id > 0)
    {
      DEBUG ("Removing quit timeout");
      g_source_remove (data->quit_timeout_id);
      data->quit_timeout_id = 0;
    }

  data->no_ui = FALSE;

  gtk_widget_show (data->contact_window);
  gtk_window_present (GTK_WINDOW (data->contact_window));

  return UNIQUE_RESPONSE_OK;
}

static gboolean
main_window_delete_event (GtkWidget *widget,
    GdkEvent *event,
    MonorailData *data)
{
  if (data->transfers == 0)
    gtk_main_quit ();

  gtk_widget_hide (widget);

  return TRUE;
}

static gboolean
dialog_delete_event (GtkWidget *widget,
    GdkEvent *event,
    MonorailSendContext *data)
{
  monorail_transfers_dec (data->monorail_data, FALSE);

  return FALSE;
}


static void
transfers_button_clicked (GtkButton *button,
    MonorailData *data)
{
  transfer_show_window (data);
}

static HildonAppMenu *
create_menu (MonorailData *data)
{
  HildonAppMenu *menu;
  GtkWidget *button;

  menu = HILDON_APP_MENU (hildon_app_menu_new ());

  button = hildon_gtk_button_new (HILDON_SIZE_AUTO);
  gtk_button_set_label (GTK_BUTTON (button), "Transfers");
  gtk_widget_show (button);

  g_signal_connect (button, "clicked",
      G_CALLBACK (transfers_button_clicked), data);

  hildon_app_menu_append (menu, GTK_BUTTON (button));

  return menu;
}

static void
send_file (MonorailSendContext *data)
{
  TpDBusDaemon *dbus;
  TpAccount *account;
  EmpathyFTFactory *factory;
  McAccount *mcaccount;
  GSList *files;

  if (data->selected_contact == NULL
      || !osso_abook_contact_is_roster_contact (data->selected_contact))
    return;

  mcaccount = osso_abook_contact_get_account (data->selected_contact);

  if (mcaccount == NULL)
    return;

  dbus = tp_dbus_daemon_dup (NULL);
  account = tp_account_new (dbus, tp_proxy_get_object_path (mcaccount), NULL);
  g_object_unref (dbus);

  factory = monorail_handler_get_factory (data->monorail_data->handler);

  DEBUG ("Starting new outgoing transfer to %s",
      osso_abook_contact_get_bound_name (data->selected_contact));

  if (data->context_type == CONTEXT_MAIN)
    {
      empathy_ft_factory_new_transfer_outgoing (factory, account,
          osso_abook_contact_get_bound_name (data->selected_contact),
          data->selected_file,
          monorail_handler_get_name (data->monorail_data->handler));
    }
  else
    {
      files = data->selected_files;
      while (files != NULL)
        {
          empathy_ft_factory_new_transfer_outgoing (factory, account,
              osso_abook_contact_get_bound_name (data->selected_contact),
              files->data,
              monorail_handler_get_name (data->monorail_data->handler));
          files = files->next;
        }
    }

  g_object_unref (account);
}

static void
confirm_dialog_response_cb (GtkDialog *dialog,
    gint response_id,
    MonorailSendContext *data)
{
  if (response_id != GTK_RESPONSE_YES)
    goto out;

  send_file (data);

  if (data->context_type == CONTEXT_MAIN)
    {
      gtk_widget_destroy (data->file_window);
      data->file_window = NULL;
    }
  else
    {
      monorail_transfers_dec (data->monorail_data, FALSE);
      gtk_widget_destroy (data->dialog);
      data->dialog = NULL;

      if (data->selected_master_contact != NULL)
        {
          g_object_unref (data->selected_master_contact);
          data->selected_master_contact = NULL;
        }

      if (data->selected_contact != NULL)
        {
          g_object_unref (data->selected_contact);
          data->selected_contact = NULL;
        }

      if (data->selected_files != NULL)
        {
          g_slist_foreach (data->selected_files, (GFunc) g_object_unref, NULL);
          g_slist_free (data->selected_files);
          data->selected_files = NULL;
        }

      data->monorail_data = NULL;
      g_slice_free (MonorailSendContext, data);
    }

out:
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
ask_to_send_file (MonorailSendContext *data)
{
  GtkWidget *dialog, *label;
  gchar *text;

  dialog = gtk_dialog_new ();
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);

  if (data->context_type == CONTEXT_MAIN)
    {
      gtk_window_set_transient_for (GTK_WINDOW (dialog),
          GTK_WINDOW (data->file_window));
    }

  gtk_widget_show (gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_YES,
          GTK_RESPONSE_YES));
  gtk_widget_show (gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_NO,
          GTK_RESPONSE_NO));

  if (data->context_type == CONTEXT_MAIN)
    {
      gchar *basename;
      basename = g_file_get_basename (data->selected_file);
      text = g_strdup_printf ("Send %s\nto %s?", basename,
          osso_abook_contact_get_display_name (data->selected_master_contact));
      g_free (basename);
      gtk_window_set_title (GTK_WINDOW (dialog), "Send file?");
    } 
  else
   {
      guint count = g_slist_length (data->selected_files);
      if (count == 1)
        {
          gchar *basename;
          basename = g_file_get_basename (data->selected_files->data);
          text = g_strdup_printf ("Send %s\nto %s?", basename,
              osso_abook_contact_get_display_name (data->selected_master_contact));
          g_free (basename);
          gtk_window_set_title (GTK_WINDOW (dialog), "Send file?");
        }
      else
        {
          text = g_strdup_printf ("Send %u files to %s?",
              g_slist_length (data->selected_files),
              osso_abook_contact_get_display_name (data->selected_master_contact));
          gtk_window_set_title (GTK_WINDOW (dialog), "Send files?");
        }
   }

  label = gtk_label_new (text);
  gtk_widget_show (label);
  gtk_box_pack_start (
      GTK_BOX (gtk_dialog_get_content_area (GTK_DIALOG (dialog))), label, TRUE, TRUE, 0);

  g_free (text);

  g_signal_connect (dialog, "response",
      G_CALLBACK (confirm_dialog_response_cb), data);

  gtk_widget_show (dialog);
}

static void
file_activated_cb (HildonFileSelection *self,
    MonorailSendContext *data)
{
  GSList *selection;

  selection = hildon_file_selection_get_selected_uris (self);

  if (data->selected_file != NULL)
    {
      g_object_unref (data->selected_file);
      data->selected_file = NULL;
    }

  /* should only be one file selected */
  data->selected_file = g_file_new_for_uri ((gchar *) selection->data);

  g_slist_foreach (selection, (GFunc) g_free, NULL);
  g_slist_free (selection);

  ask_to_send_file(data);
}

static void
show_file_manager (MonorailSendContext *data)
{
  GtkWidget *widget;
  HildonFileSystemModel *model;

  /* Create new window */
  data->file_window = hildon_stackable_window_new ();
  hildon_window_set_app_menu (HILDON_WINDOW (data->file_window),
      create_menu (data->monorail_data));
  gtk_window_set_title (GTK_WINDOW (data->file_window), "Select a file");

  model = g_object_new (HILDON_TYPE_FILE_SYSTEM_MODEL,
      "ref-widget", data->file_window,
      NULL);
  widget = hildon_file_selection_new_with_model (model);
  gtk_container_add (GTK_CONTAINER (data->file_window), widget);
  gtk_widget_show (widget);

  if (data->selected_file != NULL)
    {
      gchar *uri;
      GFile *parent;

      parent = g_file_get_parent (data->selected_file);
      uri = g_file_get_uri (parent);

      hildon_file_selection_set_current_folder_uri (
          HILDON_FILE_SELECTION (widget), uri, NULL);

      g_free (uri);
      g_object_unref (parent);
    }
  else
    {
      hildon_file_selection_set_current_folder_uri (
          HILDON_FILE_SELECTION (widget),
          "file:///home/user/MyDocs/.documents/", NULL);
    }

  g_signal_connect (widget, "file-activated",
      G_CALLBACK (file_activated_cb), data);

  gtk_widget_show (data->file_window);

  g_object_unref (model);
}

/* We're using an idle to do this because if it was hidden in a
 * ::contact-activated callback, then the tree path would be invalidated
 * and so would the OssoABookContact.
 */
static gboolean
hide_live_search (gpointer user_data)
{
  MonorailSendContext *data = user_data;

  gtk_widget_hide (data->live_search);

  return FALSE;
}

typedef struct
{
  MonorailSendContext *data;
  GtkWidget *dialog;
  GHashTable *button_to_contact;
  OssoABookContact *master_contact;
} RosterContactChooserData;

static void
roster_contact_chooser_response_cb (GtkDialog *dialog,
    gint response_id,
    RosterContactChooserData *chooser_data)
{
  g_hash_table_destroy (chooser_data->button_to_contact);
  g_object_unref (chooser_data->master_contact);
  g_slice_free (RosterContactChooserData, chooser_data);
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
roster_contact_button_clicked_cb (GtkButton *button,
    RosterContactChooserData *chooser_data)
{
  MonorailSendContext *data = chooser_data->data;
  OssoABookContact *contact;

  contact = g_hash_table_lookup (chooser_data->button_to_contact, button);

  if (contact == NULL)
    return;

  if (data->selected_contact != NULL)
    g_object_unref (data->selected_contact);

  data->selected_contact = g_object_ref (contact);

  if (data->selected_master_contact != NULL)
    g_object_unref (data->selected_master_contact);

  data->selected_master_contact = g_object_ref (chooser_data->master_contact);

  g_object_unref (chooser_data->master_contact);
  g_hash_table_destroy (chooser_data->button_to_contact);
  gtk_widget_destroy (GTK_WIDGET (chooser_data->dialog));
  g_slice_free (RosterContactChooserData, chooser_data);

  g_idle_add (hide_live_search, data);

  if (data->context_type == CONTEXT_DBUS)
    ask_to_send_file (data);
  else
    show_file_manager (data);
}

static void
show_roster_contact_chooser (OssoABookContact *contact,
    MonorailSendContext *data)
{
  GtkWidget *dialog;
  GList *roster_contacts, *l;
  RosterContactChooserData *chooser_data;

  dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), "Choose an account");

  chooser_data = g_slice_new0 (RosterContactChooserData);
  chooser_data->data = data;
  chooser_data->dialog = dialog;
  chooser_data->master_contact = g_object_ref (contact);

  chooser_data->button_to_contact = g_hash_table_new_full (g_direct_hash,
      g_direct_equal, NULL, (GDestroyNotify) g_object_unref);

  g_signal_connect (dialog, "response",
      G_CALLBACK (roster_contact_chooser_response_cb), chooser_data);

  roster_contacts = osso_abook_contact_get_roster_contacts (contact);

  for (l = roster_contacts; l != NULL; l = l->next)
    {
      OssoABookContact *c = l->data;
      GtkWidget *button;
      McAccount *account;
      McProfile *profile;

      if (!monorail_contact_chooser_can_send_to_roster_contact (
              MONORAIL_CONTACT_CHOOSER (data->contact_view), c))
        continue;

      account = osso_abook_contact_get_account (c);
      profile = mc_profile_lookup (mc_account_compat_get_profile (account));

      button = osso_abook_button_new_with_presence (HILDON_SIZE_FINGER_HEIGHT,
          mc_profile_get_icon_name (profile),
          mc_profile_get_display_name (profile),
          osso_abook_contact_get_bound_name (c),
          OSSO_ABOOK_PRESENCE (c));

      g_hash_table_insert (chooser_data->button_to_contact, button,
          g_object_ref (c));

      g_signal_connect (button, "clicked",
          G_CALLBACK (roster_contact_button_clicked_cb), chooser_data);

      gtk_box_pack_start (
          GTK_BOX (gtk_dialog_get_content_area (GTK_DIALOG (dialog))),
          button, TRUE, TRUE, 0);
      gtk_widget_show (button);

      g_object_unref (profile);
    }

  g_list_free (roster_contacts);

  gtk_widget_show (dialog);
}

static void
contact_activated_cb (OssoABookContactView *view,
    GtkTreePath *path,
    MonorailSendContext *data)
{
  GList *selection;
  OssoABookContact *contact, *roster_contact = NULL;
  GList *roster, *l;
  guint count = 0;

  selection = osso_abook_contact_view_get_selection (view);

  /* Only one contact is allowed to be selected at one time. */
  contact = selection->data;

  roster = osso_abook_contact_get_roster_contacts (contact);

  for (l = roster; l != NULL; l = l->next)
    {
      if (monorail_contact_chooser_can_send_to_roster_contact (
              MONORAIL_CONTACT_CHOOSER (data->contact_view),
              OSSO_ABOOK_CONTACT (l->data)))
        {
          count++;
          roster_contact = l->data;
        }
    }

  if (count == 1)
    {
      if (data->selected_contact != NULL)
        g_object_unref (data->selected_contact);

      if (data->selected_master_contact != NULL)
        g_object_unref (data->selected_master_contact);

      data->selected_contact = g_object_ref (roster_contact);
      data->selected_master_contact = g_object_ref (contact);
      g_idle_add (hide_live_search, data);

      if (data->context_type == CONTEXT_DBUS)
        ask_to_send_file (data);
      else
        show_file_manager (data);
    }
  else
    {
      show_roster_contact_chooser (contact, data);
    }

  g_list_free (selection);
}

static void
notify_show_transfers (MonorailNotify *notify,
    MonorailData *data)
{
  transfer_show_window (data);
}

static void
contact_view_done_cb (MonorailContactChooser *chooser,
    GtkWindow *window)
{
  hildon_gtk_window_set_progress_indicator (
      GTK_WINDOW (window), 0);
}

static void
contact_view_working_cb (MonorailContactChooser *chooser,
    GtkWindow *window)
{
  hildon_gtk_window_set_progress_indicator (
      GTK_WINDOW (window), 1);
}

/* Send a list of files as requested from the D-Bus Send interface */
static void
send_files_cb (MonorailSend *monorail_send,
    char **filenames,
    MonorailData *data)
{
  GtkWidget *vbox;
  GtkTreeView *tree_view;

  MonorailSendContext *dialog_context;

  /* Disable the timeout to void quitting Monorail while the dialog is waiting
   * for the user */
  monorail_transfers_inc (data, FALSE);

  dialog_context = g_slice_new0 (MonorailSendContext);
  dialog_context->context_type = CONTEXT_DBUS;
  dialog_context->monorail_data = data;
  dialog_context->selected_contact = NULL;

  dialog_context->selected_files = NULL;
  while (*filenames != NULL)
    {
      GFile *f;

      /* Applications using sharing-dialog use both format and we should
       * accept both. */
      if (g_str_has_prefix (*filenames, "file:"))
        f = g_file_new_for_uri (*filenames);
      else
        f = g_file_new_for_path (*filenames);

      dialog_context->selected_files =
        g_slist_prepend (dialog_context->selected_files, f);
      filenames++;
    }

  dialog_context->dialog = gtk_dialog_new ();
  hildon_gtk_window_set_progress_indicator (
      GTK_WINDOW (dialog_context->dialog), 1);

  vbox = GTK_DIALOG (dialog_context->dialog)->vbox;
  gtk_window_set_title (GTK_WINDOW (dialog_context->dialog), "Select a contact");
  g_signal_connect (dialog_context->dialog, "delete-event",
      G_CALLBACK (dialog_delete_event), dialog_context);

  dialog_context->contact_view = monorail_contact_chooser_new ();
  gtk_box_pack_start (GTK_BOX (vbox), dialog_context->contact_view, TRUE, TRUE, 0);
  gtk_widget_show (dialog_context->contact_view);

  g_signal_connect (dialog_context->contact_view, "done",
      G_CALLBACK (contact_view_done_cb), dialog_context->dialog);
  g_signal_connect (dialog_context->contact_view, "working",
      G_CALLBACK (contact_view_working_cb), dialog_context->dialog);
  g_signal_connect (dialog_context->contact_view, "contact-activated",
      G_CALLBACK (contact_activated_cb), dialog_context);

  tree_view = osso_abook_tree_view_get_tree_view (
      OSSO_ABOOK_TREE_VIEW (dialog_context->contact_view));

  dialog_context->live_search = osso_abook_live_search_new_with_filter (
      osso_abook_tree_view_get_filter_model (
          OSSO_ABOOK_TREE_VIEW (dialog_context->contact_view)));
  hildon_live_search_widget_hook (HILDON_LIVE_SEARCH (dialog_context->live_search),
      dialog_context->dialog, GTK_WIDGET (tree_view));
  gtk_box_pack_start (GTK_BOX (vbox), dialog_context->live_search, FALSE, FALSE, 0);

  gtk_widget_show_all (dialog_context->dialog);
  gtk_widget_hide (dialog_context->live_search);
}

int
main (int argc,
    char **argv)
{
  MonorailData *data;
  MonorailSendContext *main_context;
  GtkWidget *vbox;
  gboolean no_ui = FALSE;
  UniqueApp *unique_app;
  GtkTreeView *tree_view;
  GOptionEntry entries[] = {
    {"no-ui", 'n', 0, G_OPTION_ARG_NONE, &no_ui,
      "Start without a file sending UI", NULL
    },
    { NULL },
  };

  gtk_init_with_args (&argc, &argv, " - File Transfer User Interface",
      entries, NULL, NULL);
  hildon_init ();
  g_thread_init (NULL);

  tp_debug_set_flags (g_getenv ("TP_GLIB_DEBUG"));

  unique_app = unique_app_new_with_commands ("org.maemo.Monorail", NULL,
      "show", COMMAND_SHOW, NULL);

  if (unique_app_is_running (unique_app))
    {
      unique_app_send_message (unique_app, COMMAND_SHOW, NULL);
      g_object_unref (unique_app);
      return EXIT_SUCCESS;
    }

  data = g_slice_new0 (MonorailData);
  data->transfers = 0;
  data->active_transfers = 0;
  data->quit_timeout_id = 0;
  data->no_ui = no_ui;

  main_context = g_slice_new0 (MonorailSendContext);
  main_context->context_type = CONTEXT_MAIN;
  main_context->monorail_data = data;
  main_context->selected_file = NULL;
  main_context->selected_files = NULL;
  main_context->selected_contact = NULL;

  data->contact_window = hildon_stackable_window_new ();
  hildon_gtk_window_set_progress_indicator (
      GTK_WINDOW (data->contact_window), 1);
  hildon_window_set_app_menu (HILDON_WINDOW (data->contact_window),
      create_menu (data));
  gtk_window_set_title (GTK_WINDOW (data->contact_window), "Select a contact");
  gtk_container_set_border_width (GTK_CONTAINER (data->contact_window), 5);

  g_signal_connect (data->contact_window, "delete-event",
      G_CALLBACK (main_window_delete_event), data);

  data->handler = monorail_handler_new (MONORAIL_CLIENT_NAME, TRUE);
  g_signal_connect (data->handler, "new-ft-handler",
      G_CALLBACK (transfer_new_handler), data);
  g_signal_connect (data->handler, "approve-ft-channel",
      G_CALLBACK (approve_channel), data);
  g_signal_connect (data->handler, "incoming-ft",
      G_CALLBACK (incoming_transfer), data);

  g_signal_connect (unique_app, "message-received",
      G_CALLBACK (unique_app_message_cb), data);

  data->notify = monorail_notify_new ();

  g_signal_connect (data->notify, "show-transfers",
      G_CALLBACK (notify_show_transfers), data);

  data->send = monorail_send_new ();

  g_signal_connect (data->send, "send-files",
      G_CALLBACK (send_files_cb), data);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 0);

  main_context->contact_view = monorail_contact_chooser_new ();
  gtk_box_pack_start (GTK_BOX (vbox), main_context->contact_view, TRUE, TRUE, 0);
  gtk_widget_show (main_context->contact_view);

  g_signal_connect (main_context->contact_view, "done",
      G_CALLBACK (contact_view_done_cb), data->contact_window);
  g_signal_connect (main_context->contact_view, "working",
      G_CALLBACK (contact_view_working_cb), data->contact_window);
  g_signal_connect (main_context->contact_view, "contact-activated",
      G_CALLBACK (contact_activated_cb), main_context);

  tree_view = osso_abook_tree_view_get_tree_view (
      OSSO_ABOOK_TREE_VIEW (main_context->contact_view));

  main_context->live_search = osso_abook_live_search_new_with_filter (
      osso_abook_tree_view_get_filter_model (
          OSSO_ABOOK_TREE_VIEW (main_context->contact_view)));
  hildon_live_search_widget_hook (
      HILDON_LIVE_SEARCH (main_context->live_search),
      data->contact_window, GTK_WIDGET (tree_view));
  gtk_box_pack_start (GTK_BOX (vbox), main_context->live_search, FALSE, FALSE,
      0);
  gtk_widget_hide (main_context->live_search);

  gtk_container_add (GTK_CONTAINER (data->contact_window), vbox);
  gtk_widget_show (vbox);

  if (!no_ui)
    gtk_widget_show (data->contact_window);
  else
    monorail_transfers_dec (data, FALSE);

  gtk_main ();

  if (!data->no_ui)
    gtk_widget_destroy (data->contact_window);
  g_object_unref (data->handler);

  if (main_context->selected_master_contact != NULL)
    g_object_unref (main_context->selected_master_contact);

  if (main_context->selected_contact != NULL)
    g_object_unref (main_context->selected_contact);

  if (main_context->selected_file != NULL)
    g_object_unref (main_context->selected_file);

  if (data->notify != NULL)
    g_object_unref (data->notify);

  if (data->send != NULL)
    g_object_unref (data->send);

  if (data->transfers_data != NULL)
    transfers_data_free ((TransfersData *) data->transfers_data);

  g_object_unref (unique_app);

  g_slice_free (MonorailData, data);
  g_slice_free (MonorailSendContext, main_context);

  return 0;
}
