/*
 * approve.c
 *
 * Copyright (C) 2009 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <telepathy-glib/defs.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/util.h>

#include "approve.h"
#include "debug.h"

static void
approve_response (GtkDialog *dialog,
    gint response_id,
    ApproveData *data)
{
  gtk_widget_destroy (GTK_WIDGET (dialog));

  if (response_id == GTK_RESPONSE_YES)
    {
      gchar *handler;

      handler = g_strconcat (TP_CLIENT_BUS_NAME_BASE,
          monorail_handler_get_name (data->monorail_data->handler),
          NULL);

      tp_cli_channel_dispatch_operation_call_handle_with (data->operation, -1,
          handler, NULL, NULL, NULL, NULL);

      g_free (handler);
    }
  else
    {
      tp_cli_channel_call_close (data->channel, -1, NULL, NULL, NULL, NULL);
    }

  monorail_transfers_dec (data->monorail_data, FALSE);

  if (data->abook_wait_closure != NULL)
    {
      osso_abook_waitable_cancel (OSSO_ABOOK_WAITABLE (data->abook),
          data->abook_wait_closure);
    }

  g_free (data->account_object_path);
  g_free (data->target_id);
  g_free (data->size_str);
  g_free (data->filename);
  g_object_unref (data->operation);
  g_object_unref (data->channel);
  g_object_unref (data->abook);
  g_slice_free (ApproveData, data);
}

static gboolean
contact_matches_incoming (OssoABookContact *contact,
    gpointer user_data)
{
  gboolean ret = FALSE;
  GList *roster, *l;
  ApproveData *data = user_data;

  if (!osso_abook_contact_has_roster_contacts (contact))
    return FALSE;

  roster = osso_abook_contact_get_roster_contacts (contact);

  for (l = roster; l != NULL; l = l->next)
    {
      OssoABookContact *c = l->data;
      McAccount *acc;

      acc = osso_abook_contact_get_account (c);

      if (acc == NULL)
        continue;

      if (!tp_strdiff (data->account_object_path,
              tp_proxy_get_object_path (acc))
          && !tp_strdiff (data->target_id,
              osso_abook_contact_get_bound_name (c)))
        {
          ret = TRUE;
          break;
        }
    }

  g_list_free (roster);

  return ret;
}

static void
abook_ready (OssoABookWaitable *wait,
    const GError *error,
    ApproveData *data)
{
  OssoABookRoster *abook_roster = OSSO_ABOOK_ROSTER (wait);
  GList *contacts;
  OssoABookContact *contact = NULL;
  gchar *text;

  data->abook_wait_closure = NULL;

  contacts = osso_abook_aggregator_find_contacts_full (
      OSSO_ABOOK_AGGREGATOR (abook_roster),
      contact_matches_incoming, data);

  contact = contacts->data;

  text = g_strdup_printf ("Accept '%s' (%s)\nfrom %s?",
      data->filename, data->size_str,
      contact != NULL ? osso_abook_contact_get_display_name (contact) :
      data->target_id);

  gtk_label_set_text (GTK_LABEL (data->label), text);

  g_list_free (contacts);
  g_free (text);

  /* Show the dialog only when the text is ready to avoid a glitch in the UI */
  gtk_widget_show (data->dialog);
}

void
approve_channel (MonorailHandler *handler,
    TpChannel **channels,
    TpChannelDispatchOperation *operation,
    GHashTable *cdo_properties,
    MonorailData *monorail_data)
{
  GtkWidget *dialog, *content;
  TpChannel **ptr, *channel = NULL;
  GHashTable *properties;
  ApproveData *data;
  guint64 filesize;

  for (ptr = channels; *ptr != NULL; ptr++)
    {
      if (!tp_strdiff (tp_channel_get_channel_type (TP_CHANNEL (*ptr)),
              TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER))
        {
          channel = *ptr;
          break;
        }
    }

  if (channel == NULL)
    {
      DEBUG ("No file transfer channel in the channel list");
      return;
    }

  monorail_transfers_inc (monorail_data, FALSE);

  data = g_slice_new0 (ApproveData);
  data->monorail_data = monorail_data;
  data->operation = g_object_ref (operation);
  data->channel = g_object_ref (channel);

  g_object_get (channel, "channel-properties", &properties, NULL);

  dialog = gtk_dialog_new_with_buttons ("Incoming File Transfer",
      GTK_WINDOW (monorail_data->contact_window),
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
      GTK_STOCK_YES,
      GTK_RESPONSE_YES,
      GTK_STOCK_NO,
      GTK_RESPONSE_NO,
      NULL);
  data->dialog = dialog;

  content = gtk_dialog_get_content_area (GTK_DIALOG (dialog));

  filesize = tp_asv_get_uint64 (properties,
      TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER ".Size", NULL);

  data->filename = g_strdup (tp_asv_get_string (properties,
          TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER ".Filename"));
  data->account_object_path = g_strdup (tp_asv_get_boxed (cdo_properties,
          TP_IFACE_CHANNEL_DISPATCH_OPERATION ".Account",
          DBUS_TYPE_G_OBJECT_PATH));
  data->target_id = g_strdup (tp_asv_get_string (properties,
          TP_IFACE_CHANNEL ".TargetID"));

  data->size_str = g_format_size_for_display (filesize);

  data->label = gtk_label_new (NULL);

  data->abook = g_object_ref (osso_abook_aggregator_get_default (NULL));
  data->abook_wait_closure = osso_abook_waitable_call_when_ready (
      OSSO_ABOOK_WAITABLE (data->abook),
      (OssoABookWaitableCallback) abook_ready, data, NULL);

  gtk_widget_show (data->label);

  gtk_container_add (GTK_CONTAINER (content), data->label);

  gtk_window_set_transient_for (GTK_WINDOW (dialog),
      GTK_WINDOW (monorail_data->contact_window));

  g_signal_connect (dialog, "response", G_CALLBACK (approve_response), data);
}

