/*
 * Copyright (C) 2010 Nokia Corporation
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *   @author Alban Crequy <alban.crequy@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libintl.h>
#include <libosso-abook/osso-abook.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-gtype-specialized.h>
#include <dbus/dbus-glib-lowlevel.h>

static void (*old_table_constructed) (GObject *object) = NULL;

static DBusGConnection *connection = NULL;

typedef struct {
  OssoABookMenuExtension parent_instance;
} FunsharingExtension;

typedef struct {
  OssoABookMenuExtensionClass parent_class;
} FunsharingExtensionClass;

G_GNUC_INTERNAL GType
funsharing_menu_extension_get_type (void) G_GNUC_CONST;

/* Defines the new plugin object */
OSSO_ABOOK_DEFINE_MENU_PLUGIN
        (FunsharingExtension,
         funsharing_menu_extension,
         OSSO_ABOOK_TYPE_MENU_EXTENSION,,);


static void
share_file_via_im (GSList *uris)
{
  DBusMessage *message;
  DBusMessageIter iter;
  DBusMessageIter sub_iter;
  int ret;

  message = dbus_message_new_method_call ("org.maemo.Monorail",
      "/org/maemo/Monorail/Send", "org.maemo.Monorail.Send", "SendMultiple");
  dbus_message_iter_init_append (message, &iter);
  dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "s", &sub_iter);

  while (uris != NULL)
    {
      printf ("Adding file: '%s'\n", (gchar *) uris->data);
      dbus_message_iter_append_basic (&sub_iter, DBUS_TYPE_STRING, &uris->data);
      uris = uris->next;
    }
  dbus_message_iter_close_container (&iter, &sub_iter);

  ret = dbus_connection_send
    (dbus_g_connection_get_connection (connection), message, NULL);

  dbus_message_unref (message);
}

static void
telepathy_write_contact_cb (OssoABookContact *contact G_GNUC_UNUSED,
                            GFile            *file,
                            const GError     *error,
                            gpointer          user_data G_GNUC_UNUSED)
{
  gchar *uri;
  GSList *files_list;

  if (error) {
    osso_abook_handle_gerror (NULL, g_error_copy (error));
    return;
  }

  uri = g_file_get_uri (file);
  files_list = g_slist_prepend (NULL, uri);

  share_file_via_im (files_list);

  g_slist_free (files_list);
  g_free (uri);
}

static GFile *
get_tmp_file (void)
{
  const char *tmpdir;
  GFile *file;
  char *path;

  tmpdir = g_get_tmp_dir ();

  if (-1 == g_mkdir_with_parents (tmpdir, 01777))
  {
    g_warning ("%s: Cannot create directory %s: %s",
             G_STRLOC, tmpdir, g_strerror (errno));

    return NULL;
  }

  path = g_build_filename (tmpdir, "abook-funsharing-extension-XXXXXX", NULL);
  if (!mkdtemp (path))
    {
      g_warning ("%s: Cannot create directory %s: %s",
          G_STRLOC, path, g_strerror (errno));
      g_free (path);
      return NULL;
    }

  file = g_file_new_for_path (path);
  g_free (path);

  return file;
}

static gboolean
send_avatar_confirmation_show (OssoABookSendContactsDialog *self,
    OssoABookContact *contact)
{
  GtkWidget *note;
  gint response;
  GdkPixbuf *avatar;

  avatar = osso_abook_avatar_get_image (OSSO_ABOOK_AVATAR (contact));
  if (avatar == NULL)
    return FALSE;

  note = hildon_note_new_confirmation (GTK_WINDOW (self),
      dgettext ("osso-addressbook", "addr_nc_send_avatar"));

  response = gtk_dialog_run (GTK_DIALOG (note));
  gtk_widget_destroy (note);

  return response == GTK_RESPONSE_OK;
}

static void
share_via_im_button_clicked (HildonButton *button G_GNUC_UNUSED,
    gpointer user_data)
{
  OssoABookSendContactsDialog *dialog =
    OSSO_ABOOK_SEND_CONTACTS_DIALOG (user_data);
  OssoABookContact *contact;
  gboolean send_avatar;
  GFile *vcard_file;

  g_object_get (dialog, "contact", &contact,
      NULL);
  g_return_if_fail (OSSO_ABOOK_IS_CONTACT (contact));

  send_avatar = send_avatar_confirmation_show (dialog, contact);

  vcard_file = get_tmp_file ();
  if (vcard_file)
    {
      osso_abook_contact_write_to_file (contact, EVC_FORMAT_VCARD_30,
          send_avatar, TRUE, vcard_file, telepathy_write_contact_cb,
          NULL);
      g_object_unref (vcard_file);
    }

  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
}

static void
new_osso_abook_send_contacts_dialog_constructed (GObject *object)
{
  GtkWidget *vbox = GTK_DIALOG (object)->vbox;
  GtkWidget *button;
  GtkWidget *table = NULL;
  GList *list, *cur;

  /* The table and its buttons are created in old_table_constructed(). If
   * old_table_constructed() is NULL, we cannot do anything */
  if (old_table_constructed == NULL)
    {
      g_warning ("%s: No constructor function for "
          "OssoABookSendContactsDialog",
          G_STRLOC);
      return;
    }

  old_table_constructed (object);

  list = cur = gtk_container_get_children (GTK_CONTAINER (vbox));
  while (cur)
    {
      if (GTK_IS_TABLE (cur->data))
        {
          table = cur->data;
          break;
        }
      cur = cur->next;
    }
  g_list_free (list);
  if (!table)
    {
      g_warning ("%s: OssoABookSendContactsDialog does not have a GtkTable",
          G_STRLOC);
      return;
    }

  button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
   HILDON_BUTTON_ARRANGEMENT_VERTICAL,
   "IM",
   NULL);

  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 1, 2,
      GTK_FILL|GTK_EXPAND,
      GTK_FILL|GTK_EXPAND,
      0, 0);
  gtk_widget_show (button);

  g_signal_connect (button, "clicked",
      G_CALLBACK (share_via_im_button_clicked), object);
}

static void
funsharing_menu_extension_init (FunsharingExtension *ext G_GNUC_UNUSED)
{
  GObjectClass *class;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);

  class = g_type_class_ref(osso_abook_send_contacts_dialog_get_type ());
  old_table_constructed = class->constructed;
  class->constructed = new_osso_abook_send_contacts_dialog_constructed;
}

static OssoABookMenuEntry entries[] = {
};

static const OssoABookMenuEntry *
get_menu_entries (OssoABookMenuExtension *menu_extension G_GNUC_UNUSED)
{
  /* We cannot return NULL, otherwise libosso-abook will stop checking the
   * other plugins. Yes, it is a bug in libosso-abook. */
  return entries;
}

static int
get_n_menu_entries (OssoABookMenuExtension *menu_extension G_GNUC_UNUSED)
{
  return G_N_ELEMENTS (entries);
}

static void
funsharing_menu_extension_class_init (FunsharingExtensionClass *cls)
{
  OssoABookMenuExtensionClass *menu_extension_class;

  menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
  menu_extension_class->get_n_menu_entries = get_n_menu_entries;
  menu_extension_class->get_menu_entries = get_menu_entries;
  /* Which menus should be extended with our buttons, separated by a ";" */
  menu_extension_class->name = OSSO_ABOOK_MENU_NAME_MAIN_VIEW;
}

static void
funsharing_menu_extension_class_finalize (FunsharingExtensionClass *cls
    G_GNUC_UNUSED)
{
}

