/*
 * transfer.c
 *
 * Copyright (C) 2009-2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "transfer.h"

#include <hildon-mime.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-gtype-specialized.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <hildon/hildon.h>

#include "debug.h"

enum
{
  COL_PROGRESS_PERCENT = 0,
  COL_TRANSFER_STRING,
  COL_REMAINING_STRING,
  COL_SIGNAL_PROGRESS_ID,
  COL_SIGNAL_DONE_ID,
  COL_SIGNAL_ERROR_ID,
  COL_FT_HANDLER,
  COL_GFILE,
  COL_STATUS,
  NUM_COLS
};

enum
{
  STATUS_PROGRESS = 0,
  STATUS_DONE,
  STATUS_CANCELED,
  STATUS_ERROR
};

typedef struct
{
  EmpathyFTHandler *ft_handler;
  GtkTreeIter iter;
  gboolean set;
} FindIterData;

static gboolean
find_iter_foreach_func (GtkTreeModel *model,
    GtkTreePath *path,
    GtkTreeIter *iter,
    gpointer user_data)
{
  FindIterData *data = user_data;
  EmpathyFTHandler *handler;

  gtk_tree_model_get (model, iter,
      COL_FT_HANDLER, &handler,
      -1);

  if (handler == data->ft_handler)
    {
      data->iter = *iter;
      data->set = TRUE;
    }

  g_object_unref (handler);

  return data->set;
}

static gboolean
find_iter (GtkListStore *store,
    EmpathyFTHandler *ft_handler,
    GtkTreeIter *iter)
{
  FindIterData *data;

  data = g_slice_new0 (FindIterData);
  data->ft_handler = ft_handler;
  data->set = FALSE;

  gtk_tree_model_foreach (GTK_TREE_MODEL (store),
      find_iter_foreach_func, data);

  if (data->set)
    *iter = data->iter;

  g_slice_free (FindIterData, data);

  return iter != NULL;
}

void
transfers_data_free (TransfersData *data)
{
  g_object_unref (data->pannable_area);
  g_object_unref (data->no_transfers_label);
  gtk_widget_destroy (data->dialog);
  g_slice_free (TransfersData, data);
}

static void
show_or_hide_view (TransfersData *data)
{
  if (data->num_transfers == 0)
    {
      data->view_visible = FALSE;
      gtk_widget_hide (data->pannable_area);
      gtk_widget_show (data->no_transfers_label);
    }
  else
    {
      data->view_visible = TRUE;
      gtk_widget_hide (data->no_transfers_label);
      gtk_widget_show (data->pannable_area);
    }
}

static void
cancel_clicked_cb (GtkWidget *b G_GNUC_UNUSED,
    TransfersData *data)
{
  GtkTreeSelection *selection;
  gboolean selected;
  GtkTreeIter iter;
  EmpathyFTHandler *handler;
  gulong done, error, progress;
  gchar *text, *basename;
  guint status;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (data->view));
  selected = gtk_tree_selection_get_selected (selection, NULL, &iter);
  if (!selected)
    {
      hildon_banner_show_information (data->view, NULL, "No transfer selected");
      return;
    }
  gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
      COL_FT_HANDLER, &handler,
      COL_SIGNAL_PROGRESS_ID, &progress,
      COL_SIGNAL_DONE_ID, &done,
      COL_SIGNAL_ERROR_ID, &error,
      COL_STATUS, &status,
      -1);

  if (status != STATUS_PROGRESS)
    {
      hildon_banner_show_information (data->view, NULL,
          "Selected transfer is not in progress");
      return;
    }

  g_signal_handler_disconnect (handler, progress);
  g_signal_handler_disconnect (handler, done);
  g_signal_handler_disconnect (handler, error);

  basename = g_file_get_basename (empathy_ft_handler_get_gfile (handler));

  text = g_strdup_printf ("Error %s %s\nYou cancelled the transfer",
      empathy_ft_handler_is_incoming (handler) ? "receiving" : "sending",
      basename);

  gtk_list_store_set (data->store, &iter,
      COL_TRANSFER_STRING, text,
      COL_REMAINING_STRING, "",
      COL_SIGNAL_PROGRESS_ID, 0,
      COL_SIGNAL_DONE_ID, 0,
      COL_SIGNAL_ERROR_ID, 0,
      COL_FT_HANDLER, NULL,
      COL_STATUS, STATUS_CANCELED,
      -1);

  g_free (text);
  g_free (basename);

  empathy_ft_handler_cancel_transfer (handler);

  g_object_unref (handler);

  monorail_transfers_dec (data->monorail_data, TRUE);
}

static void
open_clicked_cb (GtkWidget *button G_GNUC_UNUSED,
    TransfersData *data)
{
  GtkTreeSelection *selection;
  gboolean selected;
  GtkTreeIter iter;
  GFile *file;
  gchar *uri;
  GSList uri_list;
  DBusGConnection *connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
  guint status;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (data->view));
  selected = gtk_tree_selection_get_selected (selection, NULL, &iter);
  if (!selected)
    {
      hildon_banner_show_information (data->view, NULL, "No transfer selected");
      return;
    }

  gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
      COL_GFILE, &file,
      COL_STATUS, &status,
      -1);

  if (status != STATUS_DONE)
    {
      hildon_banner_show_information (data->view, NULL,
          "Selected transfer is not finished");
      return;
    }

  gtk_widget_hide (data->dialog);
  data->dialog_visible = FALSE;
  monorail_transfers_dec (data->monorail_data, FALSE);

  uri = g_file_get_uri (file);
  uri_list.data = uri;
  uri_list.next = NULL;
  hildon_mime_open_file_list (
    dbus_g_connection_get_connection (connection),
    &uri_list);
  g_free (uri);
}

static void
clear_clicked_cb (GtkWidget *button G_GNUC_UNUSED,
    TransfersData *data)
{
  GtkTreeIter iter;
  gboolean valid;

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (data->store), &iter);
  while (valid)
    {
      guint status;
      gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
          COL_STATUS, &status,
          -1);

      if (status != STATUS_PROGRESS)
        {
          valid = gtk_list_store_remove (data->store, &iter);
          data->num_transfers--;
          monorail_transfers_dec (data->monorail_data, FALSE);
        }
      else
        {
          valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (data->store),
              &iter);
        }
    }
  show_or_hide_view (data);
}

static GtkWidget *
create_tree_view (TransfersData *data)
{
  GtkWidget *view;
  GtkTreeSelection *selection;
  GtkCellRenderer *renderer;

  view = hildon_gtk_tree_view_new (HILDON_UI_MODE_EDIT);

  /* Progress */
  renderer = gtk_cell_renderer_progress_new ();
  g_object_set (renderer,
      "height", 105, /* HILDON_HEIGHT_THUMB */
      NULL);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
      -1, "", renderer, "value", COL_PROGRESS_PERCENT, NULL);

  /* Status string */
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer,
      "height", 105, /* HILDON_HEIGHT_THUMB */
      "xalign", 0.0,
      NULL);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
      -1, "", renderer, "text", COL_TRANSFER_STRING, NULL);

  /* Status string */
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer,
      "height", 105, /* HILDON_HEIGHT_THUMB */
      "xalign", 1.0,
      NULL);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
      -1, "", renderer, "text", COL_REMAINING_STRING, NULL);

  gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (data->store));

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_BROWSE);

  gtk_widget_show (view);

  return view;
}

static gboolean
dialog_delete_event_cb (GtkWidget *widget,
    GdkEvent *event,
    TransfersData *data)
{
  gtk_widget_hide (widget);
  monorail_transfers_dec (data->monorail_data, FALSE);
  return TRUE;
}

static void
create_transfer_window (MonorailData *monorail_data)
{
  TransfersData *data;
  GtkWidget *vbox;
  GtkWidget *action_area;
  GtkWidget *open_button;
  GtkWidget *cancel_button;
  GtkWidget *clear_button;

  data = g_slice_new0 (TransfersData);
  data->monorail_data = monorail_data;

  monorail_data->transfers_data = data;

  data->num_transfers = 0;

  data->store = gtk_list_store_new (NUM_COLS, G_TYPE_INT, G_TYPE_STRING,
      G_TYPE_STRING, G_TYPE_ULONG, G_TYPE_ULONG, G_TYPE_ULONG,
      EMPATHY_TYPE_FT_HANDLER, G_TYPE_FILE, G_TYPE_UINT);

  data->dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (data->dialog), "File Transfers");
  gtk_container_set_border_width (GTK_CONTAINER (data->dialog), 5);
  gtk_widget_show (data->dialog);
  data->dialog_visible = TRUE;

  /* avoid to quit Monorail while the dialog is shown */
  monorail_transfers_inc (monorail_data, FALSE);

  gtk_window_set_transient_for (GTK_WINDOW (data->dialog),
      GTK_WINDOW (monorail_data->contact_window));

  g_signal_connect (data->dialog, "delete-event",
      G_CALLBACK (dialog_delete_event_cb), data);

  action_area = gtk_dialog_get_action_area (GTK_DIALOG (data->dialog));

  open_button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "Open",
      NULL);
  g_signal_connect (open_button, "clicked",
      G_CALLBACK (open_clicked_cb), data);
  gtk_box_pack_start (GTK_BOX (action_area), open_button, TRUE, TRUE, 0);
  gtk_widget_show (open_button);

  cancel_button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "Cancel",
      NULL);
  g_signal_connect (cancel_button, "clicked",
      G_CALLBACK (cancel_clicked_cb), data);
  gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (cancel_button);

  clear_button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "Clear",
      NULL);
  g_signal_connect (clear_button, "clicked",
      G_CALLBACK (clear_clicked_cb), data);
  gtk_box_pack_end (GTK_BOX (action_area), clear_button, TRUE, TRUE, 0);
  gtk_widget_show (clear_button);

  vbox = gtk_dialog_get_content_area (GTK_DIALOG (data->dialog));

  data->view = create_tree_view (data);

  data->pannable_area = g_object_ref (hildon_pannable_area_new ());
  gtk_container_add (GTK_CONTAINER (data->pannable_area), data->view);
  gtk_widget_show (data->view);

  hildon_pannable_area_set_size_request_policy (
      HILDON_PANNABLE_AREA (data->pannable_area),
      HILDON_SIZE_REQUEST_CHILDREN);

  data->no_transfers_label = g_object_ref (gtk_label_new ("(no transfers)"));

  gtk_box_pack_start (GTK_BOX (vbox), data->pannable_area, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), data->no_transfers_label, TRUE, TRUE, 0);

  show_or_hide_view (data);
}

static gchar *
transfer_format_interval (guint interval)
{
  gint hours, mins, secs;

  hours = interval / 3600;
  interval -= hours * 3600;
  mins = interval / 60;
  interval -= mins * 60;
  secs = interval;

  if (hours > 0)
    return g_strdup_printf ("%u:%02u:%02u", hours, mins, secs);
  else
    return g_strdup_printf ("%02u:%02u", mins, secs);
}

static void
transfer_progress (EmpathyFTHandler *handler,
    guint64 current_bytes,
    guint64 total_bytes,
    guint remaining_time,
    gdouble speed,
    TransfersData *data)
{
  gint percentage;
  gchar *total_str, *current_str, *text;
  gchar *speed_str = NULL;
  const gchar *receiving_sending;
  gchar *basename;
  GtkTreeIter iter;
  gchar *remaining;

  if (!find_iter (data->store, handler, &iter))
    {
      DEBUG ("Failed to find iter");
      return;
    }

  /* Transfer string */
  total_str = g_format_size_for_display (total_bytes);
  current_str = g_format_size_for_display (current_bytes);

  if (speed > 0)
    speed_str = g_format_size_for_display ((goffset) speed);

  receiving_sending = empathy_ft_handler_is_incoming (handler)
    ? "Receiving" : "Sending";

  basename = g_file_get_basename (empathy_ft_handler_get_gfile (handler));

  if (speed_str != NULL)
    {
      text = g_strdup_printf ("%s %s\n%s of %s at %s/s",
          receiving_sending, basename, current_str, total_str, speed_str);
    }
  else
    {
      text = g_strdup_printf ("%s %s\n%s of %s",
          receiving_sending, basename, current_str, total_str);
    }

  g_free (basename);
  g_free (total_str);
  g_free (current_str);
  g_free (speed_str);

  /* Percentage gint */
  if (total_bytes != 0)
    percentage = current_bytes * 100 / total_bytes;
  else
    percentage = -1;

  remaining = transfer_format_interval (remaining_time);

  /* Set values */
  gtk_list_store_set (data->store, &iter,
      COL_PROGRESS_PERCENT, percentage,
      COL_TRANSFER_STRING, text,
      COL_REMAINING_STRING, remaining,
      COL_STATUS, STATUS_PROGRESS,
      -1);

  g_free (remaining);
  g_free (text);
}

static void
transfer_done (EmpathyFTHandler *handler,
    EmpathyTpFile *tp_file,
    TransfersData *data)
{
  gchar *text, *basename;
  GtkTreeIter iter;
  gulong done, error, progress;

  if (!find_iter (data->store, handler, &iter))
    {
      DEBUG ("Failed to find iter");
      return;
    }

  basename = g_file_get_basename (
      empathy_ft_handler_get_gfile (handler));
  text = g_strdup_printf ("%s %s\nFile transfer completed",
      empathy_ft_handler_is_incoming (handler) ? "Received" : "Sent",
      basename);
  g_free (basename);

  gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
      COL_SIGNAL_PROGRESS_ID, &progress,
      COL_SIGNAL_DONE_ID, &done,
      COL_SIGNAL_ERROR_ID, &error,
      -1);

  g_signal_handler_disconnect (handler, progress);
  g_signal_handler_disconnect (handler, done);
  g_signal_handler_disconnect (handler, error);

  gtk_list_store_set (data->store, &iter,
      COL_TRANSFER_STRING, text,
      COL_REMAINING_STRING, "",
      COL_SIGNAL_PROGRESS_ID, 0,
      COL_SIGNAL_DONE_ID, 0,
      COL_SIGNAL_ERROR_ID, 0,
      COL_FT_HANDLER, NULL,
      COL_STATUS, STATUS_DONE,
      -1);

  g_free (text);

  monorail_transfers_dec (data->monorail_data, TRUE);
}

static void
transfer_error (EmpathyFTHandler *handler,
    GError *error,
    TransfersData *data)
{
  GtkTreeIter iter;
  gchar *text, *basename;
  const gchar *receiving_sending;
  gulong done, error_id, progress;

  if (!find_iter (data->store, handler, &iter))
    {
      DEBUG ("Failed to find iter");
      return;
    }

  DEBUG ("%s", error->message);

  receiving_sending = empathy_ft_handler_is_incoming (handler)
    ? "Receiving" : "Sending";

  basename = g_file_get_basename (empathy_ft_handler_get_gfile (handler));
  text = g_strdup_printf ("%s %s\n%s",
      receiving_sending, basename, error->message);
  g_free (basename);

  gtk_tree_model_get (GTK_TREE_MODEL (data->store), &iter,
      COL_SIGNAL_PROGRESS_ID, &progress,
      COL_SIGNAL_DONE_ID, &done,
      COL_SIGNAL_ERROR_ID, &error_id,
      -1);

  g_signal_handler_disconnect (handler, progress);
  g_signal_handler_disconnect (handler, done);
  g_signal_handler_disconnect (handler, error_id);

  gtk_list_store_set (data->store, &iter,
      COL_TRANSFER_STRING, text,
      COL_REMAINING_STRING, "",
      COL_SIGNAL_PROGRESS_ID, 0,
      COL_SIGNAL_DONE_ID, 0,
      COL_SIGNAL_ERROR_ID, 0,
      COL_FT_HANDLER, NULL,
      COL_STATUS, STATUS_ERROR,
      -1);

  g_free (text);

  monorail_transfers_dec (data->monorail_data, TRUE);
}

void
transfer_new_handler (MonorailHandler *handler,
    EmpathyFTHandler *ft_handler,
    MonorailData *monorail_data)
{
  gulong progress, done, error;
  TransfersData *data;
  GtkTreeIter iter;
  gchar *text, *basename;
  GFile *file;

  monorail_transfers_inc (monorail_data, TRUE);

  /* This needs to be around as we're passing the TransfersData to the signal
   * callbacks, so let's create it now. */
  if (monorail_data->transfers_data == NULL)
    create_transfer_window (monorail_data);

  data = monorail_data->transfers_data;

  progress = g_signal_connect (ft_handler, "transfer-progress",
      G_CALLBACK (transfer_progress), data);
  done = g_signal_connect (ft_handler, "transfer-done",
      G_CALLBACK (transfer_done), data);
  error = g_signal_connect (ft_handler, "transfer-error",
      G_CALLBACK (transfer_error), data);

  file = g_file_dup (empathy_ft_handler_get_gfile (ft_handler));
  basename = g_file_get_basename (file);
  text = g_strdup_printf ("%s %s\n%s",
      empathy_ft_handler_is_incoming (ft_handler) ? "Receiving" : "Sending",
      basename,
      "Waiting for the recipient's response");
  g_free (basename);

  data->num_transfers++;
  monorail_transfers_inc (monorail_data, FALSE);
  gtk_list_store_append (data->store, &iter);
  gtk_list_store_set (data->store, &iter,
      COL_PROGRESS_PERCENT, 0,
      COL_TRANSFER_STRING, text,
      COL_REMAINING_STRING, "",
      COL_SIGNAL_PROGRESS_ID, progress,
      COL_SIGNAL_DONE_ID, done,
      COL_SIGNAL_ERROR_ID, error,
      COL_FT_HANDLER, ft_handler,
      COL_GFILE, file,
      COL_STATUS, STATUS_PROGRESS,
      -1);

  show_or_hide_view (data);

  g_free (text);

  transfer_show_window (monorail_data);
}

void
transfer_show_window (MonorailData *monorail_data)
{
  TransfersData *data = (TransfersData *) monorail_data->transfers_data;

  if (monorail_data->transfers_data == NULL)
    create_transfer_window (monorail_data);
  else 
    {
      if (!data->dialog_visible)
        monorail_transfers_inc (monorail_data, FALSE);
    }

  gtk_window_present (
      GTK_WINDOW (((TransfersData *) monorail_data->transfers_data)->dialog));
  data->dialog_visible = TRUE;
}
