/*
 * libfunsharing.c
 *
 * Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Author: Alban Crequy <alban.crequy@collabora.co.uk>
 */
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-gtype-specialized.h>
#include <dbus/dbus-glib-lowlevel.h>

#define __USE_GNU 1

#include <dlfcn.h>

/* functions exported by libsharing-dialog */
void sharing_dialog_with_file (void *osso, void *parent, const gchar *uri);
void sharing_dialog_with_files (void *osso, void *parent, GSList *uris);
void sharing_dialog_button_mask_with_file (void *osso,
    void *parent, const gchar *uris, guint share_button_mask);
void sharing_dialog_button_mask_with_files(void *osso, void *parent,
    GSList *uris, unsigned int share_button_mask);


static int inited = 0;

static void (*old_table_constructed) (GObject *object) = NULL;

G_GNUC_UNUSED static void  (*original_sharing_dialog_with_file)
	(void *, void *, const gchar *) = NULL;
G_GNUC_UNUSED static void  (*original_sharing_dialog_with_files)
	(void *, void *, GSList *) = NULL;
G_GNUC_UNUSED static void  (*original_sharing_dialog_button_mask_with_file)
	(void *, void *, const gchar *, unsigned int) = NULL;
static void  (*original_sharing_dialog_button_mask_with_files)
	(void *, void *, GSList *, unsigned int) = NULL;

static DBusGConnection *connection = NULL;

/* A copy of uris used when the user choose our button 'Share via IM' */
static GSList *copy_uris = NULL;

static void
maybe_init() 
{
  char *str;
  void *handle;

  if (inited)
    return;

  dlerror();
  handle = dlopen ("/usr/lib/funsharing/libsharingdialog.so.0.0.0.real",
      RTLD_NOW);
  str = dlerror();
  if (str)
    printf ("dlopen error: %s\n", str);

  original_sharing_dialog_button_mask_with_files
	  = (void (*) (void *, void *, GSList *, unsigned int))
	  dlsym(handle, "sharing_dialog_button_mask_with_files");

  str = dlerror();
  if (str)
    printf ("dlsym error: %s\n", str);

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);

  printf ("Init: %p\n",
		  original_sharing_dialog_button_mask_with_files);

  inited = 1;
}

static void
share_file_via_im (GSList *uris)
{
  DBusMessage *message;
  DBusMessageIter iter;
  DBusMessageIter sub_iter;
  int ret;

  message = dbus_message_new_method_call ("org.maemo.Monorail",
      "/org/maemo/Monorail/Send", "org.maemo.Monorail.Send", "SendMultiple");
  dbus_message_iter_init_append (message, &iter);
  dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "s", &sub_iter);

  while (uris != NULL)
    {
      printf ("Adding file: '%s'\n", (gchar *) uris->data);
      dbus_message_iter_append_basic (&sub_iter, DBUS_TYPE_STRING, &uris->data);
      uris = uris->next;
    }
  dbus_message_iter_close_container (&iter, &sub_iter);

  ret = dbus_connection_send
    (dbus_g_connection_get_connection (connection), message, NULL);

  dbus_message_unref (message);
}

static void
share_via_im_button_clicked (HildonButton *button,
    gpointer user_data G_GNUC_UNUSED)
{
  GtkWidget *dialog = GTK_WIDGET (button);

  /* We don't have the reference on the dialog because the libsharing-dialog
   * API does not give it to us. But we find it anyway! */
  while (dialog != NULL && ! GTK_IS_DIALOG (dialog))
  {
    dialog = gtk_widget_get_parent (dialog);
  }

  if (copy_uris)
  {
    share_file_via_im (copy_uris);
  }
  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
}

/* This function replaces temporarily GtkTable's constructed. The API from
 * libsharing-dialog does not give us a reference to the dialog or the table.
 * By replacing constructed(), we get the reference on the table and we can add
 * our 'Share by IM' button.
 */
static void
new_table_constructed(GObject *object)
{
  GtkWidget *button;
  GtkWidget *table;

  if (old_table_constructed != NULL)
    old_table_constructed (object);

  table = GTK_WIDGET (object);
  if (!table)
    return;

  button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
   HILDON_BUTTON_ARRANGEMENT_VERTICAL,
   "Share via IM",
   NULL);

  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 1, 2,
      GTK_FILL|GTK_EXPAND,
      GTK_FILL|GTK_EXPAND,
      0, 0);

  g_signal_connect (button, "clicked",
      G_CALLBACK (share_via_im_button_clicked), table);
}

/* Implementations of libsharing-dialog functions */

void sharing_dialog_with_file (void *osso, void *parent, const gchar *uri)
{
  GSList uris;
  uris.next = NULL;
  uris.data = (gpointer) uri;
  sharing_dialog_button_mask_with_files (osso, parent, &uris, 0x7);
}

void sharing_dialog_with_files (void *osso, void *parent, GSList *uris)
{
  sharing_dialog_button_mask_with_files (osso, parent, uris, 0x7);
}

void sharing_dialog_button_mask_with_file (void *osso,
    void *parent, const gchar *uri, guint share_button_mask)
{
  GSList uris;
  uris.next = NULL;
  uris.data = (gpointer) uri;
  sharing_dialog_button_mask_with_files (osso, parent, &uris,
      share_button_mask);
}

void
sharing_dialog_button_mask_with_files(void *osso, void *parent, GSList *uris,
		unsigned int share_button_mask)
{
  
  GObjectClass *class;

  maybe_init();

  class = g_type_class_ref(GTK_TYPE_TABLE);
  old_table_constructed = class->constructed;

  class->constructed = new_table_constructed;

  printf ("Diverted sharing_dialog_button_mask_with_files called.\n");
  if (original_sharing_dialog_button_mask_with_files)
  {
    copy_uris = uris;
    original_sharing_dialog_button_mask_with_files(osso, parent, uris, share_button_mask);
    copy_uris = NULL;
  }

  class->constructed = old_table_constructed;
  old_table_constructed = NULL;
}

