/*
 * status-plugin.c
 *
 * Copyright (C) 2009 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <hildon/hildon.h>

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/proxy-subclass.h>

#include "extensions/extensions.h"

#include "status-plugin.h"

struct _MonorailStatusPluginPrivate
{
  GdkPixbuf *icon;
  GtkWidget *menu_button;
  TpDBusDaemon *dbus;
  TpProxy *proxy;
  TpProxySignalConnection *changed_signal;
};

HD_DEFINE_PLUGIN_MODULE (MonorailStatusPlugin, monorail_status_plugin,
    HD_TYPE_STATUS_MENU_ITEM);

#define DEBUG(format, ...) \
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, format, ##__VA_ARGS__)

static void
monorail_status_plugin_class_finalize (MonorailStatusPluginClass *klass)
{
}

static void
num_transfers_changed_cb (TpProxy *proxy,
    guint arg_num,
    gpointer user_data,
    GObject *weak_object)
{
  MonorailStatusPlugin *plugin = MONORAIL_STATUS_PLUGIN (user_data);
  gchar *text;

  DEBUG ("NumTransfersChanged: %u", arg_num);

  if (arg_num > 0)
    {
      text = g_strdup_printf ("%u active transfer%s",
          arg_num, (arg_num > 1) ? "s" : "");
      hildon_button_set_value (HILDON_BUTTON (plugin->priv->menu_button),
          text);
      g_free (text);

      hd_status_plugin_item_set_status_area_icon (
          HD_STATUS_PLUGIN_ITEM (plugin), plugin->priv->icon);
      gtk_widget_show (GTK_WIDGET (plugin));
    }
  else
    {
      hd_status_plugin_item_set_status_area_icon (
          HD_STATUS_PLUGIN_ITEM (plugin), NULL);
      gtk_widget_hide (GTK_WIDGET (plugin));
    }
}

static void
name_owner_changed_cb (TpDBusDaemon *bus_daemon,
    const gchar *name,
    const gchar *new_owner,
    gpointer user_data)
{
  MonorailStatusPlugin *plugin = MONORAIL_STATUS_PLUGIN (user_data);

  DEBUG ("NameOwnerChanged: %s -> %s", name, new_owner);

  if (new_owner == NULL || new_owner[0] == '\0')
    {
      if (plugin->priv->proxy == NULL)
        return;

      DEBUG ("Freeing proxy and disconnecting from signal");
      tp_proxy_signal_connection_disconnect (plugin->priv->changed_signal);
      plugin->priv->changed_signal = NULL;

      g_object_unref (plugin->priv->proxy);
      plugin->priv->proxy = NULL;
    }
  else
    {
      g_assert (plugin->priv->proxy == NULL);

      plugin->priv->proxy = g_object_new (TP_TYPE_PROXY,
          "bus-name", "org.maemo.Monorail",
          "dbus-connection", tp_proxy_get_dbus_connection (bus_daemon),
          "object-path", "/org/maemo/Monorail/Notifications",
          NULL);

      tp_proxy_add_interface_by_id (plugin->priv->proxy,
          MONORAIL_IFACE_QUARK_NOTIFICATIONS);

      plugin->priv->changed_signal =
        monorail_cli_notifications_connect_to_num_transfers_changed (
            plugin->priv->proxy, num_transfers_changed_cb, plugin, NULL, NULL,
            NULL);
    }
}

static void
monorail_status_plugin_dispose (GObject *obj)
{
  MonorailStatusPlugin *plugin = MONORAIL_STATUS_PLUGIN (obj);

  if (plugin->priv->icon != NULL)
    {
      g_object_unref (plugin->priv->icon);
      plugin->priv->icon = NULL;
    }

  if (plugin->priv->dbus != NULL)
    {
      tp_dbus_daemon_cancel_name_owner_watch (plugin->priv->dbus,
          "org.maemo.Monorail", name_owner_changed_cb, plugin);

      g_object_unref (plugin->priv->dbus);
      plugin->priv->dbus = NULL;
    }
}

static void
monorail_status_plugin_class_init (MonorailStatusPluginClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = monorail_status_plugin_dispose;

  g_type_class_add_private (klass, sizeof (MonorailStatusPluginPrivate));
}

static void
menu_button_clicked_cb (GtkButton *button,
    gpointer user_data)
{
  MonorailStatusPlugin *plugin = MONORAIL_STATUS_PLUGIN (user_data);

  if (plugin->priv->proxy == NULL)
    return;

  monorail_cli_notifications_call_show_transfers (plugin->priv->proxy, -1,
      NULL, NULL, NULL, NULL);
}

static void
monorail_status_plugin_init (MonorailStatusPlugin *plugin)
{
  MonorailStatusPluginPrivate *priv;
  GtkIconTheme *icon_theme;

  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "plugin init");

  plugin->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin,
      MONORAIL_TYPE_STATUS_PLUGIN, MonorailStatusPluginPrivate);
  priv = plugin->priv;

  monorail_cli_init ();

  icon_theme = gtk_icon_theme_get_default ();

  priv->icon = gtk_icon_theme_load_icon (icon_theme, "ovi_widget_files",
      18, GTK_ICON_LOOKUP_NO_SVG, NULL);

  priv->menu_button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
      HILDON_BUTTON_ARRANGEMENT_VERTICAL, "File Transfers", NULL);
  hildon_button_set_image (HILDON_BUTTON (priv->menu_button),
      gtk_image_new_from_icon_name ("ovi_widget_files", GTK_ICON_SIZE_BUTTON));
  hildon_button_set_alignment (HILDON_BUTTON (priv->menu_button), 0, 0, 1, 1);
  gtk_widget_show (priv->menu_button);

  g_signal_connect (priv->menu_button, "clicked",
      G_CALLBACK (menu_button_clicked_cb), plugin);

  gtk_container_add (GTK_CONTAINER (plugin), priv->menu_button);

  priv->dbus = tp_dbus_daemon_dup (NULL);

  tp_dbus_daemon_watch_name_owner (priv->dbus, "org.maemo.Monorail",
      name_owner_changed_cb, plugin, NULL);
}
