/*
 * monorail-handler.c
 *
 * Copyright (C) 2009 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/defs.h>
#include <telepathy-glib/util.h>
#include "monorail-handler.h"

#include "monorail.h"
#include "monorail-marshal.h"
#include "handler.h"
#include "empathy-ft-factory.h"
#include "debug.h"

#define HANDLER_DATA_KEY "MonorailHandler"

#define GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MONORAIL_TYPE_HANDLER, \
      MonorailHandlerPrivate))

G_DEFINE_TYPE(MonorailHandler, monorail_handler, G_TYPE_OBJECT)

enum
{
  PROP_0,
  PROP_HANDLER_NAME,
  PROP_APPROVER
};

enum /* signals */
{
  APPROVE_FT_CHANNEL,
  INCOMING_FT,
  NEW_FT_HANDLER,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

typedef struct _MonorailHandlerPrivate MonorailHandlerPrivate;
struct _MonorailHandlerPrivate
{
  TpDBusDaemon *dbus;
  TpHandler *handler;
  EmpathyFTFactory *factory;
  GHashTable *ft_handlers;
  gchar *handler_name;
  gboolean approver;
};

static gboolean
remove_ft_handler (gpointer key,
    gpointer value,
    gpointer user_data)
{
  return value == user_data;
}

static gboolean
handle_channels (TpHandler *self,
    TpAccount *account,
    TpConnection *connection,
    TpChannel **channels,
    TpChannelRequest **requests,
    guint64 user_action_time,
    GHashTable *handler_info)
{
  MonorailHandler *handler;
  MonorailHandlerPrivate *priv;

  TpChannel **channel_ptr;
  TpChannelRequest **request_ptr;

  EmpathyFTHandler *ft_handler = NULL;
  TpChannel *channel = NULL;

  handler = MONORAIL_HANDLER (g_object_get_data (G_OBJECT (self), HANDLER_DATA_KEY));
  priv = GET_PRIVATE (handler);

  for (channel_ptr = channels; *channel_ptr != NULL; channel_ptr++)
    {
      TpChannel *c = TP_CHANNEL (*channel_ptr);

      if (!tp_strdiff (tp_channel_get_channel_type (c),
              TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER))
        {
          channel = c;
          break;
        }
    }

  if (channel == NULL)
    {
      DEBUG ("Failed to find a channel in the list that is FT");
      return FALSE;
    }

  for (request_ptr = requests; *request_ptr != NULL; request_ptr++)
    {
      TpChannelRequest *r = TP_CHANNEL_REQUEST (*request_ptr);

      ft_handler = g_hash_table_lookup (priv->ft_handlers,
          tp_proxy_get_object_path (r));

      if (ft_handler != NULL)
        break;
    }

  if (ft_handler == NULL)
    {
      DEBUG ("Failed to find a FT handler being handled by monorail; must be incoming");
      empathy_ft_factory_claim_channel (priv->factory, account, channel);
    }
  else
    {
      empathy_ft_handler_outgoing_set_channel (ft_handler, channel);

      g_hash_table_foreach_remove (priv->ft_handlers, remove_ft_handler,
          ft_handler);
    }

  return TRUE;
}

static void
add_cdo (TpHandler *handler,
    TpChannel **channels,
    TpChannelDispatchOperation *dispatch_operation,
    GHashTable *properties)
{
  MonorailHandler *self;

  self = MONORAIL_HANDLER (g_object_get_data (G_OBJECT (handler), HANDLER_DATA_KEY));

  g_signal_emit (self, signals[APPROVE_FT_CHANNEL], 0, channels,
      dispatch_operation, properties);
}

static void
handler_channel_requested (EmpathyFTHandler *ft_handler,
    const gchar *request_path,
    gpointer user_data)
{
  MonorailHandler *handler = MONORAIL_HANDLER (user_data);
  MonorailHandlerPrivate *priv = GET_PRIVATE (handler);

  g_hash_table_insert (priv->ft_handlers, g_strdup (request_path), ft_handler);
}

static void
new_ft_handler (EmpathyFTFactory *factory,
    EmpathyFTHandler *ft_handler,
    GError *error,
    gpointer user_data)
{
  MonorailHandler *handler = MONORAIL_HANDLER (user_data);

  if (error != NULL)
    {
      DEBUG ("Error with new FT handler: %s", error->message);
    }
  else
    {
      DEBUG ("Got new FT handler");

      g_signal_connect (ft_handler, "channel-requested",
          G_CALLBACK (handler_channel_requested), handler);

      g_signal_emit (handler, signals[NEW_FT_HANDLER], 0, ft_handler);

      empathy_ft_handler_start_transfer (ft_handler);
    }
}

static void
new_incoming_transfer (EmpathyFTFactory *factory,
    EmpathyFTHandler *ft_handler,
    GError *error,
    gpointer user_data)
{
  MonorailHandler *handler = MONORAIL_HANDLER (user_data);

  if (error != NULL)
    {
      DEBUG ("Error with new incoming transfer: %s", error->message);
    }
  else
    {
      g_signal_emit (handler, signals[INCOMING_FT], 0, ft_handler);
    }
}

static void
monorail_handler_dispose (GObject *self)
{
  MonorailHandlerPrivate *priv = GET_PRIVATE (self);

  if (priv->dbus != NULL)
    {
      g_object_unref (priv->dbus);
      priv->dbus = NULL;
    }

  if (priv->handler != NULL)
    {
      g_object_unref (priv->handler);
      priv->handler = NULL;
    }

  if (priv->factory != NULL)
    {
      g_object_unref (priv->factory);
      priv->factory = NULL;
    }

  if (priv->ft_handlers != NULL)
    {
      g_hash_table_destroy (priv->ft_handlers);
      priv->ft_handlers = NULL;
    }

  if (priv->handler_name != NULL)
    {
      g_free (priv->handler_name);
      priv->handler_name = NULL;
    }

  if (G_OBJECT_CLASS (monorail_handler_parent_class)->dispose)
    G_OBJECT_CLASS (monorail_handler_parent_class)->dispose (self);
}

static void
monorail_handler_constructed (GObject *self)
{
  MonorailHandlerPrivate *priv = GET_PRIVATE (self);
  GPtrArray *filter;
  GHashTable *map;
  gchar *name;

  priv->dbus = tp_dbus_daemon_dup (NULL);

  filter = g_ptr_array_new ();
  map = tp_asv_new (
        TP_IFACE_CHANNEL ".ChannelType", G_TYPE_STRING,
            TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER,
        NULL
      );
  g_ptr_array_add (filter, map);

  if (priv->approver)
    {
      priv->handler = tp_handler_new (filter, filter, FALSE,
          handle_channels, add_cdo);
    }
  else
    {
      GPtrArray *empty_filter;
      empty_filter = g_ptr_array_new ();
      priv->handler = tp_handler_new (filter, empty_filter, FALSE,
          handle_channels, add_cdo);
      g_ptr_array_free (empty_filter, TRUE);
    }

  /* TODO: make TpHandler be able to have user_data, or weak_object */
  g_object_set_data (G_OBJECT (priv->handler), HANDLER_DATA_KEY, self);

  name = g_strconcat (TP_CLIENT_BUS_NAME_BASE,
      priv->handler_name, NULL);

  g_assert (tp_dbus_daemon_request_name (priv->dbus,
          name, TRUE, NULL));

  g_free (name);
  name = g_strconcat (TP_CLIENT_OBJECT_PATH_BASE,
      priv->handler_name, NULL);

  dbus_g_connection_register_g_object (
      tp_proxy_get_dbus_connection (TP_PROXY (priv->dbus)),
      name, G_OBJECT (priv->handler));

  g_free (name);

  g_ptr_array_free (filter, TRUE);

  priv->factory = empathy_ft_factory_dup_singleton ();

  g_signal_connect (priv->factory, "new-ft-handler",
      G_CALLBACK (new_ft_handler), self);
  g_signal_connect (priv->factory, "new-incoming-transfer",
      G_CALLBACK (new_incoming_transfer), self);

  priv->ft_handlers = g_hash_table_new_full (g_str_hash, g_str_equal,
      (GDestroyNotify) g_free, NULL);
}

static void
monorail_handler_get_property (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  MonorailHandlerPrivate *priv = GET_PRIVATE (object);

  switch (property_id)
    {
      case PROP_HANDLER_NAME:
        g_value_set_string (value, priv->handler_name);
        break;
      case PROP_APPROVER:
        g_value_set_boolean (value, priv->approver);
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
monorail_handler_set_property (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec)
{
  MonorailHandlerPrivate *priv = GET_PRIVATE (object);

  switch (property_id)
    {
      case PROP_HANDLER_NAME:
        priv->handler_name = g_value_dup_string (value);
        break;
      case PROP_APPROVER:
        priv->approver = g_value_get_boolean (value);
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
monorail_handler_class_init (MonorailHandlerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = monorail_handler_dispose;
  object_class->constructed = monorail_handler_constructed;
  object_class->get_property = monorail_handler_get_property;
  object_class->set_property = monorail_handler_set_property;

  g_object_class_install_property (object_class, PROP_HANDLER_NAME,
      g_param_spec_string ("handler-name",
          "handler-name", "handler-name", NULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_APPROVER,
      g_param_spec_boolean ("approver",
          "approver", "approver", TRUE,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_CONSTRUCT_ONLY));

  signals[APPROVE_FT_CHANNEL] = g_signal_new ("approve-ft-channel",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      _monorail_marshal_VOID__POINTER_OBJECT_POINTER,
      G_TYPE_NONE, 3,
      G_TYPE_POINTER, TP_TYPE_CHANNEL_DISPATCH_OPERATION, G_TYPE_POINTER);

  signals[INCOMING_FT] = g_signal_new ("incoming-ft",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      _monorail_marshal_VOID__OBJECT,
      G_TYPE_NONE, 1,
      EMPATHY_TYPE_FT_HANDLER);

  signals[NEW_FT_HANDLER] = g_signal_new ("new-ft-handler",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      _monorail_marshal_VOID__OBJECT,
      G_TYPE_NONE, 1,
      EMPATHY_TYPE_FT_HANDLER);

  g_type_class_add_private (klass, sizeof (MonorailHandlerPrivate));
}

static void
monorail_handler_init (MonorailHandler *self)
{
}

MonorailHandler *
monorail_handler_new (const gchar *handler_name,
    gboolean approver)
{
  return g_object_new (MONORAIL_TYPE_HANDLER,
      "handler-name", handler_name,
      "approver", approver,
      NULL);
}

EmpathyFTFactory *
monorail_handler_get_factory (MonorailHandler *self)
{
  MonorailHandlerPrivate *priv;

  g_return_val_if_fail (MONORAIL_IS_HANDLER (self), NULL);

  priv = GET_PRIVATE (self);

  return priv->factory;
}

const gchar *
monorail_handler_get_name (MonorailHandler *self)
{
  MonorailHandlerPrivate *priv;

  g_return_val_if_fail (MONORAIL_IS_HANDLER (self), NULL);

  priv = GET_PRIVATE (self);

  return priv->handler_name;
}
