/*
 * empathy-ft-factory.c - Source for EmpathyFTFactory
 * Copyright (C) 2009 Collabora Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Author: Cosimo Cecchi <cosimo.cecchi@collabora.co.uk>
 */

/* empathy-ft-factory.c */

#include <glib.h>

#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/util.h>

#include "empathy-ft-factory.h"
#include "empathy-ft-handler.h"

#include "monorail-marshal.h"
#include "monorail.h"
#include "debug.h"

/**
 * SECTION:empathy-ft-factory
 * @title:EmpathyFTFactory
 * @short_description: creates #EmpathyFTHandler objects
 * @include: libempathy/empathy-ft-factory.h
 *
 * #EmpathyFTFactory takes care of the creation of the #EmpathyFTHandler
 * objects used for file transfer. As the creation of the handlers is
 * async, a client will have to connect to the ::new-ft-handler signal
 * to receive the handler.
 * In case of an incoming file transfer, the handler will need the destination
 * file before being useful; as this is usually decided by the user (e.g. with
 * a file selector), a ::new-incoming-transfer is emitted by the factory when
 * a destination file is needed, which can be set later with
 * empathy_ft_factory_set_destination_for_incoming_handler().
 */

G_DEFINE_TYPE (EmpathyFTFactory, empathy_ft_factory, G_TYPE_OBJECT);

enum {
  NEW_FT_HANDLER,
  NEW_INCOMING_TRANSFER,
  LAST_SIGNAL
};

static gpointer factory_singleton = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

static GObject *
do_constructor (GType type,
    guint n_props,
    GObjectConstructParam *props)
{
	GObject *retval;

	if (factory_singleton != NULL) {
		retval = g_object_ref (factory_singleton);
	} else {
		retval = G_OBJECT_CLASS (empathy_ft_factory_parent_class)->constructor
			(type, n_props, props);

		factory_singleton = retval;
		g_object_add_weak_pointer (retval, &factory_singleton);
	}

	return retval;
}

static void
empathy_ft_factory_class_init (EmpathyFTFactoryClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->constructor = do_constructor;

  /**
   * EmpathyFTFactory::new-ft-handler
   * @factory: the object which received the signal
   * @handler: the handler made available by the factory
   * @error: a #GError or %NULL
   *
   * The signal is emitted when a new #EmpathyFTHandler is available.
   * Note that @handler is never %NULL even if @error is set, as you might want
   * to display the error in an UI; in that case, the handler won't support
   * any transfer.
   */
  signals[NEW_FT_HANDLER] =
    g_signal_new ("new-ft-handler",
      G_TYPE_FROM_CLASS (klass),
      G_SIGNAL_RUN_LAST, 0,
      NULL, NULL,
      _monorail_marshal_VOID__OBJECT_POINTER,
      G_TYPE_NONE, 2, EMPATHY_TYPE_FT_HANDLER, G_TYPE_POINTER);

  /**
   * EmpathyFTFactory::new-incoming-transfer
   * @factory: the object which received the signal
   * @handler: the incoming handler being constructed
   * @error: a #GError or %NULL
   *
   * The signal is emitted when a new incoming #EmpathyFTHandler is being
   * constructed, and needs a destination #GFile to be useful.
   * Clients that connect to this signal will have to call
   * empathy_ft_factory_set_destination_for_incoming_handler() when they
   * have a #GFile.
   * Note that @handler is never %NULL even if @error is set, as you might want
   * to display the error in an UI; in that case, the handler won't support
   * any transfer.
   */
  signals[NEW_INCOMING_TRANSFER] =
    g_signal_new ("new-incoming-transfer",
      G_TYPE_FROM_CLASS (klass),
      G_SIGNAL_RUN_LAST, 0,
      NULL, NULL,
      _monorail_marshal_VOID__OBJECT_POINTER,
      G_TYPE_NONE, 2, EMPATHY_TYPE_FT_HANDLER, G_TYPE_POINTER);
}

static void
empathy_ft_factory_init (EmpathyFTFactory *self)
{
  /* do nothing */
}

static void
ft_handler_outgoing_ready_cb (EmpathyFTHandler *handler,
    GError *error,
    gpointer user_data)
{
  EmpathyFTFactory *factory = user_data;

  g_signal_emit (factory, signals[NEW_FT_HANDLER], 0, handler, error);
}

static void
ft_handler_incoming_ready_cb (EmpathyFTHandler *handler,
    GError *error,
    gpointer user_data)
{
  EmpathyFTFactory *factory = user_data;

  g_signal_emit (factory, signals[NEW_INCOMING_TRANSFER], 0, handler, error);
}

/* public methods */

/**
 * empathy_ft_factory_dup_singleton:
 *
 * Gives the caller a reference to the #EmpathyFTFactory singleton,
 * (creating it if necessary).
 *
 * Return value: an #EmpathyFTFactory object
 */
EmpathyFTFactory *
empathy_ft_factory_dup_singleton (void)
{
  return g_object_new (EMPATHY_TYPE_FT_FACTORY, NULL);
}

typedef struct
{
  gchar *jid;
  gchar *preferred_handler_name;
  GFile *source;
  EmpathyFTFactory *factory;
  TpAccount *account;
} GetConnectionData;

static void
get_connection_data_free (gpointer in)
{
  GetConnectionData *data = in;
  g_free (data->jid);
  g_free (data->preferred_handler_name);
  g_object_unref (data->source);
  g_object_unref (data->factory);
  g_object_unref (data->account);
  g_slice_free (GetConnectionData, data);
}

static void
connection_ready_cb (TpConnection *connection,
    const GError *error,
    gpointer user_data)
{
  GetConnectionData *data = user_data;

  if (error != NULL)
    {
      DEBUG ("Failed to ready the connection: %s", error->message);
      goto out;
    }

  empathy_ft_handler_new_outgoing (data->account, connection,
      data->jid, data->source, data->preferred_handler_name,
      ft_handler_outgoing_ready_cb, data->factory);

out:
  get_connection_data_free (data);
  g_object_unref (connection);
}

static void
get_account_connection_cb (TpProxy *proxy,
    const GValue *value,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  GetConnectionData *data = user_data;
  TpConnection *connection;
  TpDBusDaemon *dbus;

  if (error != NULL)
    {
      DEBUG ("Failed to get connection: %s", error->message);
      get_connection_data_free (data);
      return;
    }

  if (!tp_strdiff (g_value_get_boxed (value), "/"))
    {
      DEBUG ("No connection on account");
      get_connection_data_free (data);
      return;
    }

  dbus = tp_dbus_daemon_dup (NULL);

  connection = tp_connection_new (dbus, NULL, g_value_get_boxed (value), NULL);

  tp_connection_call_when_ready (connection, connection_ready_cb, data);

  g_object_unref (dbus);
}

/**
 * empathy_ft_factory_new_transfer_outgoing:
 * @factory: an #EmpathyFTFactory
 * @contact: the #EmpathyContact destination of the transfer
 * @source: the #GFile to be transferred to @contact
 *
 * Trigger the creation of an #EmpathyFTHandler object to send @source to
 * the specified @contact.
 */
void
empathy_ft_factory_new_transfer_outgoing (EmpathyFTFactory *factory,
    TpAccount *account,
    const gchar *jid,
    GFile *source,
    const gchar *preferred_handler_name)
{
  GetConnectionData *data;

  g_return_if_fail (EMPATHY_IS_FT_FACTORY (factory));
  g_return_if_fail (G_IS_FILE (source));

  data = g_slice_new0 (GetConnectionData);
  data->jid = g_strdup (jid);
  data->preferred_handler_name = g_strdup (preferred_handler_name);
  data->source = g_object_ref (source);
  data->factory = g_object_ref (factory);
  data->account = g_object_ref (account);

  tp_cli_dbus_properties_call_get (account, -1, TP_IFACE_ACCOUNT,
      "Connection", get_account_connection_cb, data, NULL, NULL);
}

/**
 * empathy_ft_factory_claim_channel:
 * @factory: an #EmpathyFTFactory
 * @operation: the #EmpathyDispatchOperation wrapping the channel
 *
 * Let the @factory claim the channel, starting the creation of a new
 * incoming #EmpathyFTHandler.
 */
void
empathy_ft_factory_claim_channel (EmpathyFTFactory *factory,
    TpAccount *account,
    TpChannel *channel)
{
  EmpathyTpFile *tp_file;

  g_return_if_fail (EMPATHY_IS_FT_FACTORY (factory));

  /* own a reference to the EmpathyTpFile */
  tp_file = empathy_tp_file_new (channel, TRUE);

  empathy_ft_handler_new_incoming (account, tp_file,
      ft_handler_incoming_ready_cb, factory);

  g_object_unref (tp_file);
}

/**
 * empathy_ft_factory_set_destination_for_incoming_handler:
 * @factory: an #EmpathyFTFactory
 * @handler: the #EmpathyFTHandler to set the destination of
 * @destination: the #GFile destination of the transfer
 *
 * Sets @destination as destination file for the transfer. After the call of
 * this method, the ::new-ft-handler will be emitted for the incoming handler.
 */
void
empathy_ft_factory_set_destination_for_incoming_handler (
    EmpathyFTFactory *factory,
    EmpathyFTHandler *handler,
    GFile *destination)
{
  g_return_if_fail (EMPATHY_IS_FT_FACTORY (factory));
  g_return_if_fail (EMPATHY_IS_FT_HANDLER (handler));
  g_return_if_fail (G_IS_FILE (destination));

  empathy_ft_handler_incoming_set_destination (handler, destination);

  g_signal_emit (factory, signals[NEW_FT_HANDLER], 0, handler, NULL);
}
