/*
 * contact-chooser.c
 *
 * Copyright (C) 2009 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/util.h>
#include <telepathy-glib/proxy-subclass.h>

#include "extensions/extensions.h"

#include "contact-chooser.h"
#include "debug.h"

#define GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MONORAIL_TYPE_CONTACT_CHOOSER, \
      MonorailContactChooserPrivate))

G_DEFINE_TYPE (MonorailContactChooser, monorail_contact_chooser,
    OSSO_ABOOK_TYPE_CONTACT_VIEW)

/* This widget doesn't handle:
 *  1. accounts going offline
 *  2. accounts coming online
 *  3. contacts going offline
 *  4. contacts going online
 *  5. new contacts appearing
 * (6. custom requestable channel classes)
 *
 * As the model doesn't persist, however, closing and re-opening the dialog
 * will fix any problems. These should of course be fixed in the long term.
 */

enum /* signals */
{
  DONE,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

typedef struct _MonorailContactChooserPrivate MonorailContactChooserPrivate;
struct _MonorailContactChooserPrivate
{
  /* (owned) gchar* -> (owned) GHashTable: TpHandle -> gboolean */
  GHashTable *handle_capable;

  gboolean made_all_calls;
  guint calls_in_progress;

  TpDBusDaemon *dbus;
};

static void
monorail_contact_chooser_finalize (GObject *self)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);

  if (priv->handle_capable != NULL)
    {
      g_hash_table_destroy (priv->handle_capable);
      priv->handle_capable = NULL;
    }

  if (G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->finalize)
    G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->finalize (self);
}

static void
monorail_contact_chooser_dispose (GObject *self)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);

  if (priv->dbus != NULL)
    {
      g_object_unref (priv->dbus);
      priv->dbus = NULL;
    }

  if (G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->dispose)
    G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->dispose (self);
}

static void
check_all_calls_made (MonorailContactChooser *self)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);

  if (!priv->made_all_calls || priv->calls_in_progress != 0)
    return;

  g_signal_emit (self, signals[DONE], 0);
}

static void
free_garray (gpointer data)
{
  g_array_free ((GArray *) data, TRUE);
}

/* https://bugzilla.gnome.org/show_bug.cgi?id=65987 */
static gchar *
strreplace (const gchar *string,
    const gchar *search,
    const gchar *replacement)
{
  gchar *str, **arr;

  if (replacement == NULL)
    replacement = "";

  arr = g_strsplit (string, search, -1);
  if (arr != NULL && arr[0] != NULL)
    str = g_strjoinv (replacement, arr);
  else
    str = g_strdup (string);

  g_strfreev (arr);

  return str;
}

static void
get_contact_caps_cb (TpProxy *proxy,
    GHashTable *contact_caps,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  MonorailContactChooser *chooser = MONORAIL_CONTACT_CHOOSER (weak_object);
  MonorailContactChooserPrivate *priv = GET_PRIVATE (chooser);
  GHashTable *contact_ft_capabilites;
  GHashTableIter iter;
  gpointer key, value;

  priv->calls_in_progress--;

  if (error != NULL)
    {
      DEBUG ("error: %s", error->message);
      return;
    }

  if (contact_caps == NULL)
    {
      DEBUG ("contact capabilities hash table is NULL");
      return;
    }

  contact_ft_capabilites = g_hash_table_new (g_direct_hash, g_direct_equal);

  g_hash_table_iter_init (&iter, contact_caps);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      guint handle = GPOINTER_TO_UINT (key);
      GPtrArray *classes = value;
      gboolean out = FALSE;
      guint i;

      for (i = 0; i < classes->len; i++)
        {
          GHashTable *class;
          GHashTableIter class_iter;
          gpointer class_key, class_value;

          GValueArray *array;

          array = g_ptr_array_index (classes, i);
          class = g_value_get_boxed (g_value_array_get_nth (array, 0));

          g_hash_table_iter_init (&class_iter, class);
          while (g_hash_table_iter_next (&class_iter, &class_key, &class_value))
            {
              if (!tp_strdiff ((const gchar *) class_key,
                      TP_IFACE_CHANNEL ".ChannelType")
                  && !tp_strdiff (g_value_get_string ((GValue *) class_value),
                      TP_IFACE_CHANNEL_TYPE_FILE_TRANSFER))
                {
                  out = TRUE;
                  goto superbreak;
                }
            }

        }

superbreak:
      DEBUG ("Contact %u on connection %p has%s ft capabilites",
          handle, proxy, out ? "" : " no");

      g_hash_table_insert (contact_ft_capabilites, GUINT_TO_POINTER (handle),
          GINT_TO_POINTER (out));
    }

  g_hash_table_insert (priv->handle_capable,
      user_data, /* we already own this string */
      contact_ft_capabilites);

  gtk_tree_model_filter_refilter (
      GTK_TREE_MODEL_FILTER (osso_abook_tree_view_get_filter_model (
              OSSO_ABOOK_TREE_VIEW (chooser))));

  check_all_calls_made (chooser);
}

static void
call_table_foreach (gpointer key,
    gpointer value,
    gpointer user_data)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (user_data);
  TpProxy *proxy;
  gchar *bus_name;
  const gchar *connection_path;

  McAccount *account = key;
  const GArray *handles = value;

  connection_path = mc_account_get_connection_path (account);

  /* offline */
  if (connection_path == NULL)
    return;

  bus_name = strreplace (connection_path + 1, "/", ".");

  proxy = g_object_new (TP_TYPE_PROXY,
      "dbus-daemon", priv->dbus,
      "dbus-connection", ((TpProxy *) priv->dbus)->dbus_connection,
      "bus-name", bus_name,
      "object-path", mc_account_get_connection_path (account),
      NULL);

  tp_proxy_add_interface_by_id (proxy,
      MONORAIL_IFACE_QUARK_CONNECTION_INTERFACE_CONTACT_CAPABILITIES);

  priv->calls_in_progress++;
  monorail_cli_connection_interface_contact_capabilities_call_get_contact_capabilities (
      proxy, -1, handles, get_contact_caps_cb,
      g_strdup (tp_proxy_get_object_path (account)), NULL,
      G_OBJECT (user_data));

  g_free (bus_name);
  g_object_unref (proxy);
}

static void
roster_ready_cb (OssoABookWaitable *waitable,
    const GError *error,
    gpointer user_data)
{
  MonorailContactChooser *self = MONORAIL_CONTACT_CHOOSER (user_data);
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);
  OssoABookContactModel *model;
  GtkTreeIter iter;
  GHashTable *call_table;

  if (error != NULL)
    {
      DEBUG ("error: %s", error->message);
      return;
    }

  model = OSSO_ABOOK_CONTACT_MODEL (
      osso_abook_tree_view_get_base_model (OSSO_ABOOK_TREE_VIEW (self)));

  if (!gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
    return;

  call_table = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL,
      free_garray);

  while (TRUE)
    {
      OssoABookContact *contact;
      GList *roster_contacts, *l;

      gtk_tree_model_get (GTK_TREE_MODEL (model), &iter,
          OSSO_ABOOK_LIST_STORE_COLUMN_CONTACT, &contact,
          -1);

      roster_contacts = osso_abook_contact_get_roster_contacts (contact);

      for (l = roster_contacts; l != NULL; l = l->next)
        {
          OssoABookContact *c = l->data;
          McAccount *account;
          TpHandle handle;
          GArray *contacts;

          account = osso_abook_contact_get_account (c);
          handle = osso_abook_presence_get_handle (OSSO_ABOOK_PRESENCE (c));

          contacts = g_hash_table_lookup (call_table, account);

          if (contacts == NULL)
            {
              contacts = g_array_new (FALSE, FALSE, sizeof (TpHandle));
              g_array_append_val (contacts, handle);
              g_hash_table_insert (call_table, account, contacts);
            }
          else
            {
              g_array_append_val (contacts, handle);
            }
        }

      g_list_free (roster_contacts);
      g_object_unref (contact);

      if (!gtk_tree_model_iter_next (GTK_TREE_MODEL (model), &iter))
        break;
    }

  g_hash_table_foreach (call_table, call_table_foreach, self);
  g_hash_table_destroy (call_table);

  priv->made_all_calls = TRUE;
  check_all_calls_made (self);
}

static gboolean
visible_func (GtkTreeModel *model,
    GtkTreeIter *iter,
    gpointer user_data)
{
  MonorailContactChooserPrivate *priv;
  gboolean out = FALSE;
  GList *roster_contacts, *l;
  OssoABookContact *contact;

  /* not great */
  if (!MONORAIL_IS_CONTACT_CHOOSER (user_data))
    return FALSE;

  priv = GET_PRIVATE (user_data);

  gtk_tree_model_get (model, iter,
      OSSO_ABOOK_LIST_STORE_COLUMN_CONTACT, &contact,
      -1);

  if (contact == NULL)
    return FALSE;

  roster_contacts = osso_abook_contact_get_roster_contacts (contact);

  for (l = roster_contacts; l != NULL; l = l->next)
    {
      McAccount *account;
      OssoABookContact *roster_contact = l->data;
      GHashTable *table;
      TpHandle handle;
      gpointer lookup;

      handle = osso_abook_presence_get_handle (
          OSSO_ABOOK_PRESENCE (roster_contact));

      account = osso_abook_contact_get_account (roster_contact);

      table = g_hash_table_lookup (priv->handle_capable,
          tp_proxy_get_object_path (account));

      if (table == NULL)
        continue;

      lookup = g_hash_table_lookup (table, GUINT_TO_POINTER (handle));

      if (lookup != NULL && GPOINTER_TO_INT (lookup))
        {
          out = TRUE;
          break;
        }
    }

  g_list_free (roster_contacts);

  g_object_unref (contact);

  return out;
}

static void
monorail_contact_chooser_constructed (GObject *self)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);
  OssoABookContactModel *model;
  OssoABookRoster *roster;

  if (G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->constructed)
    G_OBJECT_CLASS (monorail_contact_chooser_parent_class)->constructed (self);

  priv->calls_in_progress = 0;
  priv->made_all_calls = FALSE;

  priv->dbus = tp_dbus_daemon_dup (NULL);

  priv->handle_capable = g_hash_table_new_full (g_str_hash, g_str_equal,
      (GDestroyNotify) g_free, (GDestroyNotify) g_hash_table_destroy);

  model = OSSO_ABOOK_CONTACT_MODEL (
      osso_abook_tree_view_get_base_model (OSSO_ABOOK_TREE_VIEW (self)));
  roster = osso_abook_list_store_get_roster (OSSO_ABOOK_LIST_STORE (model));
  osso_abook_waitable_call_when_ready (OSSO_ABOOK_WAITABLE (roster),
      roster_ready_cb, self, NULL);
}

static void
monorail_contact_chooser_class_init (MonorailContactChooserClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = monorail_contact_chooser_dispose;
  object_class->finalize = monorail_contact_chooser_finalize;
  object_class->constructed = monorail_contact_chooser_constructed;

  signals[DONE] = g_signal_new ("done",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      g_cclosure_marshal_VOID__VOID,
      G_TYPE_NONE, 0);

  g_type_class_add_private (klass, sizeof (MonorailContactChooserPrivate));
}

static void
monorail_contact_chooser_init (MonorailContactChooser *self)
{
}

GtkWidget *
monorail_contact_chooser_new (void)
{
  GtkWidget *out;
  OssoABookContactModel *model;
  OssoABookFilterModel *filter;

  model = osso_abook_contact_model_get_default ();
  filter = osso_abook_filter_model_new (OSSO_ABOOK_LIST_STORE (model));

  out = g_object_new (MONORAIL_TYPE_CONTACT_CHOOSER,
      "model", filter,
      "base-model", model,
      "filter-model", filter,
      "show-contact-avatar", TRUE,
      "hildon-ui-mode", HILDON_UI_MODE_NORMAL,
      NULL);

  osso_abook_filter_model_set_visible_func (filter, visible_func, out, NULL);
  gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));

  g_object_unref (filter);

  return out;
}

gboolean
monorail_contact_chooser_can_send_to_roster_contact (MonorailContactChooser *self,
    OssoABookContact *roster_contact)
{
  MonorailContactChooserPrivate *priv = GET_PRIVATE (self);
  McAccount *account;
  GHashTable *table;
  TpHandle handle;
  gpointer lookup;

  if (!osso_abook_contact_is_roster_contact (roster_contact))
    return FALSE;

  handle = osso_abook_presence_get_handle (
      OSSO_ABOOK_PRESENCE (roster_contact));

  account = osso_abook_contact_get_account (roster_contact);

  table = g_hash_table_lookup (priv->handle_capable,
      tp_proxy_get_object_path (account));

  if (table == NULL)
    return FALSE;

  lookup = g_hash_table_lookup (table, GUINT_TO_POINTER (handle));

  if (lookup != NULL && GPOINTER_TO_INT (lookup))
    return TRUE;

  return FALSE;
}
