#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QtGui>
#include <QByteArray>

#ifdef __arm__
/* The device by default supports only
 * vl4l2src for camera and xvimagesink
 * for screen */
#define VIDEO_SRC "v4l2camsrc"
#else
/* These are for the X86 SDK. Xephyr doesn't
 * support XVideo extension, so the application
 * must use ximagesink. The video source depends
 * on driver of your Video4Linux device so this
 * may have to be changed */
#define VIDEO_SRC "videotestsrc"
#endif

#define VIDEO_WIDTH 352
#define VIDEO_HEIGHT 254


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{

    ui->setupUi(this);
    _btDlg = new BluetoothDialog(this);
    _imgProcessor = new ImageProcessor(VIDEO_WIDTH, VIDEO_HEIGHT, this);
    _cam = new Camera(VIDEO_SRC, VIDEO_WIDTH, VIDEO_HEIGHT, this);
    _colorDlg = new ColorConfiguration(this);

    connect(_imgProcessor, SIGNAL(TurnExceed(bool)), this, SLOT(Turn(bool)));
    connect(_imgProcessor, SIGNAL(Distance(quint16)), this, SLOT(Distance(quint16)));
    connect(_imgProcessor, SIGNAL(Rotation(qint8)), this, SLOT(Rotation(qint8)));
    connect(_imgProcessor, SIGNAL(ObjectCount(int)), this, SLOT(Objects(int)));

    connect(_colorDlg, SIGNAL(ColorChoosed(QColor)), _imgProcessor, SLOT(SetSignColor(QColor)));
    connect(_colorDlg, SIGNAL(ColorToleranceChanged(int)), _imgProcessor, SLOT(SetColorTolerance(int)));
    connect (_btDlg, SIGNAL(NewConnection(BluetoothAdapter*)), this, SLOT(Connected(BluetoothAdapter*)));

    connect(_cam, SIGNAL(PhotoTaked(QImage*)), _imgProcessor, SLOT(PictureReady(QImage*)));
    connect(_imgProcessor, SIGNAL(ImageProcessingCompletted(QImage*)), this, SLOT(ImageReady(QImage*)));

    connect(&_takeTimer, SIGNAL(timeout()), this, SLOT(TakeTime()));
    connect(ui->actionEnable_Sobel_filtering, SIGNAL(toggled(bool)), _imgProcessor, SLOT(SetFiltering(bool)));
    connect(ui->_turnLimit, SIGNAL(valueChanged(int)), _imgProcessor, SLOT(SetTurnLimit(int)));

    connect(_colorDlg, SIGNAL(RefreshPicture()), _cam, SLOT(TakePhoto()));

    ui->actionEnable_Sobel_filtering->setChecked(_settings.value("filtering", 0).toBool());
    ui->_turnLimit->setValue(_settings.value("turnlimit", 2).toInt());
    ui->_refreshRate->setValue(_settings.value("refreshtime", 250).toInt());

    _connected = false;
    _prevTake = QTime::currentTime();
    _ellapsedMsecsFromRefresh = 0;
    TakeTime();
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::Objects(int i)
{
    ui->_objects->setText(QString("Objects: %1").arg(i));
}

void MainWindow::Turn(bool i)
{
    ui->_isTurn->setText(QString("Turn? %1").arg(i));
    QByteArray tmp = QByteArray();
    tmp.push_back((quint8)0x01);
    tmp.push_back((uchar)i);
    if (_connected) _btDevice->SendOnConnection(0, &tmp);
}

void MainWindow::Distance(quint16 i)
{
    ui->_distance->setText(QString("Distance: %1").arg(i));
    QByteArray tmp = QByteArray();
    tmp.push_back((quint8)0x02);
    tmp.push_back(*((quint8*)&i));
    tmp.push_back(*(((quint8*)&i) + 1));
    if (_connected) _btDevice->SendOnConnection(0, &tmp);
}

void MainWindow::Rotation(qint8 i)
{
    ui->_rotation->setText(QString("Rotation: %1").arg(i));
    QByteArray tmp = QByteArray();
    tmp.push_back((quint8)0x03);
    tmp.push_back(i);
    if (_connected) _btDevice->SendOnConnection(0, &tmp);
}

void MainWindow::Connected(BluetoothAdapter *device)
{
    _connected = true;
    ui->actionConnect_to_device->setEnabled(false);
    _btDevice = device;
}

void MainWindow::ImageReady(QImage* img)
{
    _settings.setValue("refreshtime", ui->_refreshRate->value());
    ui->_imgLabel->setPixmap(QPixmap::fromImage(*img));

    int ellapsedTime = _prevTake.msecsTo(QTime::currentTime());
    if (ui->_refreshRate->value() - ellapsedTime <= 0) TakeTime();
    else _takeTimer.start(ui->_refreshRate->value() - ellapsedTime);
}

void MainWindow::TakeTime()
{
    _takeTimer.stop();
    _prevTake = QTime::currentTime();
    _cam->TakePhoto();

}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type())
    {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_actionConfigure_Sign_Color_triggered()
{
    disconnect(_cam, SIGNAL(PhotoTaked(QImage*)), _imgProcessor, SLOT(PictureReady(QImage*)));
    connect(_cam, SIGNAL(PhotoTaked(QImage*)), _colorDlg, SLOT(ImageRecived(QImage*)));
    _colorDlg->exec();
    disconnect(_cam, SIGNAL(PhotoTaked(QImage*)), _colorDlg, SLOT(ImageRecived(QImage*)));
    connect(_cam, SIGNAL(PhotoTaked(QImage*)), _imgProcessor, SLOT(PictureReady(QImage*)));
    TakeTime();
}

void MainWindow::on_actionConnect_to_device_triggered()
{
    _btDlg->exec();
}
