#include "imageprocessor.h"
#include <QtGui>
#include <QtAlgorithms>

ImageProcessor::ImageProcessor(int width, int height, QObject *parent) :
    QObject(parent)
{
    _distanceTolerance = 5;
    _width = width;
    _height = height;

    _filtering = _settings.value("filtering", 0).toBool();
    _sign = QColor::fromRgb(_settings.value("sign").toInt());
    _colorTolerance = _settings.value("colortolerance", 2500).toInt();
    _turnLimit = _settings.value("turnlimit", 2).toInt();

    _filteredImg = QImage(_width, _height, QImage::Format_RGB888);
}

void ImageProcessor::PictureReady(QImage* image)
{
    ColorAndSobelFilter(image);
    AnalysePixels();
    emit ImageProcessingCompletted(&_filteredImg);
}

bool ImageProcessor::IsNearColor(uchar r, uchar g, uchar b)
{
    int redDst = (r - _sign.red()) * (r - _sign.red());
    int greenDst = (g- _sign.green()) * (g - _sign.green());
    int blueDst = (b - _sign.blue()) * (b - _sign.blue());

    if ((redDst + greenDst + blueDst) >= _colorTolerance)
        return false;
    else
        return true;

}

void ImageProcessor::ColorAndSobelFilter(const QImage *source)
{
    _borderSigns.clear();
    int GX[3][3] = {{-1,  0,  1},
                    {-2,  0,  2},
                    {-1,  0,  1}};

    int GY[3][3] = {{ 1,  2,  1},
                    { 0,  0,  0},
                    {-1, -2, -1}};

    ushort byteDepth = source->depth() / 8;
    int width = source->width() * byteDepth;
    int height = source->height();
    const uchar *sourceBits = source->bits();
    uchar *destBits = _filteredImg.bits();

    int x, y;
    bool uniqCoord;
    int matrixX, matrixY;
    int lineOffset;
    int xCoord;

    int sum;
    long sumX, sumY;

    int currColor;

    for (y = 0; y < height; ++y)
    {
        lineOffset = (y * width);
        xCoord = 0;

        for (x = 0; x < width; x+= byteDepth)
        {
            if (_filtering)
            {
                if ( y == 0 || y >= height-1 || x == 0 || x >= width-1 )
                {
                    sum = 0;
                }
                else
                {
                    sumX = 0;
                    sumY = 0;

                    for (matrixX=-1; matrixX<=1; matrixX++)
                    {
                        for(matrixY=-1; matrixY<=1; matrixY++)
                        {
                            currColor = (*(sourceBits + lineOffset + matrixY * width + x + matrixX)                 //R
                                         + *(sourceBits + lineOffset + matrixY * width + x + matrixX + 1)           //G
                                         + *(sourceBits + lineOffset + matrixY * width + x + matrixX + 2)) / 3;     //B
                            sumX += currColor * GX[matrixX+1][matrixY + 1];
                            sumY += currColor * GY[matrixX+1][matrixY + 1];
                        }
                    }
                    sum = abs(sumX) + abs(sumY);
                    if (sum > 255)
                        sum = 255;

                }
            }
            else
                sum = 0;


            if (!IsNearColor(*(sourceBits + lineOffset + x), *(sourceBits + lineOffset + x + 1), *(sourceBits + lineOffset + x + 2)))
            {
                for (ushort currByte = 0; currByte < byteDepth; currByte++)
                    *(destBits + lineOffset + x + currByte) = sum;
            }
            else
            {
                uniqCoord = true;

                for (int index = 0; index < _borderSigns.count(); index++)
                {
                    if (_borderSigns[index].LeftTop.x - _distanceTolerance > xCoord || _borderSigns[index].RightBottom.x + _distanceTolerance < xCoord)
                        continue;

                    uniqCoord = false;

                    if (_borderSigns[index].LeftTop.x < xCoord && _borderSigns[index].RightBottom.x > xCoord)
                    {
                        break;
                    }

                    if (_borderSigns[index].LeftTop.x > xCoord)
                    {
                        _borderSigns[index].LeftTop.x = xCoord;
                        _borderSigns[index].LeftTop.y = y;
                        break;
                    }

                    if (_borderSigns[index].RightBottom.x < xCoord)
                    {
                        _borderSigns[index].RightBottom.x = xCoord;
                        _borderSigns[index].RightBottom.y = y;
                        break;
                    }

                    if (_borderSigns[index].RightBottom.x == xCoord)
                    {
                        _borderSigns[index].RightBottom.y = y;
                        break;
                    }
                    break;

                }

                if (uniqCoord)
                {
                    int index = 0;
                    BorderSign newBorderItem;
                    newBorderItem.LeftTop.x = xCoord;
                    newBorderItem.RightBottom.x = xCoord;
                    newBorderItem.LeftTop.y = y;
                    newBorderItem.RightBottom.y = y;
                    if (_borderSigns.count() == 0)
                    {
                        _borderSigns.insert(index, newBorderItem);
                    }
                    else
                    {
                        for (index = 0; index < _borderSigns.count(); index++)
                        {
                            if (_borderSigns[index].LeftTop.x > xCoord)
                                break;
                        }
                        _borderSigns.insert(index, newBorderItem);
                    }
                }


                *(destBits + lineOffset + x) = _sign.red();
                *(destBits + lineOffset + x + 1) = _sign.green();
                *(destBits + lineOffset + x + 2) = _sign.blue();
            }
            xCoord++;
        }
    }


    for (int index = 0; index < _borderSigns.count(); index++)
    {
        if (index != 0 && (_borderSigns[index].LeftTop.x < _borderSigns[index - 1].RightBottom.x || _borderSigns[index].RightBottom.x < _borderSigns[index - 1].RightBottom.x))
        {
            _borderSigns.removeAt(index);
            index--;
            continue;
        }
        if (index != _borderSigns.count() - 1 && abs(_borderSigns[index].RightBottom.x - _borderSigns[index + 1].LeftTop.x) <= _distanceTolerance)
        {
            _borderSigns[index].RightBottom =
                    (_borderSigns[index].RightBottom.x > _borderSigns[index + 1].RightBottom.x ?
                        _borderSigns[index].RightBottom :
                        _borderSigns[index + 1].RightBottom);
            _borderSigns.removeAt(index + 1);
            index--;
            continue;
        }
        if (_borderSigns[index].RightBottom.x - _borderSigns[index].LeftTop.x <= 3)
        {
            _borderSigns.removeAt(index);
            index--;
            continue;
        }
    }
}

void ImageProcessor::AnalysePixels()
{
    emit ObjectCount(_borderSigns.count());

    if (_borderSigns.count() == 0)
    {
        emit TurnExceed(false);
        return;
    }
    emit TurnExceed((_borderSigns.count() < _turnLimit ? true : false));

    if (_borderSigns.count() <= 2) return;

    int index = 0;
    foreach (BorderSign currSign, _borderSigns)
    {
        if (currSign.LeftTop.x >= _width / 2) break;
        index++;
    }

    if (index >= _borderSigns.count() - 1) index = _borderSigns.count() - 2;
    double dx = _borderSigns[index + 1].LeftTop.x - _borderSigns[index].LeftTop.x;
    double dy = _borderSigns[index + 1].LeftTop.y - _borderSigns[index].LeftTop.y;
    int dst = (int)atan(dy / dx);
    emit Distance(dst);

    qint8 rot;
    rot = (_borderSigns[index + 1].LeftTop.y - _borderSigns[index].LeftTop.y);
    emit Rotation(rot);
}

void ImageProcessor::SetSignColor(QColor color)
{
    _sign = color;
    _settings.setValue("sign", _sign.rgb());
}

void ImageProcessor::SetFiltering(bool filtering)
{
    _filtering = filtering;
    _settings.setValue("filtering", _filtering);
}

void ImageProcessor::SetColorTolerance(int colorTolerance)
{
    _colorTolerance = colorTolerance * colorTolerance;
    _settings.setValue("colortolerance", _colorTolerance);
}

void ImageProcessor::SetTurnLimit(int turnLimit)
{
    _turnLimit = turnLimit;
    _settings.setValue("turnlimit", _turnLimit);
}
