#include "camera.h"
#include <QtGui>

Camera::Camera(char *source, int width, int height, QObject *parent) :
    QObject(parent)
{
    _width = width;
    _height = height;

    InitGStreamer(source);

    _takedImage = QImage(_width, _height, QImage::Format_RGB888);
}

void Camera::TakePhoto()
{    
    GstElement *image_sink;
    
    /* Get the image sink element from the pipeline */
    image_sink = gst_bin_get_by_name(GST_BIN(_pipeline),
                            "image_sink");
    
    /* Connect the "handoff"-signal of the image sink to the
     * callback. This gets called whenever the sink gets a
     * buffer it's ready to pass forward on the pipeline */
    _buffCallbackNum= g_signal_connect(
                G_OBJECT(image_sink), "handoff",
                G_CALLBACK(HandOffImage), this);
}

/* This callback will be registered to the image sink
 * after user requests a photo */
gboolean Camera::HandOffImage(
                GstElement *image_sink,
                GstBuffer *buffer,
                GstPad *pad,
                Camera *camera)
{
    unsigned char *data_photo = (unsigned char *)GST_BUFFER_DATA(buffer);

    /* Create a IMG of the data and check the status */
    camera->CreateImg(data_photo);

    /* Disconnect the handler so no more photos
     * are taken */
    g_signal_handler_disconnect(G_OBJECT(image_sink), camera->GetBuffCallbackNum());

    return TRUE;
}

void Camera::CreateImg(unsigned char *data)
{
    memcpy(_takedImage.bits(), data, _width * _height * (_takedImage.depth() / 8));

    emit PhotoTaked(&_takedImage);
}


gboolean Camera::InitGStreamer(char *source)
{
    GstElement *camera_src;
    GstElement *image_sink;

    GstElement *image_queue;

    GstElement *csp_filter;
    GstElement *image_filter;

    GstCaps *caps;
    GstBus *bus;


    /* Initialize Gstreamer */
    gst_init(NULL, NULL);

    /* Create pipeline and attach a callback to it's
     * message bus */
    _pipeline = gst_pipeline_new("nokia-camera");

    bus = gst_pipeline_get_bus(GST_PIPELINE(_pipeline));
    gst_bus_add_watch(bus, (GstBusFunc)&BusCallback, NULL);
    gst_object_unref(GST_OBJECT(bus));

    /* Create elements */

    /* Camera video stream comes from a Video4Linux driver */
    camera_src = gst_element_factory_make(source, "camera_src");

    /* Colorspace filter is needed to make sure that sinks understands
     * the stream coming from the camera */
    csp_filter = gst_element_factory_make("ffmpegcolorspace", "csp_filter");

    /* Creates separate thread for the stream from which the image
     * is captured */
    image_queue = gst_element_factory_make("queue", "image_queue");

    /* Filter to convert stream to use format that the gdkpixbuf library
     * can use */
    image_filter = gst_element_factory_make("ffmpegcolorspace", "image_filter");

    /* A dummy sink for the image stream. Goes to bitheaven */
    image_sink = gst_element_factory_make("fakesink", "image_sink");

    /* Check that elements are correctly initialized */
    if (!(_pipeline && camera_src && csp_filter
             && image_queue && image_filter && image_sink))
    {
        qDebug() << "Couldn't create pipeline elements";
        return FALSE;
    }

    /* Set image sink to emit handoff-signal before throwing away
     * it's buffer */
    g_object_set(G_OBJECT(image_sink), "signal-handoffs", TRUE, NULL);

    /* Add elements to the pipeline. This has to be done prior to
     * linking them */
    gst_bin_add_many(GST_BIN(_pipeline), camera_src, csp_filter,
                     image_queue,
                     image_filter, image_sink, NULL);

    /* Specify what kind of video is wanted from the camera */
    caps = gst_caps_new_simple("video/x-raw-yuv",
                     "format",  GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
                     "width", G_TYPE_INT, _width,
                     "height", G_TYPE_INT, _height,
                     "bpp", G_TYPE_INT, 24,
                     "framerate", GST_TYPE_FRACTION, 25, 1,
                     NULL);

    /* Link the camera source and colorspace filter using capabilities
     * specified */
    if(!gst_element_link_filtered(camera_src, csp_filter, caps))
    {
        return FALSE;
    }
    gst_caps_unref(caps);

    /* gdkpixbuf requires 8 bits per sample which is 24 bits per
     * pixel */
    caps = gst_caps_new_simple("video/x-raw-rgb",
                        "width", G_TYPE_INT, _width,
                        "height", G_TYPE_INT, _height,
                        "bpp", G_TYPE_INT, 24,
                        NULL);

    /* Link the image-branch of the pipeline. The pipeline is
     * ready after this */
    if(!gst_element_link_many(csp_filter, image_queue, image_filter, NULL)) return FALSE;
    if(!gst_element_link_filtered(image_filter, image_sink, caps)) return FALSE;

    gst_caps_unref(caps);

    gst_element_set_state(_pipeline, GST_STATE_PLAYING);

    return TRUE;
}



void Camera::BusCallback(GstBus *bus, GstMessage *message, gpointer data)
{
    gchar *message_str;
    GError *error;

    /* Report errors to the console */
    if(GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR)
    {
        gst_message_parse_error(message, &error, &message_str);
        qDebug() << "GST error: " << message_str;
        g_free(error);
        g_free(message_str);
    }

    /* Report warnings to the console */
    if(GST_MESSAGE_TYPE(message) == GST_MESSAGE_WARNING)
    {
        gst_message_parse_warning(message, &error, &message_str);
        qDebug() << "GST warning: " << message_str;
        g_free(error);
        g_free(message_str);
    }
}

guint Camera::GetBuffCallbackNum()
{
    return _buffCallbackNum;
}
